<?php

namespace App\Http\Controllers;

use App\Models\CompanySettings;
use App\Models\SystemSettings;
use Illuminate\Http\Request;

class AdministrationController extends Controller
{
    public function adminIndex()
    {
        return view('modules.administration');
    }

    public function generalSettings()
    {
        return view('modules.administration.general-settings');
    }

    public function userSettings()
   
   {
        return view('modules.administration.user-settings');
    }

    public function userManagement()
    {
        $users = \App\Models\User::with('role', 'group', 'phones')->get();
        return view('modules.administration.user-management', compact('users'));
    }

    public function createUser()
    {
        $roles = \App\Models\Role::all();
        $groups = \App\Models\Group::all();
        return view('modules.administration.create-user', compact('roles', 'groups'));
    }

    public function storeUser(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone' => 'nullable|string|max:20',
            'role' => 'nullable|exists:user_roles,user_role_id',
            'group' => 'nullable|exists:groups,id',
        ]);

        // Assuming User model exists
        \App\Models\User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
            'role_id' => $validated['role'],
            'group_id' => $validated['group'],
            'password_hash' => bcrypt('password'), // Default password, should be changed
        ]);

        return redirect()->back()->with('success', 'User added successfully.');
    }

    public function editUser($id)
    {
        $user = \App\Models\User::with('phones')->findOrFail($id);
        $roles = \App\Models\UserRole::all();
        $groups = \App\Models\Group::all(); // Use Group model, not UserGroup
        return view('modules.administration.edit-user', compact('user', 'roles', 'groups'));
    }

    public function updateUser(Request $request, $id)
    {
        $user = \App\Models\User::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'nullable|string|max:20',
            'role' => 'nullable|integer|exists:user_roles,user_role_id',
            'group' => 'nullable|integer|exists:user_groups,user_group_id', // Correct table for group validation
        ]);

        $user->name = $validated['name'];
        $user->email = $validated['email'];
        $roleId = $validated['role'] ?? null;
        if ($roleId === '') {
            $roleId = null;
        }
        $groupId = $validated['group'] ?? null;
        if ($groupId === '') {
            $groupId = null;
        }

        $user->role_id = $roleId;
        $user->group_id = $groupId; // Assign group_id directly
        $user->save();

        // Update or create first phone record
        if (isset($validated['phone'])) {
            $phoneData = ['phone_number' => $validated['phone']];
            if ($user->phones()->exists()) {
                $user->phones()->first()->update($phoneData);
            } else {
                $user->phones()->create($phoneData);
            }
        }

        // Remove code that manipulates UserGroup

        return redirect()->route('administration.user-management')->with('success', 'User updated successfully.');
    }

    public function destroyUser($id)
    {
        $user = \App\Models\User::findOrFail($id);
        $user->delete();

        return redirect()->back()->with('success', 'User deleted successfully.');
    }

    public function userPermissions($id)
    {
        $user = \App\Models\User::findOrFail($id);
        $isAdmin = $user->role === 'admin';

        $modules = [
            'Edit Modules' => [
                'Account Types' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Age Analysis' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Cases' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Clients' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Lawyers' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Case Categories' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Case Status' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Case Types' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Court Areas' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Court Types' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Defendants' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Departments' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Judges' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Judgements' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Plaintiffs' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Suppliers' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Supplier Accounts' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Supplier Areas' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Supplier Categories' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Tariffs' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Taxes' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'General Ledger' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Multi Currency' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Sub Accounts' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
            ],
            'Processing Modules' => [
                'Assignments' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Diary' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Cashbook' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Client Invoices' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Client Quotations' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Client Receipts' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Court Filings' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Journals' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Supplier Invoices' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
                'Task Management' => ['view' => true, 'add' => true, 'edit' => true, 'delete' => true],
            ],
            'Reports Modules' => [
                'Cashbook Reports' => ['view' => true, 'add' => false, 'edit' => false, 'delete' => false],
                'Client Statements' => ['view' => true, 'add' => false, 'edit' => false, 'delete' => false],
                'Fee Collection' => ['view' => true, 'add' => false, 'edit' => false, 'delete' => false],
                'General Ledger Reports' => ['view' => true, 'add' => false, 'edit' => false, 'delete' => false],
                'GL Account Details' => ['view' => true, 'add' => false, 'edit' => false, 'delete' => false],
                'Income Statements' => ['view' => true, 'add' => false, 'edit' => false, 'delete' => false],
                'Lawyer Statements' => ['view' => true, 'add' => false, 'edit' => false, 'delete' => false],
                'Outstanding Fees' => ['view' => true, 'add' => false, 'edit' => false, 'delete' => false],
                'Outstanding Sales' => ['view' => true, 'add' => false, 'edit' => false, 'delete' => false],
                'Chart of Accounts' => ['view' => true, 'add' => false, 'edit' => false, 'delete' => false],
                'Supplier Statements' => ['view' => true, 'add' => false, 'edit' => false, 'delete' => false],
                'Total Fees' => ['view' => true, 'add' => false, 'edit' => false, 'delete' => false],
                'Trial Balance' => ['view' => true, 'add' => false, 'edit' => false, 'delete' => false],
            ],
        ];

        return view('modules.administration.user-permissions', compact('user', 'isAdmin', 'modules'));
    }

    public function updateUserPermissions(Request $request, $id)
    {
        $user = \App\Models\User::findOrFail($id);
        $permissions = $request->input('permissions', []);
        $moduleViews = $request->input('module_views', []);

        // Enforce module-first logic: if module View not checked, clear all feature permissions for that module
        foreach ($permissions as $moduleName => &$modulePerms) {
            if (!isset($moduleViews[$moduleName]) || !$moduleViews[$moduleName]) {
                $modulePerms = []; // Clear all feature permissions
            }
        }

        // Ensure no delete permission for Invoices and Receipts
        $noDeleteFeatures = ['Client Invoices', 'Client Receipts', 'Supplier Invoices'];
        foreach ($permissions as $moduleName => &$modulePerms) {
            foreach ($modulePerms as $feature => &$perms) {
                if (in_array($feature, $noDeleteFeatures)) {
                    unset($perms['delete']);
                }
            }
        }

        $user->permissions = json_encode($permissions); // Ensure it's JSON
        $user->save();

        return redirect()->back()->with('success', 'Permissions updated successfully');
    }

    public function rolesIndex()
    {
        $userRoles = \App\Models\UserRole::with(['user', 'role'])->get();
        return view('modules.administration.roles', compact('userRoles'));
    }

    public function rolesCreate()
    {
        return view('modules.administration.roles-create');
    }

    public function rolesStore(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        \App\Models\UserRole::create([
            'role_name' => $validated['name'],
            'role_description' => $validated['description'],
        ]);

        return redirect()->route('administration.roles.index')->with('success', 'User role created successfully.');
    }

    public function rolesShow($id)
    {
        $userRole = \App\Models\UserRole::with('user', 'role')->findOrFail($id);
        return view('modules.administration.roles-show', compact('userRole'));
    }

    public function rolesEdit($id)
    {
        $userRole = \App\Models\UserRole::with('user', 'role')->findOrFail($id);
        return view('modules.administration.edit-role', compact('userRole'));
    }

    public function rolesUpdate(Request $request, $id)
    {
        $userRole = \App\Models\UserRole::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        $userRole->update([
            'role_name' => $validated['name'],
            'role_description' => $validated['description'],
        ]);

        return redirect()->route('administration.roles.index')->with('success', 'User role updated successfully.');
    }

    public function rolesDestroy($id)
    {
        $userRole = \App\Models\UserRole::findOrFail($id);
        $userRole->delete();

        return redirect()->route('administration.roles.index')->with('success', 'User role deleted successfully.');
    }

    public function groupsIndex()
    {
        $groups = \App\Models\Group::all();
        return view('modules.administration.groups', compact('groups'));
    }

    public function groupsCreate()
    {
        return view('modules.administration.groups-create');
    }

    public function groupsStore(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        $groupData = [
            'group_name' => $validated['name'],
            'group_description' => $validated['description'] ?? null,
        ];

        \App\Models\Group::create($groupData);

        return redirect()->route('administration.groups.index')->with('success', 'Group created successfully.');
    }

    public function groupsShow($id)
    {
        $group = \App\Models\Group::findOrFail($id);
        return view('modules.administration.groups-show', compact('group'));
    }

    public function groupsEdit($id)
    {
        $group = \App\Models\Group::findOrFail($id);
        return view('modules.administration.groups-edit', compact('group'));
    }

    public function groupsUpdate(Request $request, $id)
    {
        $group = \App\Models\Group::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        $groupData = [
            'group_name' => $validated['name'],
            'group_description' => $validated['description'] ?? null,
        ];

        $group->update($groupData);

        return redirect()->route('administration.groups.index')->with('success', 'Group updated successfully.');
    }

    public function groupsDestroy($id)
    {
        $group = \App\Models\Group::findOrFail($id);
        $group->delete();

        return redirect()->route('administration.groups.index')->with('success', 'Group deleted successfully.');
    }

    public function companyInfo()
    {
        // Load the first company settings record
        $companySettings = CompanySettings::with('currency')->first();

        return view('company-info', compact('companySettings'));
    }

    public function getCompanyInfoApi()
    {
        $companySettings = CompanySettings::with('currency')->first();

        if (!$companySettings) {
            return response()->json(['message' => 'Company settings not found'], 404);
        }

        return response()->json($companySettings);
    }

    public function updateCompanyInfoApi(Request $request)
    {
        $companySettings = CompanySettings::first();

        if (!$companySettings) {
            $companySettings = new CompanySettings();
        }

        $validated = $request->validate([
            'company_name' => 'required|string|max:255',
            'company_code' => 'required|string|max:255|unique:company_settings,company_code,' . ($companySettings->id ?? 'NULL'),
            'phone' => 'nullable|string|max:50',
            'email' => 'nullable|email|max:255',
            'address' => 'nullable|string',
            'website' => 'nullable|url|max:255',
            'tax_id' => 'nullable|string|max:100',
            'currency_id' => 'nullable|exists:currencies,id',
            'bank_details' => 'nullable|array',
            'logo_url' => 'nullable|string|max:255',
        ]);

        $companySettings->fill($validated);
        $companySettings->save();

        return response()->json(['message' => 'Company settings updated successfully', 'companySettings' => $companySettings]);
    }

    public function systemSettings()
    {
        // Load the first system settings record
        $systemSettings = SystemSettings::with('defaultCurrency')->first();

        return view('modules.administration.system-settings', compact('systemSettings'));
    }

    public function getSystemSettingsApi()
    {
        $systemSettings = SystemSettings::with('defaultCurrency')->first();

        if (!$systemSettings) {
            return response()->json(['message' => 'System settings not found'], 404);
        }

        return response()->json($systemSettings);
    }

    public function updateSystemSettingsApi(Request $request)
    {
        $systemSettings = SystemSettings::first();

        if (!$systemSettings) {
            $systemSettings = new SystemSettings();
        }

        $validated = $request->validate([
            'default_currency_id' => 'nullable|exists:currencies,id',
            'default_language' => 'required|string|max:10',
            'timezone' => 'required|string|max:50',
            'date_format' => 'required|string|max:20',
            'number_format' => 'required|string|max:20',
            'email_notifications' => 'boolean',
            'auto_backup_frequency' => 'required|string|max:20',
            'case_number_prefix' => 'required|string|max:10',
            'invoice_number_prefix' => 'required|string|max:10',
            'quotation_number_prefix' => 'required|string|max:10',
            'receipt_number_prefix' => 'required|string|max:10',
            'default_case_status' => 'required|string|max:50',
            'default_payment_terms' => 'required|integer|min:1',
            'tax_calculation_method' => 'required|string|max:20',
            'file_upload_size_limit' => 'required|integer|min:1',
            'session_timeout' => 'required|integer|min:1',
            'maintenance_mode' => 'boolean',
        ]);

        $systemSettings->fill($validated);
        $systemSettings->save();

        return response()->json(['message' => 'System settings updated successfully', 'systemSettings' => $systemSettings]);
    }

    public function approvals()
    {
        return view('modules.administration.approvals');
    }
}
