<?php

namespace App\Http\Controllers;

use App\Models\Assignment;
use Illuminate\Http\Request;

class AssignmentController extends Controller
{
    public function createAssignment()
    {
        $lawyers = \App\Models\Lawyer::all();
        $clients = \App\Models\Client::all();
        $cases = \App\Models\LegalCase::all();
        $caseTypes = \App\Models\CaseType::all();
        $caseCategories = \App\Models\CaseCategory::all();
        $caseStatuses = \App\Models\CaseStatus::all();
        $judges = \App\Models\Judge::all();
        $priorities = ['low', 'medium', 'high'];
        $taskTypes = ['document_review', 'client_meeting', 'court_appearance', 'research', 'filing', 'correspondence', 'administrative'];

        return view('modules.processing.create-assignment', compact(
            'lawyers', 'clients', 'cases', 'caseTypes', 'caseCategories', 'caseStatuses', 'judges', 'priorities', 'taskTypes'
        ));
    }

    public function storeAssignment(Request $request)
    {
        $validated = $request->validate([
            'assignment_title' => 'required|string|max:255',
            'lawyer_id' => 'required|integer|exists:lawyers,id',
            'client_id' => 'nullable|integer|exists:clients,id',
            'case_id' => 'nullable|integer|exists:legal_cases,id',
            'case_type_id' => 'nullable|integer|exists:case_types,id',
            'case_category_id' => 'nullable|integer|exists:case_categories,id',
            'case_status_id' => 'nullable|integer|exists:case_statuses,id',
            'task_type' => 'required|string|max:50',
            'priority' => 'required|in:low,medium,high',
            'due_date' => 'required|date',
            'due_time' => 'nullable|date_format:H:i',
            'estimated_hours' => 'nullable|numeric|min:0.5|max:100',
            'description' => 'required|string',
            'instructions' => 'nullable|string',
            'documents_required' => 'nullable|string',
            'court_date' => 'nullable|date',
            'court_time' => 'nullable|date_format:H:i',
            'court_location' => 'nullable|string|max:255',
            'opposing_counsel' => 'nullable|string|max:255',
            'judge_id' => 'nullable|integer|exists:judges,judge_id',
            'client_contact_info' => 'nullable|string',
            'billing_notes' => 'nullable|string',
            'status' => 'required|in:pending,in_progress,completed,cancelled',
        ]);

        // Combine date and time for due_datetime
        if ($request->has('due_time') && $request->due_time) {
            $validated['due_datetime'] = $request->due_date . ' ' . $request->due_time;
        } else {
            $validated['due_datetime'] = $request->due_date . ' 23:59:59';
        }

        // Combine court date and time if provided
        if ($request->has('court_date') && $request->court_date && $request->has('court_time') && $request->court_time) {
            $validated['court_datetime'] = $request->court_date . ' ' . $request->court_time;
        }

        // Set default values
        $validated['created_by'] = auth()->id() ?? 1;
        $validated['assigned_date'] = now();
        $validated['progress_percentage'] = 0;
        $validated['is_billable'] = true;

        /** @var Assignment $assignment */
        $assignment = Assignment::create($validated);

        // Log assignment creation
        \App\Models\AssignmentHistory::create([
            'assignment_id' => $assignment->id,
            'user_id' => auth()->id(),
            'action' => 'created',
            'description' => 'Assignment created',
        ]);

        return redirect()->route('modules.processing.assignment.list')->with('success', 'Assignment created successfully.');
    }

    public function listAssignments(Request $request)
    {
        $query = \App\Models\Assignment::with(['lawyer', 'client', 'legalCase', 'caseType', 'caseCategory', 'caseStatus']);

        // Apply filters
        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('priority') && $request->priority) {
            $query->where('priority', $request->priority);
        }

        if ($request->has('lawyer_id') && $request->lawyer_id) {
            $query->where('lawyer_id', $request->lawyer_id);
        }

        if ($request->has('task_type') && $request->task_type) {
            $query->where('task_type', $request->task_type);
        }

        if ($request->has('start_date') && $request->start_date) {
            $query->whereDate('due_datetime', '>=', $request->start_date);
        }

        if ($request->has('end_date') && $request->end_date) {
            $query->whereDate('due_datetime', '<=', $request->end_date);
        }

        $assignments = $query->orderBy('due_datetime', 'asc')->paginate(15);
        $lawyers = \App\Models\Lawyer::all();
        $taskTypes = ['document_review', 'client_meeting', 'court_appearance', 'research', 'filing', 'correspondence', 'administrative'];

        return view('modules.processing.assignment-list', compact('assignments', 'lawyers', 'taskTypes'));
    }

    public function showAssignment($id)
    {
        $assignment = \App\Models\Assignment::with(['lawyer', 'client', 'legalCase', 'caseType', 'caseCategory', 'caseStatus'])->findOrFail($id);
        return view('modules.processing.assignment-show', compact('assignment'));
    }

    public function editAssignment($id)
    {
        $assignment = \App\Models\Assignment::findOrFail($id);
        $lawyers = \App\Models\Lawyer::all();
        $clients = \App\Models\Client::all();
        $cases = \App\Models\LegalCase::all();
        $caseTypes = \App\Models\CaseType::all();
        $caseCategories = \App\Models\CaseCategory::all();
        $caseStatuses = \App\Models\CaseStatus::all();
        $priorities = ['low', 'medium', 'high'];
        $taskTypes = ['document_review', 'client_meeting', 'court_appearance', 'research', 'filing', 'correspondence', 'administrative'];

        return view('modules.processing.edit-assignment', compact(
            'assignment', 'lawyers', 'clients', 'cases', 'caseTypes', 'caseCategories', 'caseStatuses', 'priorities', 'taskTypes'
        ));
    }

    public function updateAssignment(Request $request, $id)
    {
        /** @var \App\Models\Assignment $assignment */
        $assignment = \App\Models\Assignment::findOrFail($id);

        $validated = $request->validate([
            'assignment_title' => 'required|string|max:255',
            'lawyer_id' => 'required|integer|exists:lawyers,id',
            'client_id' => 'nullable|integer|exists:clients,id',
            'case_id' => 'nullable|integer|exists:legal_cases,id',
            'case_type_id' => 'nullable|integer|exists:case_types,id',
            'case_category_id' => 'nullable|integer|exists:case_categories,id',
            'case_status_id' => 'nullable|integer|exists:case_statuses,id',
            'task_type' => 'required|string|max:50',
            'priority' => 'required|in:low,medium,high',
            'due_date' => 'required|date',
            'due_time' => 'nullable|date_format:H:i',
            'estimated_hours' => 'nullable|numeric|min:0.5|max:100',
            'description' => 'required|string',
            'instructions' => 'nullable|string',
            'documents_required' => 'nullable|string',
            'court_date' => 'nullable|date',
            'court_time' => 'nullable|date_format:H:i',
            'court_location' => 'nullable|string|max:255',
            'opposing_counsel' => 'nullable|string|max:255',
            'judge_id' => 'nullable|integer|exists:judges,judge_id',
            'client_contact_info' => 'nullable|string',
            'billing_notes' => 'nullable|string',
            'status' => 'required|in:pending,in_progress,completed,cancelled',
        ]);

        // Combine date and time for due_datetime
        if ($request->has('due_time') && $request->due_time) {
            $validated['due_datetime'] = $request->due_date . ' ' . $request->due_time;
        } else {
            $validated['due_datetime'] = $request->due_date . ' 23:59:59';
        }

        // Combine court date and time if provided
        if ($request->has('court_date') && $request->court_date && $request->has('court_time') && $request->court_time) {
            $validated['court_datetime'] = $request->court_date . ' ' . $request->court_time;
        }

        $assignment->update($validated);

        // Log assignment update
        \App\Models\AssignmentHistory::create([
            'assignment_id' => $assignment->id,
            'user_id' => auth()->id(),
            'action' => 'updated',
            'description' => 'Assignment updated',
        ]);

        return redirect()->route('modules.processing.assignment.list')->with('success', 'Assignment updated successfully.');
    }

    public function deleteAssignment($id)
    {
        /** @var \App\Models\Assignment $assignment */
        $assignment = \App\Models\Assignment::findOrFail($id);

        // Log assignment deletion
        \App\Models\AssignmentHistory::create([
            'assignment_id' => $assignment->id,
            'user_id' => auth()->id(),
            'action' => 'deleted',
            'description' => 'Assignment deleted',
        ]);

        $assignment->delete();

        return redirect()->route('modules.processing.assignment.list')->with('success', 'Assignment deleted successfully.');
    }

    public function reassignAssignment(Request $request, $id)
    {
        $validated = $request->validate([
            'new_lawyer_id' => 'required|integer|exists:lawyers,id',
            'reassignment_reason' => 'nullable|string|max:255',
        ]);

        /** @var \App\Models\Assignment $assignment */
        $assignment = \App\Models\Assignment::findOrFail($id);

        // Store old lawyer for logging
        $oldLawyer = $assignment->lawyer;
        $oldLawyerName = $oldLawyer ? $oldLawyer->first_name . ' ' . $oldLawyer->last_name : 'No Lawyer Assigned';

        $newLawyer = \App\Models\Lawyer::find($validated['new_lawyer_id']);

        // Update assignment with new lawyer
        $assignment->update([
            'lawyer_id' => $validated['new_lawyer_id'],
            'assigned_date' => now(), // Reset assigned date
        ]);

        // Log reassignment
        \App\Models\AssignmentHistory::create([
            'assignment_id' => $assignment->id,
            'user_id' => auth()->id(),
            'action' => 'reassigned',
            'description' => 'Assignment reassigned from ' . $oldLawyerName . ' to ' . $newLawyer->first_name . ' ' . $newLawyer->last_name,
        ]);

        return redirect()->back()->with('success', 'Assignment reassigned successfully from ' . $oldLawyerName . ' to ' . $newLawyer->first_name . ' ' . $newLawyer->last_name . '.');
    }

    public function assignmentReports(Request $request)
    {
        $query = \App\Models\Assignment::with(['lawyer', 'client', 'legalCase', 'caseType', 'caseCategory', 'caseStatus', 'creator', 'histories']);

        // Apply filters
        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('due_datetime', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('due_datetime', '<=', $request->date_to);
        }

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('priority') && $request->priority) {
            $query->where('priority', $request->priority);
        }

        if ($request->has('lawyer_id') && $request->lawyer_id) {
            $query->where('lawyer_id', $request->lawyer_id);
        }

        if ($request->has('task_type') && $request->task_type) {
            $query->where('task_type', $request->task_type);
        }

        $assignments = $query->get();
        $lawyers = \App\Models\Lawyer::all();

        // Generate comprehensive statistics
        $stats = [
            'total_assignments' => $assignments->count(),
            'pending_assignments' => $assignments->where('status', 'pending')->count(),
            'in_progress_assignments' => $assignments->where('status', 'in_progress')->count(),
            'completed_assignments' => $assignments->where('status', 'completed')->count(),
            'overdue_assignments' => $assignments->filter(function($assignment) {
                return $assignment->due_datetime && $assignment->due_datetime < now() && $assignment->status != 'completed';
            })->count(),
            'cancelled_assignments' => $assignments->where('status', 'cancelled')->count(),
            'average_completion_time' => $this->calculateAverageCompletionTime($assignments),
            'total_estimated_hours' => $assignments->sum('estimated_hours'),
            'total_actual_hours' => $assignments->where('status', 'completed')->sum('estimated_hours'), // Assuming estimated_hours as proxy
            'billable_assignments' => $assignments->where('is_billable', true)->count(),
            'high_priority_assignments' => $assignments->where('priority', 'high')->count(),
        ];

        // Data for charts - formatted for Chart.js
        $statusData = [
            'labels' => ['Pending', 'In Progress', 'Completed', 'Cancelled'],
            'data' => [
                $assignments->where('status', 'pending')->count(),
                $assignments->where('status', 'in_progress')->count(),
                $assignments->where('status', 'completed')->count(),
                $assignments->where('status', 'cancelled')->count(),
            ]
        ];

        $priorityData = [
            'labels' => ['Low', 'Medium', 'High'],
            'data' => [
                $assignments->where('priority', 'low')->count(),
                $assignments->where('priority', 'medium')->count(),
                $assignments->where('priority', 'high')->count(),
            ]
        ];

        // Productivity data (completed assignments per lawyer)
        $productivityData = [
            'labels' => $lawyers->map(fn($lawyer) => $lawyer->first_name . ' ' . $lawyer->last_name)->toArray(),
            'data' => $lawyers->map(fn($lawyer) => $assignments->where('lawyer_id', $lawyer->id)->where('status', 'completed')->count())->toArray()
        ];

        // Deadline analysis
        $deadlineData = [
            'labels' => ['Overdue', 'Due Today', 'Due This Week', 'Due This Month'],
            'data' => [
                $assignments->where('due_datetime', '<', now())->where('status', '!=', 'completed')->count(),
                $assignments->whereDate('due_datetime', now()->toDateString())->where('status', '!=', 'completed')->count(),
                $assignments->whereBetween('due_datetime', [now(), now()->addDays(7)])->where('status', '!=', 'completed')->count(),
                $assignments->whereBetween('due_datetime', [now(), now()->addDays(30)])->where('status', '!=', 'completed')->count(),
            ]
        ];

        // Task type distribution
        $taskTypes = $assignments->groupBy('task_type')->map->count();
        $taskTypeData = [
            'labels' => $taskTypes->keys()->toArray(),
            'data' => $taskTypes->values()->toArray()
        ];

        // Monthly trend data (last 12 months)
        $monthlyData = $this->getMonthlyAssignmentData();

        // Detailed assignments for table view
        $detailedAssignments = $assignments->map(function ($assignment) {
            return [
                'id' => $assignment->id,
                'title' => $assignment->assignment_title,
                'lawyer' => $assignment->lawyer ? $assignment->lawyer->first_name . ' ' . $assignment->lawyer->last_name : 'N/A',
                'client' => $assignment->client ? $assignment->client->client_name : 'N/A',
                'case' => $assignment->legalCase ? $assignment->legalCase->case_title : 'N/A',
                'priority' => $assignment->priority,
                'status' => $assignment->status,
                'due_date' => $assignment->due_datetime ? $assignment->due_datetime->format('M d, Y H:i') : 'N/A',
                'progress' => $assignment->progress_percentage,
                'estimated_hours' => $assignment->estimated_hours,
                'created_by' => $assignment->creator ? $assignment->creator->name : 'N/A',
                'assigned_date' => $assignment->assigned_date ? $assignment->assigned_date->format('M d, Y') : 'N/A',
                'created_date' => $assignment->created_at ? $assignment->created_at->format('M d, Y') : 'N/A',
                'is_billable' => $assignment->is_billable,
                'description' => $assignment->description,
                'instructions' => $assignment->instructions,
                'documents_required' => $assignment->documents_required,
                'court_date' => $assignment->court_datetime ? $assignment->court_datetime->format('M d, Y H:i') : 'N/A',
                'court_location' => $assignment->court_location,
                'opposing_counsel' => $assignment->opposing_counsel,
                'client_contact_info' => $assignment->client_contact_info,
                'billing_notes' => $assignment->billing_notes,
                'histories' => $assignment->histories->map(function ($history) {
                    return [
                        'action' => $history->action,
                        'description' => $history->description,
                        'created_at' => $history->created_at->format('M d, Y H:i')
                    ];
                })
            ];
        });

        return view('modules.processing.assignment-reports', compact(
            'assignments',
            'detailedAssignments',
            'lawyers',
            'stats',
            'statusData',
            'priorityData',
            'productivityData',
            'deadlineData',
            'taskTypeData',
            'monthlyData'
        ));
    }

    private function calculateAverageCompletionTime($assignments)
    {
        $completedAssignments = $assignments->where('status', 'completed');
        if ($completedAssignments->isEmpty()) {
            return 0;
        }

        $totalDays = 0;
        foreach ($completedAssignments as $assignment) {
            if ($assignment->assigned_date && $assignment->due_datetime) {
                $totalDays += $assignment->assigned_date->diffInDays($assignment->due_datetime);
            }
        }

        return round($totalDays / $completedAssignments->count(), 1);
    }

    private function getMonthlyAssignmentData()
    {
        $data = [];
        for ($i = 11; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $monthName = $date->format('M Y');
            $assignments = \App\Models\Assignment::whereYear('created_at', $date->year)
                ->whereMonth('created_at', $date->month)
                ->count();
            $data['labels'][] = $monthName;
            $data['data'][] = $assignments;
        }
        return $data;
    }

    public function exportAssignmentReports(Request $request, $format)
    {
        $query = \App\Models\Assignment::with(['lawyer', 'client', 'legalCase', 'caseType', 'caseCategory', 'caseStatus', 'creator', 'histories']);

        // Apply filters
        if ($request->has('date_from') && $request->date_from) {
            $query->whereDate('due_datetime', '>=', $request->date_from);
        }

        if ($request->has('date_to') && $request->date_to) {
            $query->whereDate('due_datetime', '<=', $request->date_to);
        }

        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('priority') && $request->priority) {
            $query->where('priority', $request->priority);
        }

        if ($request->has('lawyer_id') && $request->lawyer_id) {
            $query->where('lawyer_id', $request->lawyer_id);
        }

        if ($request->has('task_type') && $request->task_type) {
            $query->where('task_type', $request->task_type);
        }

        $assignments = $query->get();

        switch ($format) {
            case 'pdf':
                return $this->exportAssignmentReportsPDF($assignments, $request);
            case 'excel':
                return $this->exportAssignmentReportsExcel($assignments, $request);
            case 'csv':
                return $this->exportAssignmentReportsCSV($assignments, $request);
            case 'print':
                // For print, return the same view but with print-friendly styling
                return $this->assignmentReports($request)->with('print_mode', true);
            default:
                return redirect()->back()->with('error', 'Invalid export format');
        }
    }

    private function exportAssignmentReportsPDF($assignments, $request)
    {
        // Create a comprehensive HTML report for PDF export
        $html = $this->generateAssignmentReportHTML($assignments, $request);

        // For now, return HTML content that can be printed as PDF
        // In production, you would use a PDF library like DomPDF, TCPDF, or wkhtmltopdf
        return response($html)
            ->header('Content-Type', 'text/html')
            ->header('Content-Disposition', 'attachment; filename="assignment-reports-' . now()->format('Y-m-d') . '.html"');
    }

    private function generateAssignmentReportHTML($assignments, $request)
    {
        $stats = $this->calculateStats($assignments);
        $lawyers = \App\Models\Lawyer::all();

        $html = '
        <!DOCTYPE html>
        <html>
        <head>
            <title>Assignment Reports - ' . now()->format('Y-m-d') . '</title>
            <style>
                body { font-family: Arial, sans-serif; margin: 20px; }
                .header { text-align: center; margin-bottom: 30px; }
                .stats { display: flex; justify-content: space-around; margin-bottom: 30px; }
                .stat-card { border: 1px solid #ddd; padding: 15px; border-radius: 5px; text-align: center; }
                .stat-number { font-size: 24px; font-weight: bold; color: #007bff; }
                .stat-label { color: #666; }
                table { width: 100%; border-collapse: collapse; margin-top: 20px; }
                th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
                th { background-color: #f8f9fa; font-weight: bold; }
                .priority-high { background-color: #fee2e2; }
                .priority-medium { background-color: #fef3c7; }
                .priority-low { background-color: #dbeafe; }
                .status-completed { background-color: #d1fae5; }
                .status-in_progress { background-color: #dbeafe; }
                .status-pending { background-color: #fef3c7; }
                .status-cancelled { background-color: #fee2e2; }
                .footer { margin-top: 30px; text-align: center; color: #666; font-size: 12px; }
            </style>
        </head>
        <body>
            <div class="header">
                <h1>Assignment Reports</h1>
                <p>Generated on ' . now()->format('F d, Y H:i') . '</p>
            </div>

            <div class="stats">
                <div class="stat-card">
                    <div class="stat-number">' . $stats['total_assignments'] . '</div>
                    <div class="stat-label">Total Assignments</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number">' . $stats['completed_assignments'] . '</div>
                    <div class="stat-label">Completed</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number">' . $stats['in_progress_assignments'] . '</div>
                    <div class="stat-label">In Progress</div>
                </div>
                <div class="stat-card">
                    <div class="stat-number">' . $stats['overdue_assignments'] . '</div>
                    <div class="stat-label">Overdue</div>
                </div>
            </div>

            <h2>Assignment Details</h2>
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Title</th>
                        <th>Lawyer</th>
                        <th>Client</th>
                        <th>Case</th>
                        <th>Priority</th>
                        <th>Status</th>
                        <th>Due Date</th>
                        <th>Progress</th>
                        <th>Estimated Hours</th>
                    </tr>
                </thead>
                <tbody>';

        foreach ($assignments as $assignment) {
            $html .= '
                    <tr>
                        <td>' . $assignment->id . '</td>
                        <td>' . htmlspecialchars($assignment->assignment_title) . '</td>
                        <td>' . ($assignment->lawyer ? htmlspecialchars($assignment->lawyer->first_name . ' ' . $assignment->lawyer->last_name) : 'N/A') . '</td>
                        <td>' . ($assignment->client ? htmlspecialchars($assignment->client->client_name) : 'N/A') . '</td>
                        <td>' . ($assignment->legalCase ? htmlspecialchars($assignment->legalCase->case_title) : 'N/A') . '</td>
                        <td class="priority-' . $assignment->priority . '">' . ucfirst($assignment->priority) . '</td>
                        <td class="status-' . $assignment->status . '">' . ucfirst(str_replace('_', ' ', $assignment->status)) . '</td>
                        <td>' . ($assignment->due_datetime ? \Carbon\Carbon::parse($assignment->due_datetime)->format('M d, Y H:i') : 'N/A') . '</td>
                        <td>' . $assignment->progress_percentage . '%</td>
                        <td>' . ($assignment->estimated_hours ?? 'N/A') . '</td>
                    </tr>';
        }

        $html .= '
                </tbody>
            </table>

            <div class="footer">
                <p>Report generated by Prolex Law Firm Management System</p>
            </div>
        </body>
        </html>';

        return $html;
    }

    private function calculateStats($assignments)
    {
        return [
            'total_assignments' => $assignments->count(),
            'pending_assignments' => $assignments->where('status', 'pending')->count(),
            'in_progress_assignments' => $assignments->where('status', 'in_progress')->count(),
            'completed_assignments' => $assignments->where('status', 'completed')->count(),
            'overdue_assignments' => $assignments->where('due_datetime', '<', now())->where('status', '!=', 'completed')->count(),
            'cancelled_assignments' => $assignments->where('status', 'cancelled')->count(),
            'average_completion_time' => $this->calculateAverageCompletionTime($assignments),
            'total_estimated_hours' => $assignments->sum('estimated_hours'),
            'total_actual_hours' => $assignments->where('status', 'completed')->sum('estimated_hours'),
            'billable_assignments' => $assignments->where('is_billable', true)->count(),
            'high_priority_assignments' => $assignments->where('priority', 'high')->count(),
        ];
    }

    private function exportAssignmentReportsExcel($assignments, $request)
    {
        $filename = 'assignment-reports-' . now()->format('Y-m-d') . '.xlsx';

        // Create CSV content that Excel can open
        $headers = [
            'ID',
            'Title',
            'Lawyer',
            'Client',
            'Case',
            'Priority',
            'Status',
            'Due Date',
            'Progress (%)',
            'Estimated Hours',
            'Created Date',
            'Assigned Date'
        ];

        $rows = [];
        foreach ($assignments as $assignment) {
            $rows[] = [
                $assignment->id,
                $assignment->assignment_title,
                $assignment->lawyer ? $assignment->lawyer->first_name . ' ' . $assignment->lawyer->last_name : 'N/A',
                $assignment->client ? $assignment->client->client_name : 'N/A',
                $assignment->legalCase ? $assignment->legalCase->case_title : 'N/A',
                ucfirst($assignment->priority),
                ucfirst(str_replace('_', ' ', $assignment->status)),
                $assignment->due_datetime ? $assignment->due_datetime->format('M d, Y H:i') : 'N/A',
                $assignment->progress_percentage,
                $assignment->estimated_hours,
                $assignment->created_at ? $assignment->created_at->format('M d, Y') : 'N/A',
                $assignment->assigned_date ? $assignment->assigned_date->format('M d, Y') : 'N/A'
            ];
        }

        $content = implode("\n", array_map(function($row) {
            return implode(',', array_map(function($cell) {
                return '"' . str_replace('"', '""', $cell) . '"';
            }, $row));
        }, array_merge([$headers], $rows)));

        return response($content)
            ->header('Content-Type', 'text/csv')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
    }

    private function exportAssignmentReportsCSV($assignments, $request)
    {
        $filename = 'assignment-reports-' . now()->format('Y-m-d') . '.csv';

        $headers = [
            'ID',
            'Title',
            'Lawyer',
            'Client',
            'Case',
            'Priority',
            'Status',
            'Due Date',
            'Progress (%)',
            'Estimated Hours',
            'Created Date',
            'Assigned Date',
            'Description',
            'Instructions'
        ];

        $rows = [];
        foreach ($assignments as $assignment) {
            $rows[] = [
                $assignment->id,
                $assignment->assignment_title,
                $assignment->lawyer ? $assignment->lawyer->first_name . ' ' . $assignment->lawyer->last_name : 'N/A',
                $assignment->client ? $assignment->client->client_name : 'N/A',
                $assignment->legalCase ? $assignment->legalCase->case_title : 'N/A',
                ucfirst($assignment->priority),
                ucfirst(str_replace('_', ' ', $assignment->status)),
                $assignment->due_datetime ? $assignment->due_datetime->format('M d, Y H:i') : 'N/A',
                $assignment->progress_percentage,
                $assignment->estimated_hours,
                $assignment->created_at ? $assignment->created_at->format('M d, Y') : 'N/A',
                $assignment->assigned_date ? $assignment->assigned_date->format('M d, Y') : 'N/A',
                $assignment->description,
                $assignment->instructions
            ];
        }

        $content = implode("\n", array_map(function($row) {
            return implode(',', array_map(function($cell) {
                return '"' . str_replace('"', '""', $cell) . '"';
            }, $row));
        }, array_merge([$headers], $rows)));

        return response($content)
            ->header('Content-Type', 'text/csv')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
    }

    // Additional assignment methods to match routes/task-management.php
    public function assignmentAdd()
    {
        return $this->createAssignment();
    }

    public function assignmentList(Request $request)
    {
        return $this->listAssignments($request);
    }

    public function assignmentShow($id)
    {
        return $this->showAssignment($id);
    }

    public function assignmentEdit($id)
    {
        return $this->editAssignment($id);
    }

    public function assignmentUpdate(Request $request, $id)
    {
        return $this->updateAssignment($request, $id);
    }

    public function assignmentDelete($id)
    {
        return $this->deleteAssignment($id);
    }

    public function assignmentStore(Request $request)
    {
        return $this->storeAssignment($request);
    }
}
