<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use App\Mail\OtpMail;

class ForgotPasswordController extends Controller
{
    public function showForgotPasswordForm()
    {
        return view('auth.forgot-password');
    }

    public function sendOtp(Request $request)
    {
        $request->validate([
            'email' => ['required', 'email', 'exists:users,email'],
        ]);

        $code = random_int(100000, 999999);

        DB::table('password_otps')->updateOrInsert(
            ['email' => $request->email],
            ['otp' => $code, 'created_at' => now(), 'expires_at' => now()->addMinutes(10)]
        );

        try {
            Mail::to($request->email)->send(new OtpMail($code));

            // Store email in session for later steps
            session(['email' => $request->email]);

            return redirect()->route('password.verify')->with('message', 'OTP sent to your email.');
        } catch (\Exception $e) {
            return back()->withErrors(['email' => 'Failed to send OTP. Please try again later.'])->withInput();
        }
    }

    public function showVerifyOtpForm()
    {
        if (!session('email')) {
            return redirect()->route('password.request')->withErrors(['email' => 'Session expired. Please try again.']);
        }

        return view('auth.verify-otp', ['email' => session('email')]);
    }

    public function verifyOtp(Request $request)
    {
        $request->validate([
            'otp' => ['required', 'digits:6'],
        ]);

        $email = session('email');
        if (!$email) {
            return redirect()->route('password.request')->withErrors(['email' => 'Session expired. Please try again.']);
        }

        $record = DB::table('password_otps')
            ->where('email', $email)
            ->where('otp', $request->otp)
            ->where('expires_at', '>', now())
            ->first();

        if (!$record) {
            return back()->withErrors(['otp' => 'Invalid or expired OTP.'])->withInput();
        }

        // Mark OTP as verified in session
        session(['otp_verified' => true]);

        return redirect()->route('password.reset')->with('message', 'OTP verified. You may now reset your password.');
    }

    public function showResetPasswordForm()
    {
        if (!session('email') || !session('otp_verified')) {
            return redirect()->route('password.request')->withErrors(['email' => 'Session expired or OTP not verified. Please try again.']);
        }

        return view('auth.reset-password', ['email' => session('email')]);
    }

    public function resetPassword(Request $request)
    {
        $request->validate([
            'password' => ['required', 'confirmed', 'min:8'],
        ]);

        $email = session('email');
        if (!$email || !session('otp_verified')) {
            return redirect()->route('password.request')->withErrors(['email' => 'Session expired or OTP not verified. Please try again.']);
        }

        $user = User::where('email', $email)->first();

        if (!$user) {
            return back()->withErrors(['email' => 'User not found.'])->withInput();
        }

        $user->update(['password_hash' => Hash::make($request->password)]);

        DB::table('password_otps')->where('email', $email)->delete();

        // Clear session
        session()->forget(['email', 'otp_verified']);

        return redirect()->route('login')->with('success', 'Password reset successful. You may now log in.');
    }
}
