<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CaseController extends Controller
{
    public function index()
    {
        return $this->cases();
    }

    public function create()
    {
        $clients = \App\Models\Client::all();
        $lawyers = \App\Models\Lawyer::all();
        $caseTypes = \App\Models\CaseType::all();
        $caseCategories = \App\Models\CaseCategory::all();
        $caseStatuses = \App\Models\CaseStatus::all();
        $caseReasons = \App\Models\CaseReason::all();

        return view('modules.edit.add-case', compact('clients', 'lawyers', 'caseTypes', 'caseCategories', 'caseStatuses', 'caseReasons'));
    }

    public function cases()
    {
        $cases = \App\Models\LegalCase::with(['client', 'lawyer', 'caseType', 'caseCategory', 'caseStatus'])->get();
        return view('modules.edit.case-list', compact('cases'));
    }

    public function addCase()
    {
        $clients = \App\Models\Client::all();
        $lawyers = \App\Models\Lawyer::all();
        $caseTypes = \App\Models\CaseType::all();
        $caseCategories = \App\Models\CaseCategory::all();
        $caseStatuses = \App\Models\CaseStatus::all();

        return view('modules.processing.add-case', compact('clients', 'lawyers', 'caseTypes', 'caseCategories', 'caseStatuses'));
    }

    public function store(Request $request)
    {
        $allowedRoles = \App\Models\LawyerRole::pluck('name')->toArray();

        $validated = $request->validate([
            'case_id' => 'required|string|max:255',
            'case_name' => 'required|string|max:255',
            'client_id' => 'required|integer|exists:clients,id',
            'category_code' => 'required|string|max:255',
            'type_id' => 'required|integer|exists:case_types,id',
            'priority_id' => 'required|integer|exists:case_priorities,id',
            'role_id' => 'required|integer|exists:case_roles,id',
            'date_of_incident' => 'nullable|date',
            'court_appearance_date' => 'nullable|date',
            'notes' => 'nullable|string',
            'lawyers' => 'nullable|array',
            'lawyers.*.lawyer_id' => 'required|string|exists:lawyers,lawyer_id',
            'lawyers.*.role' => 'required|string|in:' . implode(',', $allowedRoles),
        ]);

        try {
            DB::transaction(function () use ($validated) {
                // Create the case
                $case = \App\Models\LegalCase::create([
                    'case_id' => $validated['case_id'],
                    'case_name' => $validated['case_name'],
                    'client_id' => $validated['client_id'],
                    'category_code' => $validated['category_code'],
                    'type_id' => $validated['type_id'],
                    'priority_id' => $validated['priority_id'],
                    'role_id' => $validated['role_id'],
                    'date_of_incident' => $validated['date_of_incident'] ?? null,
                    'court_appearance_date' => $validated['court_appearance_date'] ?? null,
                    'notes' => $validated['notes'] ?? null,
                ]);

                // Create case lawyers if provided
                if (isset($validated['lawyers'])) {
                    foreach ($validated['lawyers'] as $lawyerData) {
                        $role = \App\Models\LawyerRole::where('name', $lawyerData['role'])->first();
                        if (!$role) {
                            throw new \Exception('Invalid role: ' . $lawyerData['role']);
                        }
                        $case->caseLawyers()->create([
                            'lawyer_id' => $lawyerData['lawyer_id'],
                            'lawyer_role_id' => $role->lawyer_role_id,
                        ]);
                    }
                }
            });

            return redirect()->route('modules.edit.cases')->with('success', 'Case added successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->withErrors(['error' => 'Failed to save case: ' . $e->getMessage()]);
        }
    }

    public function edit($id)
    {
        $case = \App\Models\LegalCase::with(['client', 'lawyer', 'caseType', 'caseCategory', 'caseStatus'])->findOrFail($id);
        $clients = \App\Models\Client::all();
        $lawyers = \App\Models\Lawyer::all();
        $caseTypes = \App\Models\CaseType::all();
        $caseCategories = \App\Models\CaseCategory::all();
        $caseStatuses = \App\Models\CaseStatus::all();
        $caseOutcomes = \App\Models\CaseOutcome::all();

        return view('modules.edit.edit-case', compact('case', 'clients', 'lawyers', 'caseTypes', 'caseCategories', 'caseStatuses', 'caseOutcomes'));
    }

    public function editCase($id)
    {
        $case = \App\Models\LegalCase::findOrFail($id);
        $clients = \App\Models\Client::all();
        $lawyers = \App\Models\Lawyer::all();
        $caseTypes = \App\Models\CaseType::all();
        $caseCategories = \App\Models\CaseCategory::all();
        $caseStatuses = \App\Models\CaseStatus::all();
        $caseOutcomes = \App\Models\CaseOutcome::all();

        return view('modules.processing.edit-case', compact('case', 'clients', 'lawyers', 'caseTypes', 'caseCategories', 'caseStatuses', 'caseOutcomes'));
    }

    public function update(Request $request, $id)
    {
        $case = \App\Models\LegalCase::findOrFail($id);

        $validated = $request->validate([
            'case_id' => 'required|string|max:255',
            'type' => 'required|string|max:255',
            'client_name' => 'required|string|max:255',
            'case_category_id' => 'required|integer|exists:case_categories,id',
            'lawyer_id' => 'required|integer|exists:lawyers,id',
            'case_status_id' => 'required|integer|exists:case_statuses,id',
            'case_outcome_id' => 'nullable|integer|exists:case_outcomes,id',
        ]);

        $case->update([
            'case_code' => $validated['case_id'],
            'type_name' => $validated['type'],
            'case_category_id' => $validated['case_category_id'],
            'lawyer_id' => $validated['lawyer_id'],
            'status_code' => $validated['case_status_id'],
            'case_outcome' => $validated['case_outcome_id'],
        ]);

        return redirect()->route('modules.edit.cases')->with('success', 'Case updated successfully.');
    }

    public function updateCase(Request $request, $id)
    {
        $case = \App\Models\LegalCase::findOrFail($id);

        $validated = $request->validate([
            'case_title' => 'required|string|max:255',
            'case_number' => 'nullable|string|max:100',
            'client_id' => 'required|integer|exists:clients,id',
            'lawyer_id' => 'nullable|integer|exists:lawyers,id',
            'case_type_id' => 'required|integer|exists:case_types,id',
            'case_category_id' => 'required|integer|exists:case_categories,id',
            'case_status_id' => 'required|integer|exists:case_statuses,id',
            'filing_date' => 'nullable|date',
            'court_date' => 'nullable|date',
            'description' => 'nullable|string',
            'opposing_party' => 'nullable|string|max:255',
            'opposing_lawyer' => 'nullable|string|max:255',
            'court_name' => 'nullable|string|max:255',
            'case_reference' => 'nullable|string|max:255',
            'estimated_value' => 'nullable|numeric|min:0',
            'actual_cost' => 'nullable|numeric|min:0',
            'priority' => 'required|in:low,medium,high',
            'status' => 'required|in:active,closed,settled',
        ]);

        $case->update($validated);

        return redirect()->route('modules.processing.cases')->with('success', 'Case updated successfully.');
    }

    public function destroyCase($id)
    {
        $case = \App\Models\LegalCase::findOrFail($id);
        $case->delete();

        return redirect()->route('modules.processing.cases')->with('success', 'Case deleted successfully.');
    }

    public function showCase($id)
    {
        $case = \App\Models\LegalCase::with(['client', 'lawyer', 'caseType', 'caseCategory', 'caseStatus'])->findOrFail($id);
        return view('modules.processing.case-show', compact('case'));
    }

    public function show($id)
    {
        return $this->showCase($id);
    }

    public function caseReasons()
    {
        $caseReasons = \App\Models\CaseReason::all();
        return view('modules.processing.case-reasons', compact('caseReasons'));
    }

    public function caseStatuses()
    {
        $caseStatuses = \App\Models\CaseStatus::all();
        return view('modules.processing.case-statuses', compact('caseStatuses'));
    }

    public function caseTypes()
    {
        $caseTypes = \App\Models\CaseType::all();
        return view('modules.edit.case.case-types', compact('caseTypes'));
    }

    public function caseCategories()
    {
        $caseCategories = \App\Models\CaseCategory::all();
        return view('modules.processing.case-categories', compact('caseCategories'));
    }
}
