<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class ClientController extends Controller
{
    public function clients()
    {
        $clients = \App\Models\Client::all();
        return view('modules.processing.clients', compact('clients'));
    }

    public function addClients()
    {
        $categories = \App\Models\ClientCategory::all();
        $clients = \App\Models\Client::all();
        return view('modules.edit.add-clients', compact('categories', 'clients'));
    }

    public function storeClient(Request $request)
    {
        $validated = $request->validate([
            'client_id' => 'required|string|max:10|unique:clients,client_id',
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'phone_primary' => 'required|string|max:20',
            'email' => 'required|email|max:191',
            'national_id_passport' => 'nullable|string|max:191|unique:clients,national_id',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|string|max:20',
            'address_street' => 'nullable|string',
            'address_city' => 'nullable|string|max:100',
            'address_state' => 'nullable|string|max:100',
            'address_country' => 'nullable|string|max:100',
            'occupation_company' => 'nullable|string|max:150',
            'position_role' => 'nullable|string|max:100',
            'client_type' => 'nullable|in:IND,CORP,GOVT,NGO,SME,FIN,REAL,LEG',
            'client_category' => 'nullable|string|max:50',
            'date_onboarding' => 'nullable|date',
            'assigned_lawyer_case_manager' => 'nullable|string|max:100',
            'referred_lawyer' => 'nullable|string|max:100',
            'notes_remarks' => 'nullable|string',
        ]);

        $data = [
            'client_id' => $validated['client_id'],
            'first_name' => $validated['first_name'],
            'last_name' => $validated['last_name'],
            'phone_primary' => $validated['phone_primary'],
            'email' => $validated['email'],
            'national_id' => $validated['national_id_passport'] ?? null,
            'date_of_birth' => $validated['date_of_birth'] ?? null,
            'gender' => $validated['gender'] ?? null,
            'street_address' => $validated['address_street'] ?? null,
            'city' => $validated['address_city'] ?? null,
            'state_province' => $validated['address_state'] ?? null,
            'country' => $validated['address_country'] ?? null,
            'employer_name' => $validated['occupation_company'] ?? null,
            'position_role' => $validated['position_role'] ?? null,
            'client_type' => $validated['client_type'] ?? null,
            'client_category' => $validated['client_category'] ?? null,
            'date_onboarded' => $validated['date_onboarding'] ?? null,
            'assigned_lawyer' => $validated['assigned_lawyer_case_manager'] ?? null,
            'referred_lawyer' => $validated['referred_lawyer'] ?? null,
            'notes' => $validated['notes_remarks'] ?? null,
        ];

        \App\Models\Client::create($data);

        return redirect()->route('modules.edit.add-clients')->with('success', 'Client added successfully.');
    }

    public function editClient($id)
    {
        $client = \App\Models\Client::findOrFail($id);
        $categories = \App\Models\ClientCategory::all();

        // Get lawyer names for display
        $assignedLawyerName = null;
        $referredLawyerName = null;

        if ($client->assigned_lawyer) {
            $assignedLawyer = \App\Models\Lawyer::where('lawyer_id', $client->assigned_lawyer)->first();
            $assignedLawyerName = $assignedLawyer ? $assignedLawyer->first_name . ' ' . $assignedLawyer->last_name : null;
        }

        if ($client->referred_lawyer) {
            $referredLawyer = \App\Models\Lawyer::where('lawyer_id', $client->referred_lawyer)->first();
            $referredLawyerName = $referredLawyer ? $referredLawyer->first_name . ' ' . $referredLawyer->last_name : null;
        }

        return view('modules.edit.edit-client', compact('client', 'categories', 'assignedLawyerName', 'referredLawyerName'));
    }

    public function updateClient(Request $request, $id)
    {
        $client = \App\Models\Client::findOrFail($id);

        $validated = $request->validate([
            'client_id' => 'required|string|max:10|unique:clients,client_id,' . $id,
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'phone_primary' => 'required|string|max:20',
            'email' => 'required|email|max:191',
            'national_id_passport' => 'nullable|string|max:191|unique:clients,national_id,' . $id,
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|string|max:20',
            'address_street' => 'nullable|string',
            'address_city' => 'nullable|string|max:100',
            'address_state' => 'nullable|string|max:100',
            'address_country' => 'nullable|string|max:100',
            'occupation_company' => 'nullable|string|max:150',
            'position_role' => 'nullable|string|max:100',
            'client_type' => 'nullable|in:IND,CORP,GOVT,NGO,SME,FIN,REAL,LEG',
            'client_category' => 'nullable|string|max:50',
            'date_onboarding' => 'nullable|date',
            'assigned_lawyer_case_manager' => 'nullable|string|max:100',
            'referred_lawyer' => 'nullable|string|max:100',
            'notes_remarks' => 'nullable|string',
        ]);

        $data = [
            'client_id' => $validated['client_id'],
            'first_name' => $validated['first_name'],
            'last_name' => $validated['last_name'],
            'phone_primary' => $validated['phone_primary'],
            'email' => $validated['email'],
            'national_id' => $validated['national_id_passport'] ?? null,
            'date_of_birth' => $validated['date_of_birth'] ?? null,
            'gender' => $validated['gender'] ?? null,
            'street_address' => $validated['address_street'] ?? null,
            'city' => $validated['address_city'] ?? null,
            'state_province' => $validated['address_state'] ?? null,
            'country' => $validated['address_country'] ?? null,
            'employer_name' => $validated['occupation_company'] ?? null,
            'position_role' => $validated['position_role'] ?? null,
            'client_type' => $validated['client_type'] ?? null,
            'client_category' => $validated['client_category'] ?? null,
            'date_onboarded' => $validated['date_onboarding'] ?? null,
            'assigned_lawyer' => $validated['assigned_lawyer_case_manager'] ?? null,
            'referred_lawyer' => $validated['referred_lawyer'] ?? null,
            'notes' => $validated['notes_remarks'] ?? null,
        ];

        $client->update($data);

        return redirect()->route('modules.edit.add-clients')->with('success', 'Client updated successfully.');
    }

    public function destroyClient($id)
    {
        $client = \App\Models\Client::findOrFail($id);
        $client->delete();

        return redirect()->route('modules.edit.add-clients')->with('success', 'Client deleted successfully.');
    }

    public function showClient($id)
    {
        $client = \App\Models\Client::findOrFail($id);
        return view('modules.processing.client-show', compact('client'));
    }

    // Alias methods for route compatibility
    public function index()
    {
        return view('modules.edit.client');
    }

    public function create()
    {
        return $this->addClients();
    }

    public function store(Request $request)
    {
        return $this->storeClient($request);
    }

    public function show($id)
    {
        return $this->showClient($id);
    }

    public function edit($id)
    {
        return $this->editClient($id);
    }

    public function update(Request $request, $id)
    {
        return $this->updateClient($request, $id);
    }

    public function destroy($id)
    {
        return $this->destroyClient($id);
    }

    public function searchLawyers(Request $request)
    {
        $query = $request->get('q', '');
        $query = trim($query);

        if (strlen($query) < 2) {
            return response()->json([]);
        }

        $lawyers = \App\Models\Lawyer::query();

        // Split query by space to handle first and last name search
        $parts = explode(' ', $query, 2);

        if (count($parts) === 1) {
            // Single word: search in both first_name and last_name
            $searchTerm = $parts[0];
            $lawyers->where(function ($q) use ($searchTerm) {
                $q->where('first_name', 'LIKE', '%' . $searchTerm . '%')
                  ->orWhere('last_name', 'LIKE', '%' . $searchTerm . '%');
            });
        } else {
            // Two words: first word searches first_name, second word searches last_name
            $firstName = $parts[0];
            $lastName = $parts[1];

            $lawyers->where(function ($q) use ($firstName, $lastName) {
                $q->where('first_name', 'LIKE', '%' . $firstName . '%')
                  ->where('last_name', 'LIKE', '%' . $lastName . '%');
            });
        }

        $results = $lawyers->limit(10)->get(['lawyer_id', 'first_name', 'last_name']);

        // Format results with full_name
        $formattedResults = $results->map(function ($lawyer) {
            return [
                'lawyer_id' => $lawyer->lawyer_id,
                'full_name' => $lawyer->first_name . ' ' . $lawyer->last_name,
            ];
        });

        return response()->json($formattedResults);
    }
}
