<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class ClientInvoiceController extends Controller
{
    public function clientInvoice()
    {
        $clients = \App\Models\Client::all();
        $accounts = \App\Models\MainAccount::with('accountType')->get();
        $glAccounts = \App\Models\MainAccount::with('accountType')->get();
        $taxes = \App\Models\Tax::all();
        $currencies = \App\Models\Currency::all();
        $clientTransactions = \App\Models\ClientInvoice::with('client')->get();

        return view('modules.processing.client-invoice', compact('clients', 'accounts', 'glAccounts', 'taxes', 'currencies', 'clientTransactions'));
    }

    public function storeClientInvoice(Request $request)
    {
        $validated = $request->validate([
            'client_id' => 'required|integer|exists:clients,id',
            'invoice_number' => 'required|string|max:255',
            'invoice_date' => 'required|date',
            'due_date' => 'nullable|date',
            'amount' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'account' => 'nullable|string|max:255',
            'narration' => 'nullable|string',
            'reference' => 'nullable|string|max:255',
            'gl_account' => 'nullable|string|max:255',
            'tax_id' => 'nullable|integer|exists:taxes,id',
            'discount' => 'nullable|numeric|min:0|max:100',
            'price' => 'required|numeric|min:0',
            'quantity' => 'nullable|integer|min:1',
            'currency_id' => 'nullable|integer|exists:currencies,id',
            'operation_cost' => 'nullable|boolean',
            'tax_amount' => 'nullable|numeric|min:0',
            'case_reference' => 'nullable|string|max:255',
            'responsible_lawyer' => 'nullable|string|max:255',
            'disbursements' => 'nullable|numeric|min:0',
            'total' => 'nullable|numeric|min:0',
        ]);

        $validated['total'] = ($validated['amount'] ?? 0) + ($validated['tax_amount'] ?? 0) + ($validated['disbursements'] ?? 0);

        \App\Models\ClientInvoice::create($validated);

        return redirect()->back()->with('success', 'Client invoice created successfully.');
    }

    public function updateClientInvoice(Request $request, $id)
    {
        $invoice = \App\Models\ClientInvoice::findOrFail($id);

        $validated = $request->validate([
            'client_id' => 'required|integer|exists:clients,id',
            'invoice_number' => 'required|string|max:255',
            'invoice_date' => 'required|date',
            'due_date' => 'nullable|date',
            'amount' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'account' => 'nullable|string|max:255',
            'narration' => 'nullable|string',
            'reference' => 'nullable|string|max:255',
            'gl_account' => 'nullable|string|max:255',
            'tax_id' => 'nullable|integer|exists:taxes,id',
            'discount' => 'nullable|numeric|min:0|max:100',
            'price' => 'required|numeric|min:0',
            'quantity' => 'nullable|integer|min:1',
            'currency_id' => 'nullable|integer|exists:currencies,id',
            'operation_cost' => 'nullable|boolean',
            'tax_amount' => 'nullable|numeric|min:0',
            'case_reference' => 'nullable|string|max:255',
            'responsible_lawyer' => 'nullable|string|max:255',
            'disbursements' => 'nullable|numeric|min:0',
            'total' => 'nullable|numeric|min:0',
        ]);

        $validated['total'] = ($validated['amount'] ?? 0) + ($validated['tax_amount'] ?? 0) + ($validated['disbursements'] ?? 0);

        $invoice->update($validated);

        return redirect()->back()->with('success', 'Client invoice updated successfully.');
    }

    public function destroyClientInvoice($id)
    {
        $invoice = \App\Models\ClientInvoice::findOrFail($id);
        $invoice->delete();

        return redirect()->back()->with('success', 'Client invoice deleted successfully.');
    }

    public function processClientInvoice($invoice = null)
    {
        // Process client invoice(s)
        if ($invoice) {
            $invoice = \App\Models\ClientInvoice::findOrFail($invoice);
            $invoice->update(['status' => 'Processed']);
            $message = 'Client invoice processed successfully.';
        } else {
            $invoices = \App\Models\ClientInvoice::where('status', '!=', 'Processed')->get();
            foreach ($invoices as $inv) {
                $inv->update(['status' => 'Processed']);
            }
            $message = 'All client invoices processed successfully.';
        }

        return redirect()->back()->with('success', $message);
    }
}
