<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class ClientQuotationController extends Controller
{
    public function clientQuotation()
    {
        $clients = \App\Models\Client::all();
        $clientQuotations = \App\Models\ClientQuotation::with(['client', 'lines'])->get();

        return view('modules.processing.client-quotation', compact('clients', 'clientQuotations'));
    }

    public function storeClientQuotation(Request $request)
    {
        $validated = $request->validate([
            'client_id' => 'required|integer|exists:clients,id',
            'quotation_date' => 'required|date',
            'terms' => 'nullable|string',
            'validity_period' => 'nullable|integer|min:1|max:365',
            'notes' => 'nullable|string',
            'services' => 'required|array|min:1',
            'services.*.description' => 'required|string',
            'services.*.quantity' => 'required|integer|min:1',
            'services.*.unit_price' => 'required|numeric|min:0',
        ]);

        // Generate quotation number
        $quotationNumber = 'QUO-' . date('Y') . '-' . str_pad(\App\Models\ClientQuotation::count() + 1, 4, '0', STR_PAD_LEFT);

        // Calculate totals
        $totalAmount = 0;
        foreach ($validated['services'] as $service) {
            $totalAmount += $service['quantity'] * $service['unit_price'];
        }

        // Create quotation
        $quotation = \App\Models\ClientQuotation::create([
            'client_id' => $validated['client_id'],
            'quotation_number' => $quotationNumber,
            'quotation_date' => $validated['quotation_date'],
            'terms' => $validated['terms'] ?? null,
            'validity_period' => $validated['validity_period'] ?? 30,
            'notes' => $validated['notes'] ?? null,
            'status' => 'Draft',
        ]);

        // Create quotation lines
        foreach ($validated['services'] as $service) {
            $quotation->lines()->create([
                'description' => $service['description'],
                'quantity' => $service['quantity'],
                'unit_price' => $service['unit_price'],
                'total' => $service['quantity'] * $service['unit_price'],
            ]);
        }

        return redirect()->back()->with('success', 'Client quotation created successfully.');
    }

    public function updateClientQuotation(Request $request, $id)
    {
        $quotation = \App\Models\ClientQuotation::findOrFail($id);

        $validated = $request->validate([
            'client_id' => 'required|integer|exists:clients,id',
            'quotation_date' => 'required|date',
            'terms' => 'nullable|string',
            'validity_period' => 'nullable|integer|min:1|max:365',
            'notes' => 'nullable|string',
            'services' => 'required|array|min:1',
            'services.*.description' => 'required|string',
            'services.*.quantity' => 'required|integer|min:1',
            'services.*.unit_price' => 'required|numeric|min:0',
        ]);

        // Calculate totals
        $totalAmount = 0;
        foreach ($validated['services'] as $service) {
            $totalAmount += $service['quantity'] * $service['unit_price'];
        }

        // Update quotation
        $quotation->update([
            'client_id' => $validated['client_id'],
            'quotation_date' => $validated['quotation_date'],
            'terms' => $validated['terms'] ?? null,
            'validity_period' => $validated['validity_period'] ?? 30,
            'notes' => $validated['notes'] ?? null,
        ]);

        // Delete old lines and create new ones
        $quotation->lines()->delete();
        foreach ($validated['services'] as $service) {
            $quotation->lines()->create([
                'description' => $service['description'],
                'quantity' => $service['quantity'],
                'unit_price' => $service['unit_price'],
                'total' => $service['quantity'] * $service['unit_price'],
            ]);
        }

        return redirect()->back()->with('success', 'Client quotation updated successfully.');
    }

    public function destroyClientQuotation($id)
    {
        $quotation = \App\Models\ClientQuotation::findOrFail($id);
        $quotation->delete();

        return redirect()->back()->with('success', 'Client quotation deleted successfully.');
    }

    public function processClientQuotation($quotation = null)
    {
        if ($quotation) {
            $quotation = \App\Models\ClientQuotation::findOrFail($quotation);
            $quotation->update(['status' => 'Sent']);
            $message = 'Client quotation sent successfully.';
        } else {
            $quotations = \App\Models\ClientQuotation::where('status', 'Draft')->get();
            foreach ($quotations as $quo) {
                $quo->update(['status' => 'Sent']);
            }
            $message = 'All draft quotations sent successfully.';
        }

        return redirect()->back()->with('success', $message);
    }

    public function editClientQuotation($id)
    {
        $quotation = \App\Models\ClientQuotation::with('lines')->findOrFail($id);
        $clients = \App\Models\Client::all();
        $clientQuotations = \App\Models\ClientQuotation::with(['client', 'lines'])->get();

        return view('modules.processing.client-quotation', compact('clients', 'clientQuotations', 'quotation'));
    }
}
