<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class ClientReceiptController extends Controller
{
    public function clientReceipt()
    {
        $clients = \App\Models\Client::all();
        $accounts = \App\Models\MainAccount::with('accountType')->get();
        $glAccounts = \App\Models\MainAccount::with('accountType')->get();
        $taxes = \App\Models\Tax::all();
        $currencies = \App\Models\Currency::all();
        $clientReceipts = \App\Models\ClientReceipt::with(['client', 'lines'])->get();
        $paymentMethods = ['Cash', 'Bank Transfer', 'Cheque', 'Credit Card'];

        return view('modules.processing.client-receipt', compact('clients', 'accounts', 'glAccounts', 'taxes', 'currencies', 'clientReceipts', 'paymentMethods'));
    }

    public function storeClientReceipt(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|integer|exists:clients,id',
            'receipt_number' => 'nullable|string|max:255',
            'receipt_date' => 'required|date',
            'payment_method' => 'required|string|in:Cash,Bank Transfer,Cheque,Credit Card',
            'lines' => 'required|array|min:1',
            'lines.*.description' => 'required|string',
            'lines.*.invoice_id' => 'nullable|integer|exists:client_invoices,id',
            'lines.*.account_id' => 'required|integer|exists:main_accounts,id',
            'lines.*.amount' => 'required|numeric|min:0.01',
            'lines.*.reference' => 'nullable|string|max:255',
        ]);

        // Generate receipt number if not provided
        if (empty($validated['receipt_number'])) {
            $validated['receipt_number'] = 'RCP-' . date('Y') . '-' . str_pad(\App\Models\ClientReceipt::count() + 1, 4, '0', STR_PAD_LEFT);
        }

        // Calculate total amount from lines
        $totalAmount = collect($validated['lines'])->sum('amount');

        // Create receipt
        $receipt = \App\Models\ClientReceipt::create([
            'client_id' => $validated['customer_id'],
            'receipt_number' => $validated['receipt_number'],
            'receipt_date' => $validated['receipt_date'],
            'payment_method' => $validated['payment_method'],
            'total_amount' => $totalAmount,
            'status' => 'Pending',
        ]);

        // Create receipt lines and update invoices
        foreach ($validated['lines'] as $lineData) {
            $line = $receipt->lines()->create([
                'description' => $lineData['description'],
                'invoice_id' => $lineData['invoice_id'] ?? null,
                'account_id' => $lineData['account_id'],
                'amount' => $lineData['amount'],
                'reference' => $lineData['reference'] ?? null,
            ]);

            // If linked to invoice, update balance
            if ($lineData['invoice_id']) {
                $invoice = \App\Models\ClientInvoice::find($lineData['invoice_id']);
                if ($invoice) {
                    $currentBalance = $invoice->balance_due;
                    $newBalance = max(0, $currentBalance - $lineData['amount']);
                    // Note: Assuming balance_due is stored; if not, calculate from total - applied payments
                    // For now, we'll assume balance_due exists or add logic to calculate
                }
            }
        }

        return redirect()->back()->with('success', 'Client receipt created successfully.');
    }

    public function updateClientReceipt(Request $request, $id)
    {
        $receipt = \App\Models\ClientReceipt::findOrFail($id);

        $validated = $request->validate([
            'customer_id' => 'required|integer|exists:clients,id',
            'receipt_number' => 'nullable|string|max:255',
            'receipt_date' => 'required|date',
            'payment_method' => 'required|string|in:Cash,Bank Transfer,Cheque,Credit Card',
            'lines' => 'required|array|min:1',
            'lines.*.description' => 'required|string',
            'lines.*.invoice_id' => 'nullable|integer|exists:client_invoices,id',
            'lines.*.account_id' => 'required|integer|exists:main_accounts,id',
            'lines.*.amount' => 'required|numeric|min:0.01',
            'lines.*.reference' => 'nullable|string|max:255',
        ]);

        // Generate receipt number if not provided
        if (empty($validated['receipt_number'])) {
            $validated['receipt_number'] = 'RCP-' . date('Y') . '-' . str_pad(\App\Models\ClientReceipt::count() + 1, 4, '0', STR_PAD_LEFT);
        }

        // Calculate total amount from lines
        $totalAmount = collect($validated['lines'])->sum('amount');

        // Update receipt
        $receipt->update([
            'client_id' => $validated['customer_id'],
            'receipt_number' => $validated['receipt_number'],
            'receipt_date' => $validated['receipt_date'],
            'payment_method' => $validated['payment_method'],
            'total_amount' => $totalAmount,
        ]);

        // Delete old lines
        $receipt->lines()->delete();

        // Create new lines
        foreach ($validated['lines'] as $lineData) {
            $receipt->lines()->create([
                'description' => $lineData['description'],
                'invoice_id' => $lineData['invoice_id'] ?? null,
                'account_id' => $lineData['account_id'],
                'amount' => $lineData['amount'],
                'reference' => $lineData['reference'] ?? null,
            ]);
        }

        return redirect()->back()->with('success', 'Client receipt updated successfully.');
    }

    public function destroyClientReceipt($id)
    {
        $receipt = \App\Models\ClientReceipt::findOrFail($id);
        $receipt->delete();

        return redirect()->back()->with('success', 'Client receipt deleted successfully.');
    }

    public function processClientReceipt($receipt = null)
    {
        // Process client receipt(s)
        if ($receipt) {
            $receipt = \App\Models\ClientReceipt::findOrFail($receipt);
            $receipt->update(['status' => 'Processed']);
            $message = 'Client receipt processed successfully.';
        } else {
            $receipts = \App\Models\ClientReceipt::where('status', '!=', 'Processed')->get();
            foreach ($receipts as $rec) {
                $rec->update(['status' => 'Processed']);
            }
            $message = 'All client receipts processed successfully.';
        }

        return redirect()->back()->with('success', $message);
    }

    public function editClientReceipt($id)
    {
        $receipt = \App\Models\ClientReceipt::with('lines')->findOrFail($id);
        $clients = \App\Models\Client::all();
        $accounts = \App\Models\MainAccount::with('accountType')->get();
        $glAccounts = \App\Models\MainAccount::with('accountType')->get();
        $taxes = \App\Models\Tax::all();
        $currencies = \App\Models\Currency::all();
        $paymentMethods = ['Cash', 'Bank Transfer', 'Cheque', 'Credit Card'];

        return view('modules.processing.client-receipt', compact('clients', 'accounts', 'glAccounts', 'taxes', 'currencies', 'paymentMethods', 'receipt'));
    }

    public function getCustomerInvoices($id)
    {
        $invoices = \App\Models\ClientInvoice::where('client_id', $id)
            ->select('id', 'invoice_number', 'total', 'balance_due')
            ->get();

        return response()->json($invoices);
    }
}
