<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\CourtFiling;
use App\Models\CourtFilingType;
use App\Models\CourtFilingStatus;

class CourtFilingController extends Controller
{
    public function index(Request $request)
    {
        $query = CourtFiling::with(['legalCase', 'filingType', 'filingStatus', 'courtArea', 'judge']);

        // Apply filters
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('filing_title', 'like', "%{$search}%")
                  ->orWhere('filing_code', 'like', "%{$search}%")
                  ->orWhereHas('legalCase', function($caseQuery) use ($search) {
                      $caseQuery->where('case_title', 'like', "%{$search}%")
                               ->orWhere('client_name', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->has('filing_type') && $request->filing_type) {
            $query->whereHas('filingType', function($q) use ($request) {
                $q->where('filing_type_name', $request->filing_type);
            });
        }

        if ($request->has('filing_status') && $request->filing_status) {
            $query->whereHas('filingStatus', function($q) use ($request) {
                $q->where('status_code', $request->filing_status);
            });
        }

        if ($request->has('start_date') && $request->start_date) {
            $query->whereDate('filing_date', '>=', $request->start_date);
        }

        if ($request->has('end_date') && $request->end_date) {
            $query->whereDate('filing_date', '<=', $request->end_date);
        }

        $courtFilings = $query->paginate(10);

        // Get statistics from the full query (not paginated)
        $statsQuery = CourtFiling::with(['legalCase', 'filingStatus']);
        $totalFilings = $statsQuery->count();
        $draftCount = (clone $statsQuery)->whereHas('filingStatus', function($q) {
            $q->where('status_code', 'draft');
        })->count();
        $readyForFilingCount = (clone $statsQuery)->whereHas('filingStatus', function($q) {
            $q->where('status_code', 'ready_for_filing');
        })->count();
        $filedCount = (clone $statsQuery)->whereHas('filingStatus', function($q) {
            $q->where('status_code', 'filed');
        })->count();
        $rejectedCount = (clone $statsQuery)->whereHas('filingStatus', function($q) {
            $q->where('status_code', 'rejected');
        })->count();
        $awaitingResponseCount = (clone $statsQuery)->whereHas('filingStatus', function($q) {
            $q->where('status_code', 'awaiting_response');
        })->count();

        return view('modules.processing.court-filings-list', compact(
            'courtFilings',
            'totalFilings',
            'draftCount',
            'readyForFilingCount',
            'filedCount',
            'rejectedCount',
            'awaitingResponseCount'
        ));
    }

    public function create()
    {
        $cases = \App\Models\LegalCase::all();
        $filingTypes = CourtFilingType::all();
        $filingStatuses = CourtFilingStatus::all();
        $courtAreas = \App\Models\CourtArea::all();
        $judges = \App\Models\Judge::all();
        $lawyers = \App\Models\Lawyer::all();

        return view('modules.processing.court-filings-create', compact(
            'cases', 'filingTypes', 'filingStatuses', 'courtAreas', 'judges', 'lawyers'
        ));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'case_id' => 'required|exists:legal_cases,id',
            'filing_type_id' => 'required|exists:court_filing_types,filing_type_id',
            'filing_status_id' => 'required|exists:court_filing_statuses,filing_status_id',
            'filing_code' => 'required|string|max:50|unique:court_filings,filing_code',
            'filing_title' => 'required|string|max:191',
            'filing_description' => 'nullable|string',
            'court_area_id' => 'nullable|string|max:20',
            'judge_id' => 'nullable|exists:judges,id',
            'filing_date' => 'required|date',
            'due_date' => 'nullable|date',
            'court_acknowledgment' => 'nullable|date',
            'document_path' => 'nullable|string|max:191',
            'version' => 'nullable|string|max:50',
            'next_action' => 'nullable|string|max:191',
            'internal_notes' => 'nullable|string',
            'external_notes' => 'nullable|string',
        ]);

        $validated['created_by'] = auth()->user()->id ?? null;
        $validated['updated_by'] = auth()->user()->id ?? null;

        CourtFiling::create($validated);

        return redirect()->route('court-filings.index')->with('success', 'Court filing created successfully.');
    }

    public function show($filing_id)
    {
        $courtFiling = CourtFiling::with(['legalCase', 'filingType', 'filingStatus', 'courtArea', 'judge'])->findOrFail($filing_id);

        return view('modules.processing.court-filings-show', compact('courtFiling'));
    }

    public function edit($filing_id)
    {
        $courtFiling = CourtFiling::findOrFail($filing_id);
        $cases = \App\Models\LegalCase::all();
        $filingTypes = CourtFilingType::all();
        $filingStatuses = CourtFilingStatus::all();
        $courtAreas = \App\Models\CourtArea::all();
        $judges = \App\Models\Judge::all();

        return view('modules.processing.court-filings-edit', compact(
            'courtFiling', 'cases', 'filingTypes', 'filingStatuses', 'courtAreas', 'judges'
        ));
    }

    public function update(Request $request, $filing_id)
    {
        $courtFiling = CourtFiling::findOrFail($filing_id);

        $validated = $request->validate([
            'case_id' => 'required|exists:legal_cases,id',
            'filing_type_id' => 'required|exists:court_filing_types,filing_type_id',
            'filing_status_id' => 'required|exists:court_filing_statuses,filing_status_id',
            'filing_code' => 'required|string|max:50|unique:court_filings,filing_code,' . $courtFiling->filing_id . ',filing_id',
            'filing_title' => 'required|string|max:191',
            'filing_description' => 'nullable|string',
            'court_area_id' => 'nullable|string|max:20',
            'judge_id' => 'nullable|exists:judges,id',
            'filing_date' => 'required|date',
            'due_date' => 'nullable|date',
            'court_acknowledgment' => 'nullable|date',
            'document_path' => 'nullable|string|max:191',
            'version' => 'nullable|string|max:50',
            'next_action' => 'nullable|string|max:191',
            'internal_notes' => 'nullable|string',
            'external_notes' => 'nullable|string',
        ]);

        $validated['updated_by'] = auth()->user()->id ?? null;

        $courtFiling->update($validated);

        return redirect()->route('court-filings.show', $courtFiling->filing_id)->with('success', 'Court filing updated successfully.');
    }

    public function destroy($filing_id)
    {
        $courtFiling = CourtFiling::findOrFail($filing_id);
        $courtFiling->delete();

        return redirect()->route('court-filings.index')->with('success', 'Court filing deleted successfully.');
    }
}
