<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class CurrencyController extends Controller
{
    public function currencies()
    {
        $currencies = \App\Models\Currency::all();
        $homeCurrency = \App\Models\HomeCurrency::with('currency')->first();

        return view('modules.processing.currencies', compact('currencies', 'homeCurrency'));
    }

    public function addCurrency()
    {
        return view('modules.processing.add-currency');
    }

    public function storeCurrency(Request $request)
    {
        $validated = $request->validate([
            'currency_code' => 'required|string|max:3|unique:currencies,currency_code',
            'currency_name' => 'required|string|max:100',
            'symbol' => 'nullable|string|max:10',
            'exchange_rate' => 'required|numeric|min:0',
            'is_active' => 'boolean',
        ]);

        \App\Models\Currency::create($validated);

        return redirect()->route('modules.processing.currencies')->with('success', 'Currency added successfully.');
    }

    public function editCurrency($id)
    {
        $currency = \App\Models\Currency::findOrFail($id);
        return view('modules.processing.edit-currency', compact('currency'));
    }

    public function updateCurrency(Request $request, $id)
    {
        $currency = \App\Models\Currency::findOrFail($id);

        $validated = $request->validate([
            'currency_code' => 'required|string|max:3|unique:currencies,currency_code,' . $id,
            'currency_name' => 'required|string|max:100',
            'symbol' => 'nullable|string|max:10',
            'exchange_rate' => 'required|numeric|min:0',
            'is_active' => 'boolean',
        ]);

        $currency->update($validated);

        return redirect()->route('modules.processing.currencies')->with('success', 'Currency updated successfully.');
    }

    public function destroyCurrency($id)
    {
        $currency = \App\Models\Currency::findOrFail($id);
        $currency->delete();

        return redirect()->route('modules.processing.currencies')->with('success', 'Currency deleted successfully.');
    }

    public function setHomeCurrency(Request $request)
    {
        $validated = $request->validate([
            'currency_id' => 'required|integer|exists:currencies,currency_id',
        ]);

        // Remove existing home currency
        \App\Models\HomeCurrency::query()->delete();

        // Set new home currency
        \App\Models\HomeCurrency::create([
            'currency_id' => $validated['currency_id'],
        ]);

        return redirect()->back()->with('success', 'Home currency updated successfully.');
    }

    public function updateExchangeRates(Request $request)
    {
        $validated = $request->validate([
            'rates' => 'required|array',
            'rates.*.id' => 'required|integer|exists:currencies,currency_id',
            'rates.*.exchange_rate' => 'required|numeric|min:0',
        ]);

        foreach ($validated['rates'] as $rateData) {
            $currency = \App\Models\Currency::where('currency_id', $rateData['id'])->first();
            if ($currency) {
                $currency->update(['exchange_rate' => $rateData['exchange_rate']]);
            }
        }

        return redirect()->back()->with('success', 'Exchange rates updated successfully.');
    }

    public function updateHomeCurrency(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required|integer',
            'currency' => 'required|integer|exists:currencies,currency_id',
        ]);

        $homeCurrency = \App\Models\HomeCurrency::where('home_currency_id', $validated['id'])->firstOrFail();
        $homeCurrency->update([
            'currency_id' => $validated['currency'],
        ]);

        return redirect()->back()->with('home_success', 'Home Currency updated successfully')->with('active_tab', 'home-currency');
    }
}
