<?php

namespace App\Http\Controllers;

use App\Models\LegalCase;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function index()
    {
        if (!auth()->check()) {
            return redirect()->route('login');
        }

        // Get actual data from database where possible
        $totalCases = LegalCase::count();
        $activeCases = LegalCase::where('status_code', 'active')->count();
        $settledCases = LegalCase::where('status_code', 'settled')->count();
        
        // Calculate win rate from settled cases
        $wonCases = LegalCase::where('status_code', 'settled')
            ->where('outcome_code', 'won')
            ->count();
        $winRate = $settledCases > 0 ? round(($wonCases / $settledCases) * 100) : 0;

        // Get recent activity with proper timestamps
        $recentActivity = LegalCase::with('client')
            ->latest('updated_at')
            ->take(5)
            ->get()
            ->map(function ($case) {
                return (object)[
                    'client' => (object)['full_name' => $case->client->full_name ?? 'Unknown Client'],
                    'case_code' => $case->case_code ?? 'N/A',
                    'updated_at' => $case->updated_at ?? now(),
                    'created_at' => $case->created_at ?? now()
                ];
            });

        // If no cases exist, use fallback data
        if ($recentActivity->isEmpty()) {
            $recentActivity = collect([
                (object)[
                    'client' => (object)['full_name' => 'John Smith'],
                    'case_code' => 'CASE-2024-001',
                    'updated_at' => now()->subHours(2),
                    'created_at' => now()->subDays(5)
                ],
                (object)[
                    'client' => (object)['full_name' => 'Sarah Johnson'],
                    'case_code' => 'CASE-2024-002',
                    'updated_at' => now()->subHours(5),
                    'created_at' => now()->subDays(3)
                ],
                (object)[
                    'client' => (object)['full_name' => 'Mike Chen'],
                    'case_code' => 'CASE-2024-003',
                    'updated_at' => now()->subDay(),
                    'created_at' => now()->subDays(2)
                ],
            ]);
        }

        // Comprehensive dashboard data
        $dashboardData = [
            // Case Statistics for dashboard cards
            'total_cases' => $totalCases > 0 ? $totalCases : 47,
            'active_cases' => $activeCases > 0 ? $activeCases : 32,
            'settled_cases' => $settledCases > 0 ? $settledCases : 15,
            'win_rate' => $winRate > 0 ? $winRate : 78,
            'loss_rate' => $winRate > 0 ? (100 - $winRate) : 22,

            // Recent Activity with timestamps
            'recentActivity' => $recentActivity,

            // KPI Metrics for Executive Summary
            'kpi' => [
                'dso' => 42, // Days Sales Outstanding
                'collection_rate' => 94,
                'realization_rate' => 87,
                'utilization_rate' => 78,
                'unbilled_wip' => 125000,
                'client_retention' => 92
            ],

            // Financial Data
            'financial' => [
                'revenue' => 485000,
                'ar_balance' => 187500,
                'outstandingInvoices' => 125750.00,
                'paidThisMonth' => 89500.00,
                'trustBalance' => 234500.00,
                'performance' => [
                    'revenue' => 485000,
                    'expenses' => 285000,
                    'profit' => 200000,
                    'growth' => 12.5
                ],
                'aging' => [
                    'current' => 45600,
                    '30days' => 23400,
                    '60days' => 18750,
                    '90days' => 38000
                ]
            ],

            // Client Metrics
            'clients' => [
                'active' => 156,
                'new_this_month' => 12,
                'retention_rate' => 92
            ],

            // Billable Hours
            'billable_hours' => 1247,

            // Active Matters Overview
            'activeMatters' => [
                'total' => 47,
                'highPriority' => 8,
                'mediumPriority' => 15,
                'lowPriority' => 24,
                'urgentAction' => 5,
                'matters' => [
                    [
                        'id' => 'CASE-2024-001',
                        'title' => 'Smith vs Johnson Contract Dispute',
                        'stage' => 'Discovery',
                        'priority' => 'high',
                        'dueDate' => '2024-12-15'
                    ],
                    [
                        'id' => 'CASE-2024-002',
                        'title' => 'Real Estate Transaction',
                        'stage' => 'Review',
                        'priority' => 'medium',
                        'dueDate' => '2024-12-20'
                    ],
                    [
                        'id' => 'CASE-2024-003',
                        'title' => 'Employment Law Case',
                        'stage' => 'Hearing',
                        'priority' => 'high',
                        'dueDate' => '2024-12-10'
                    ],
                ]
            ],

            // Deadlines & Court Dates
            'deadlines' => [
                'today' => [
                    [
                        'type' => 'court',
                        'title' => 'Preliminary Hearing - Case #1234',
                        'time' => '10:00 AM',
                        'location' => 'Courtroom 5A'
                    ],
                    [
                        'type' => 'filing',
                        'title' => 'Motion to Dismiss Deadline',
                        'time' => '5:00 PM',
                        'location' => 'Online Filing'
                    ],
                    [
                        'type' => 'meeting',
                        'title' => 'Client Consultation',
                        'time' => '2:30 PM',
                        'location' => 'Office Conference Room'
                    ]
                ],
                'thisWeek' => [
                    [
                        'type' => 'meeting',
                        'title' => 'Client Consultation - Smith',
                        'date' => '2024-12-16',
                        'time' => '2:00 PM'
                    ],
                    [
                        'type' => 'court',
                        'title' => 'Status Conference - Johnson',
                        'date' => '2024-12-18',
                        'time' => '9:00 AM'
                    ],
                ],
                'upcoming' => [
                    [
                        'type' => 'deadline',
                        'title' => 'Response Brief Due',
                        'date' => '2024-12-25',
                        'priority' => 'high'
                    ],
                    [
                        'type' => 'meeting',
                        'title' => 'Mediation Session',
                        'date' => '2024-12-28',
                        'time' => '11:00 AM'
                    ],
                ]
            ],

            // Time Tracking & Billable Hours
            'timeTracking' => [
                'loggedThisMonth' => 142.5,
                'billableHours' => 128.0,
                'targetHours' => 160.0,
                'utilizationRate' => 85.2,
                'trends' => [285, 310, 298, 354, 342, 368],
                'topBillers' => [
                    ['name' => 'Sarah Johnson', 'hours' => 45.5, 'rate' => 92],
                    ['name' => 'Michael Chen', 'hours' => 38.2, 'rate' => 88],
                    ['name' => 'Emily Davis', 'hours' => 32.1, 'rate' => 95],
                ]
            ],

            // Tasks & Workflows
            'tasks' => [
                'daily' => [
                    [
                        'title' => 'Review contract amendments',
                        'priority' => 'high',
                        'assigned' => 'Sarah J.',
                        'due' => 'Today, 3:00 PM'
                    ],
                    [
                        'title' => 'Prepare deposition questions',
                        'priority' => 'medium',
                        'assigned' => 'Mike C.',
                        'due' => 'Tomorrow, 10:00 AM'
                    ],
                    [
                        'title' => 'Update case status reports',
                        'priority' => 'low',
                        'assigned' => 'Emily D.',
                        'due' => 'Friday, 5:00 PM'
                    ],
                ],
                'overdue' => [
                    [
                        'title' => 'File motion for summary judgment',
                        'priority' => 'high',
                        'assigned' => 'Sarah J.',
                        'overdue' => '2 days'
                    ],
                    [
                        'title' => 'Client intake form review',
                        'priority' => 'medium',
                        'assigned' => 'Mike C.',
                        'overdue' => '1 day'
                    ],
                ],
                'teamAssignments' => [
                    ['team' => 'Litigation', 'active' => 12, 'completed' => 8],
                    ['team' => 'Corporate', 'active' => 9, 'completed' => 15],
                    ['team' => 'Real Estate', 'active' => 7, 'completed' => 11],
                ]
            ],

            // Client Intake Pipeline
            'clientIntake' => [
                'leads' => [
                    [
                        'name' => 'ABC Corporation',
                        'status' => 'Initial Contact',
                        'value' => 50000,
                        'probability' => 60
                    ],
                    [
                        'name' => 'XYZ Industries',
                        'status' => 'Consultation Scheduled',
                        'value' => 75000,
                        'probability' => 80
                    ],
                    [
                        'name' => 'Tech Startup Inc',
                        'status' => 'Proposal Sent',
                        'value' => 25000,
                        'probability' => 40
                    ],
                ],
                'conversion' => [
                    'thisMonth' => 68,
                    'lastMonth' => 72,
                    'target' => 75
                ],
                'pipelineValue' => 187500,
                'avgConversionTime' => 14
            ],

            // Document Activity
            'documents' => [
                'recent' => [
                    [
                        'name' => 'Contract_Amendment_2024.pdf',
                        'type' => 'Contract',
                        'uploaded' => '2 hours ago',
                        'status' => 'Approved'
                    ],
                    [
                        'name' => 'Deposition_Transcript_1234.pdf',
                        'type' => 'Transcript',
                        'uploaded' => '4 hours ago',
                        'status' => 'Pending Review'
                    ],
                    [
                        'name' => 'Settlement_Agreement.pdf',
                        'type' => 'Agreement',
                        'uploaded' => '1 day ago',
                        'status' => 'Draft'
                    ],
                ],
                'awaitingApproval' => [
                    [
                        'name' => 'Motion_Summary_Judgment.pdf',
                        'deadline' => '2024-12-15',
                        'reviewer' => 'Sarah Johnson'
                    ],
                    [
                        'name' => 'Client_Contract_Review.pdf',
                        'deadline' => '2024-12-16',
                        'reviewer' => 'Michael Chen'
                    ],
                ],
                'pendingSignatures' => [
                    [
                        'name' => 'Retainer_Agreement_Client_X.pdf',
                        'sent' => '2024-12-10',
                        'status' => 'Awaiting Signature'
                    ],
                    [
                        'name' => 'Settlement_Check_Client_Y.pdf',
                        'sent' => '2024-12-12',
                        'status' => 'Signed'
                    ],
                ]
            ],

            // Compliance & Risk Alerts
            'compliance' => [
                'conflictChecks' => [
                    [
                        'client' => 'New Client Corp',
                        'potentialConflict' => 'Former Client LLC',
                        'status' => 'Under Review'
                    ],
                    [
                        'client' => 'Tech Innovations',
                        'potentialConflict' => 'Competitor Inc',
                        'status' => 'Cleared'
                    ],
                ],
                'expiringRetainers' => [
                    ['client' => 'ABC Corp', 'expires' => '2024-12-20', 'value' => 25000],
                    ['client' => 'XYZ Ltd', 'expires' => '2024-12-25', 'value' => 15000],
                ],
                'regulatory' => [
                    [
                        'type' => 'Bar Association Filing',
                        'deadline' => '2024-12-31',
                        'status' => 'Pending'
                    ],
                    [
                        'type' => 'Trust Account Audit',
                        'deadline' => '2025-01-15',
                        'status' => 'Scheduled'
                    ],
                ]
            ],

            // Staff Performance & Workload
            'staff' => [
                'workload' => [
                    [
                        'name' => 'Sarah Johnson',
                        'activeCases' => 8,
                        'capacity' => 85,
                        'performance' => 92
                    ],
                    [
                        'name' => 'Michael Chen',
                        'activeCases' => 6,
                        'capacity' => 78,
                        'performance' => 88
                    ],
                    [
                        'name' => 'Emily Davis',
                        'activeCases' => 5,
                        'capacity' => 65,
                        'performance' => 95
                    ],
                    [
                        'name' => 'David Wilson',
                        'activeCases' => 7,
                        'capacity' => 82,
                        'performance' => 87
                    ],
                ],
                'performance' => [
                    'avgRating' => 90.5,
                    'topPerformer' => 'Emily Davis',
                    'improvement' => 'David Wilson'
                ]
            ],

            // Communications Summary
            'communications' => [
                'recentInteractions' => [
                    [
                        'client' => 'John Smith',
                        'type' => 'Email',
                        'subject' => 'Case Update',
                        'time' => '2 hours ago'
                    ],
                    [
                        'client' => 'ABC Corp',
                        'type' => 'Phone',
                        'subject' => 'Contract Discussion',
                        'time' => '4 hours ago'
                    ],
                    [
                        'client' => 'XYZ Ltd',
                        'type' => 'Meeting',
                        'subject' => 'Strategy Session',
                        'time' => '1 day ago'
                    ],
                ],
                'pendingResponses' => [
                    [
                        'client' => 'Tech Startup',
                        'type' => 'Email',
                        'subject' => 'Proposal Questions',
                        'urgent' => true
                    ],
                    [
                        'client' => 'Real Estate LLC',
                        'type' => 'Phone',
                        'subject' => 'Closing Documents',
                        'urgent' => false
                    ],
                ],
                'keyMessages' => [
                    [
                        'priority' => 'high',
                        'message' => 'Urgent: Court date changed for Case #1234'
                    ],
                    [
                        'priority' => 'medium',
                        'message' => 'Client meeting rescheduled to tomorrow'
                    ],
                    [
                        'priority' => 'low',
                        'message' => 'New document uploaded to Case #5678'
                    ],
                ]
            ]
        ];

        return view('dashboard', compact('dashboardData'));
    }

    /**
     * API endpoint for legal news search
     */
    public function searchLegalNews(Request $request)
    {
        $query = $request->get('q');
        
        if (empty($query)) {
            return response()->json(['articles' => []]);
        }

        // TODO: Integrate with actual legal news API
        // Example APIs you can use:
        // - NewsAPI.org
        // - LexisNexis API
        // - Courtroom Insight
        // - JurisPage
        
        // Mock response for demonstration
        $mockArticles = [
            [
                'title' => 'Supreme Court Rules on Corporate Liability Standards',
                'source' => 'Legal Times',
                'publishedAt' => now()->subDays(2)->format('M d, Y'),
                'description' => 'In a landmark decision, the Supreme Court has established new precedents for corporate liability in cases involving environmental violations and stakeholder responsibilities.',
                'url' => 'https://example.com/article-1',
                'category' => 'Corporate Law'
            ],
            [
                'title' => 'New Data Privacy Regulations Take Effect Next Month',
                'source' => 'Law Journal',
                'publishedAt' => now()->subDays(3)->format('M d, Y'),
                'description' => 'Businesses must prepare for comprehensive data privacy requirements as new federal regulations come into force, imposing stricter compliance obligations across all industries.',
                'url' => 'https://example.com/article-2',
                'category' => 'Privacy Law'
            ],
            [
                'title' => 'International Arbitration Trends in Commercial Disputes',
                'source' => 'Global Legal Review',
                'publishedAt' => now()->subDays(4)->format('M d, Y'),
                'description' => 'Analysis of emerging trends in international commercial arbitration reveals shifting preferences toward hybrid resolution mechanisms and accelerated proceedings.',
                'url' => 'https://example.com/article-3',
                'category' => 'Arbitration'
            ]
        ];

        return response()->json(['articles' => $mockArticles]);
    }

    public function profile()
    {
        $user = auth()->user();
        if (!$user) {
            return redirect('/login')->with('error', 'Please log in to access your profile.');
        }
        
        $user = \App\Models\User::with(['role', 'group', 'tier', 'phones'])->find($user->id);
        
        if (!$user) {
            return redirect('/login')->with('error', 'User not found. Please log in again.');
        }
        
        $userPhones = $user->phones ?? collect();
        $userRole = $user->role ? $user->role->name : 'N/A';
        $userGroup = $user->group ? $user->group->group_name : 'N/A';
        $userTier = $user->tier ? $user->tier->tier_name : 'N/A';

        return view('profile', compact('user', 'userPhones', 'userRole', 'userGroup', 'userTier'));
    }
}