<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class JournalController extends Controller
{
    public function journal()
    {
        return view('modules.processing.journal');
    }

    public function clientJournal()
    {
        $clientAccount = \App\Models\ClientAccount::first() ?? (object)['code' => '1100'];
        $clientAccounts = \App\Models\ClientAccount::with('client')->get();
        $currencies = \App\Models\Currency::all();
        $tempEntries = \App\Models\ClientJournal::with('clientAccount.client')->get();

        return view('modules.processing.journals.client', compact('clientAccount', 'clientAccounts', 'currencies', 'tempEntries'));
    }

    public function storeClientJournal(Request $request)
    {
        $validated = $request->validate([
            'client_account_id' => 'required|integer|exists:client_accounts,id',
            'date' => 'required|date',
            'narration' => 'required|string',
            'tr_code' => 'required|in:DR,CR',
            'currency_id' => 'required|integer|exists:currencies,id',
            'amount' => 'required|numeric|min:0',
            'reference' => 'nullable|string|max:255',
        ]);

        \App\Models\ClientJournal::create($validated);

        return redirect()->back()->with('success', 'Client journal entry added successfully.');
    }

    public function updateClientJournal(Request $request, $id)
    {
        $entry = \App\Models\ClientJournal::findOrFail($id);

        $validated = $request->validate([
            'client_account_id' => 'required|integer|exists:client_accounts,id',
            'date' => 'required|date',
            'narration' => 'required|string',
            'tr_code' => 'required|in:DR,CR',
            'currency_id' => 'required|integer|exists:currencies,id',
            'amount' => 'required|numeric|min:0',
            'reference' => 'nullable|string|max:255',
        ]);

        $entry->update($validated);

        return redirect()->back()->with('success', 'Client journal entry updated successfully.');
    }

    public function processClientJournal(Request $request)
    {
        // Process all client journal entries
        $entries = \App\Models\ClientJournal::where('status', '!=', 'Processed')->get();
        foreach ($entries as $entry) {
            $entry->update(['status' => 'Processed']);
        }

        return redirect()->back()->with('success', 'All client journal entries processed successfully.');
    }

    public function processSingleClientJournal($id)
    {
        $entry = \App\Models\ClientJournal::findOrFail($id);
        $entry->update(['status' => 'Processed']);

        return redirect()->back()->with('success', 'Client journal entry processed successfully.');
    }

    public function deleteClientJournal($id)
    {
        $entry = \App\Models\ClientJournal::findOrFail($id);
        $entry->delete();

        return redirect()->back()->with('success', 'Client journal entry deleted successfully.');
    }

    public function generalJournal()
    {
        $accounts = \App\Models\MainAccount::with('accountType')->get();
        $currencies = \App\Models\Currency::all();
        $tempEntries = \App\Models\GeneralJournal::with('account', 'currency')->get();

        return view('modules.processing.journals.general', compact('accounts', 'currencies', 'tempEntries'));
    }

    public function supplierJournal()
    {
        $supplierAccount = \App\Models\Supplier::first() ?? (object)['code' => '2100'];
        $suppliers = \App\Models\Supplier::all();
        $currencies = \App\Models\Currency::all();
        $tempEntries = collect(); // Assuming no specific model, using empty collection for now

        return view('modules.processing.journals.supplier', compact('supplierAccount', 'suppliers', 'currencies', 'tempEntries'));
    }

    public function storeGeneralJournal(Request $request)
    {
        // Basic implementation - you may need to adjust based on your model
        $validated = $request->validate([
            'account_id' => 'required|integer',
            'date' => 'required|date',
            'narration' => 'required|string',
            'tr_code' => 'required|in:DR,CR',
            'currency_id' => 'required|integer',
            'amount' => 'required|numeric|min:0',
        ]);

        // Assuming you have a GeneralJournal model
        \App\Models\GeneralJournal::create($validated);

        return redirect()->back()->with('success', 'General journal entry added successfully.');
    }

    public function processGeneralJournal(Request $request)
    {
        // Process all general journal entries
        $entries = \App\Models\GeneralJournal::where('status', '!=', 'Processed')->get();
        foreach ($entries as $entry) {
            $entry->update(['status' => 'Processed']);
        }

        return redirect()->back()->with('success', 'All general journal entries processed successfully.');
    }

    public function processSingleGeneralJournal($id)
    {
        $entry = \App\Models\GeneralJournal::findOrFail($id);
        $entry->update(['status' => 'Processed']);

        return redirect()->back()->with('success', 'General journal entry processed successfully.');
    }

    public function updateGeneralJournal(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required|integer',
            'account_id' => 'required|integer',
            'date' => 'required|date',
            'narration' => 'required|string',
            'tr_code' => 'required|in:DR,CR',
            'currency_id' => 'required|integer',
            'amount' => 'required|numeric|min:0',
        ]);

        $entry = \App\Models\GeneralJournal::findOrFail($validated['id']);
        $entry->update($validated);

        return redirect()->back()->with('success', 'General journal entry updated successfully.');
    }

    public function deleteGeneralJournal($id)
    {
        $entry = \App\Models\GeneralJournal::findOrFail($id);
        $entry->delete();

        return redirect()->back()->with('success', 'General journal entry deleted successfully.');
    }
}
