<?php

namespace App\Http\Controllers;

use App\Models\Judge;
use Illuminate\Http\Request;

class JudgeController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $judges = Judge::all();
        return view('modules.edit.judges', compact('judges'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('modules.edit.judge');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'judge_code' => 'required|string|max:255|unique:judges',
            'full_name' => 'required|string|max:255',
            'court_area_code' => 'nullable|exists:court_areas,area_code',
            'court_type_name' => 'nullable|exists:court_types,type_name',
            'contact' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'notes' => 'nullable|string',
        ]);

        $validated['is_active'] = $request->has('is_active');

        Judge::create($validated);

        return redirect()->route('modules.edit.judges.index')->with('success', 'Judge created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $judge_code)
    {
        $judge = Judge::where('judge_code', $judge_code)->firstOrFail();
        return view('modules.edit.judge-show', compact('judge'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $judge_code)
    {
        $judge = Judge::where('judge_code', $judge_code)->firstOrFail();
        return view('modules.edit.edit-judge', compact('judge'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $judge_code)
    {
        $judge = Judge::where('judge_code', $judge_code)->firstOrFail();

        $validated = $request->validate([
            'judge_code' => 'required|string|max:255|unique:judges,judge_code,' . $judge->judge_code . ',judge_code',
            'full_name' => 'required|string|max:255',
            'court_area_code' => 'nullable|exists:court_areas,area_code',
            'court_type_name' => 'nullable|exists:court_types,type_name',
            'contact' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'notes' => 'nullable|string',
        ]);

        $validated['is_active'] = $request->has('is_active');

        $judge->update($validated);

        return redirect()->route('modules.edit.judges.index')->with('success', 'Judge updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $judge_code)
    {
        $judge = Judge::where('judge_code', $judge_code)->firstOrFail();
        $judge->delete();

        return redirect()->route('modules.edit.judges.index')->with('success', 'Judge deleted successfully.');
    }
}
