<?php

namespace App\Http\Controllers;

use App\Models\MainAccount;
use App\Models\AccountType;
use App\Models\SubAccount;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;

class LedgerController extends Controller
{
    public function create()
    {
        $accounts = AccountType::all();
        $records = MainAccount::with('accountType')->whereNull('parent_id')->get();
        return view('modules.edit.general-ledger', compact('accounts', 'records'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'code' => 'required|string|max:255',
            'account_name' => 'required|string|max:255',
            'account_type' => 'required|string',
            'financial_category' => 'required|string',
        ]);

        MainAccount::create([
            'code' => $request->code,
            'name' => $request->account_name,
            'account_type_id' => $request->account_type,
            'financial_category' => $request->financial_category,
        ]);

        Session::flash('success', 'Account created successfully.');
        return redirect()->route('modules.edit.general-ledger');
    }

    public function edit(Request $request)
    {
        $request->validate([
            'id' => 'required|integer',
            'code' => 'required|string|max:255',
            'account_name' => 'required|string|max:255',
            'account_type' => 'required|string',
            'financial_category' => 'required|string',
        ]);

        $account = MainAccount::findOrFail($request->id);
        $account->update([
            'code' => $request->code,
            'name' => $request->account_name,
            'account_type_id' => $request->account_type,
            'financial_category' => $request->financial_category,
        ]);

        Session::flash('success', 'Account updated successfully.');
        return redirect()->route('modules.edit.general-ledger');
    }

    public function destroy($id)
    {
        $account = MainAccount::findOrFail($id);
        $account->delete();

        Session::flash('success', 'Account deleted successfully.');
        return redirect()->route('modules.edit.general-ledger');
    }

    public function addSubAccount($id)
    {
        $parentAccount = MainAccount::findOrFail($id);
        $accounts = AccountType::all();
        $subAccounts = MainAccount::where('parent_id', $id)->with('accountType', 'parent')->get();
        $parentCodeParts = explode('.', $parentAccount->code);
        $defaultMainNumber = $parentCodeParts[1] ?? '';
        return view('modules.edit.sub-accounts', compact('parentAccount', 'accounts', 'subAccounts', 'parentCodeParts', 'defaultMainNumber'));
    }

    public function storeSubAccount(Request $request)
    {
        $request->validate([
            'main_account_code' => 'required|string|max:255',
            'sub_account_code' => 'required|string|max:255',
            'sub_account_number' => 'nullable|string|max:255',
            'account_name' => 'required|string|max:255',
            'parent_id' => 'required|integer',
        ]);

        $parentAccount = MainAccount::findOrFail($request->parent_id);

        // Save to subaccounts table
        SubAccount::create([
            'main_account_code' => $request->main_account_code,
            'main_account_name' => $parentAccount->name,
            'sub_account_code' => $request->sub_account_code,
            'sub_account_name' => $request->account_name,
            'company_code' => null, // Can be set if needed
        ]);

        // Also save to general_ledger_accounts table for consistency
        // Generate unique code by combining main account code and sub account code
        $code = $request->main_account_code . '.' . $request->sub_account_code;
        if ($request->sub_account_number) {
            $code .= '.' . $request->sub_account_number;
        }

        // Check if code already exists and generate a unique one
        $originalCode = $code;
        $counter = 1;
        while (MainAccount::where('code', $code)->exists()) {
            if ($request->sub_account_number) {
                $code = $request->main_account_code . '.' . $request->sub_account_code . '.' . ((int)$request->sub_account_number + $counter);
            } else {
                $code = $originalCode . '.' . $counter;
            }
            $counter++;
        }

        MainAccount::create([
            'code' => $code,
            'name' => $request->account_name,
            'account_type_id' => $parentAccount->account_type_id,
            'financial_category' => $parentAccount->financial_category,
            'parent_id' => $request->parent_id,
        ]);

        Session::flash('success', 'Sub account created successfully.');
        return redirect()->route('subacc.add', $request->parent_id);
    }

    public function editSubAccount(Request $request)
    {
        $request->validate([
            'id' => 'required|integer',
            'main_account_code' => 'required|string|max:255',
            'sub_account_code' => 'required|string|max:255',
            'sub_account_number' => 'nullable|string|max:255',
            'account_name' => 'required|string|max:255',
            'parent_id' => 'required|integer',
        ]);

        $parentAccount = MainAccount::findOrFail($request->parent_id);
        $account = MainAccount::findOrFail($request->id);

        // Update subaccounts table
        $subAccount = SubAccount::where('main_account_code', $parentAccount->code)
            ->where('sub_account_code', $request->sub_account_code)
            ->first();
        if ($subAccount) {
            $subAccount->update([
                'sub_account_name' => $request->account_name,
            ]);
        }

        // Generate code
        $code = $request->main_account_code . '.' . $request->sub_account_code;
        if ($request->sub_account_number) {
            $code .= '.' . $request->sub_account_number;
        }

        // Check if code already exists (excluding current account)
        $originalCode = $code;
        $counter = 1;
        while (MainAccount::where('code', $code)->where('id', '!=', $request->id)->exists()) {
            if ($request->sub_account_number) {
                $code = $request->main_account_code . '.' . $request->sub_account_code . '.' . ((int)$request->sub_account_number + $counter);
            } else {
                $code = $originalCode . '.' . $counter;
            }
            $counter++;
        }

        $account->update([
            'code' => $code,
            'name' => $request->account_name,
            'account_type_id' => $parentAccount->account_type_id,
            'financial_category' => $parentAccount->financial_category,
        ]);

        Session::flash('success', 'Sub account updated successfully.');
        return redirect()->route('subacc.add', $request->parent_id);
    }

    public function deleteSubAccount($id)
    {
        $account = MainAccount::findOrFail($id);
        $parentId = $account->parent_id;

        // Delete from subaccounts table
        $subAccount = SubAccount::where('main_account_code', $account->code)->first();
        if ($subAccount) {
            $subAccount->delete();
        }

        $account->delete();

        Session::flash('success', 'Sub account deleted successfully.');
        return redirect()->route('subacc.add', $parentId);
    }
}
