<?php

namespace App\Http\Controllers;

use App\Models\AccountType;
use App\Models\MainAccount;
use App\Models\Assignment;
use Illuminate\Http\Request;
use App\Models\Currency;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\CaseParty;
use App\Models\CaseRole;
use App\Models\LegalCase;
use App\Models\AssignmentHistory;

class ModuleController extends Controller
{
    public function edit()
    {
        return view('modules.edit');
    }

    public function processing()
    {
        return view('modules.processing');
    }

    public function taskManagement()
    {
        return view('modules.processing.task-management');
    }

    public function diary()
    {
        return view('modules.processing.diary-list');
    }

    public function assignmentCollaboration()
    {
        return view('modules.processing.assignment-collaboration');
    }

    public function reports()
    {
        return view('modules.reports');
    }

    public function courtFilings()
    {
        $courtFilings = \App\Models\CourtFiling::with(['legalCase', 'filingType', 'filingStatus', 'courtArea', 'judge'])->paginate(10);

        // Get statistics from the full query (not paginated)
        $statsQuery = \App\Models\CourtFiling::with(['legalCase', 'filingStatus']);
        $totalFilings = $statsQuery->count();
        $draftCount = (clone $statsQuery)->whereHas('filingStatus', function($q) {
            $q->where('status_code', 'draft');
        })->count();
        $readyForFilingCount = (clone $statsQuery)->whereHas('filingStatus', function($q) {
            $q->where('status_code', 'ready_for_filing');
        })->count();
        $filedCount = (clone $statsQuery)->whereHas('filingStatus', function($q) {
            $q->where('status_code', 'filed');
        })->count();
        $rejectedCount = (clone $statsQuery)->whereHas('filingStatus', function($q) {
            $q->where('status_code', 'rejected');
        })->count();
        $awaitingResponseCount = (clone $statsQuery)->whereHas('filingStatus', function($q) {
            $q->where('status_code', 'awaiting_response');
        })->count();

        return view('modules.processing.court-filings-list', compact(
            'courtFilings',
            'totalFilings',
            'draftCount',
            'readyForFilingCount',
            'filedCount',
            'rejectedCount',
            'awaitingResponseCount'
        ));
    }

    public function clientInvoice()
    {
        $clients = \App\Models\Client::all();
        $accounts = \App\Models\MainAccount::with('accountType')->get();
        $glAccounts = \App\Models\MainAccount::with('accountType')->get();
        $taxes = \App\Models\Tax::all();
        $currencies = \App\Models\Currency::all();
        $clientTransactions = \App\Models\ClientInvoice::with('client')->get();

        return view('modules.processing.client-invoice', compact('clients', 'accounts', 'glAccounts', 'taxes', 'currencies', 'clientTransactions'));
    }

    public function clientReceipt()
    {
        $clients = \App\Models\Client::all();
        $accounts = \App\Models\MainAccount::with('accountType')->get();
        $glAccounts = \App\Models\MainAccount::with('accountType')->get();
        $taxes = \App\Models\Tax::all();
        $currencies = \App\Models\Currency::all();
        $clientReceipts = \App\Models\ClientReceipt::with(['client', 'lines'])->get();
        $paymentMethods = ['Cash', 'Bank Transfer', 'Cheque', 'Credit Card'];

        return view('modules.processing.client-receipt', compact('clients', 'accounts', 'glAccounts', 'taxes', 'currencies', 'clientReceipts', 'paymentMethods'));
    }

    public function storeClientInvoice(Request $request)
    {
        $validated = $request->validate([
            'client_id' => 'required|integer|exists:clients,id',
            'invoice_number' => 'required|string|max:255',
            'invoice_date' => 'required|date',
            'due_date' => 'nullable|date',
            'amount' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'account' => 'nullable|string|max:255',
            'narration' => 'nullable|string',
            'reference' => 'nullable|string|max:255',
            'gl_account' => 'nullable|string|max:255',
            'tax_id' => 'nullable|integer|exists:taxes,id',
            'discount' => 'nullable|numeric|min:0|max:100',
            'price' => 'required|numeric|min:0',
            'quantity' => 'nullable|integer|min:1',
            'currency_id' => 'nullable|integer|exists:currencies,id',
            'operation_cost' => 'nullable|boolean',
            'tax_amount' => 'nullable|numeric|min:0',
            'case_reference' => 'nullable|string|max:255',
            'responsible_lawyer' => 'nullable|string|max:255',
            'disbursements' => 'nullable|numeric|min:0',
            'total' => 'nullable|numeric|min:0',
        ]);

        $validated['total'] = ($validated['amount'] ?? 0) + ($validated['tax_amount'] ?? 0) + ($validated['disbursements'] ?? 0);

        \App\Models\ClientInvoice::create($validated);

        return redirect()->back()->with('success', 'Client invoice created successfully.');
    }

    public function updateClientInvoice(Request $request, $id)
    {
        $invoice = \App\Models\ClientInvoice::findOrFail($id);

        $validated = $request->validate([
            'client_id' => 'required|integer|exists:clients,id',
            'invoice_number' => 'required|string|max:255',
            'invoice_date' => 'required|date',
            'due_date' => 'nullable|date',
            'amount' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'account' => 'nullable|string|max:255',
            'narration' => 'nullable|string',
            'reference' => 'nullable|string|max:255',
            'gl_account' => 'nullable|string|max:255',
            'tax_id' => 'nullable|integer|exists:taxes,id',
            'discount' => 'nullable|numeric|min:0|max:100',
            'price' => 'required|numeric|min:0',
            'quantity' => 'nullable|integer|min:1',
            'currency_id' => 'nullable|integer|exists:currencies,id',
            'operation_cost' => 'nullable|boolean',
            'tax_amount' => 'nullable|numeric|min:0',
            'case_reference' => 'nullable|string|max:255',
            'responsible_lawyer' => 'nullable|string|max:255',
            'disbursements' => 'nullable|numeric|min:0',
            'total' => 'nullable|numeric|min:0',
        ]);

        $validated['total'] = ($validated['amount'] ?? 0) + ($validated['tax_amount'] ?? 0) + ($validated['disbursements'] ?? 0);

        $invoice->update($validated);

        return redirect()->back()->with('success', 'Client invoice updated successfully.');
    }

    public function destroyClientInvoice($id)
    {
        $invoice = \App\Models\ClientInvoice::findOrFail($id);
        $invoice->delete();

        return redirect()->back()->with('success', 'Client invoice deleted successfully.');
    }

    public function processClientInvoice($invoice = null)
    {
        // Process client invoice(s)
        if ($invoice) {
            $invoice = \App\Models\ClientInvoice::findOrFail($invoice);
            $invoice->update(['status' => 'Processed']);
            $message = 'Client invoice processed successfully.';
        } else {
            $invoices = \App\Models\ClientInvoice::where('status', '!=', 'Processed')->get();
            foreach ($invoices as $inv) {
                $inv->update(['status' => 'Processed']);
            }
            $message = 'All client invoices processed successfully.';
        }

        return redirect()->back()->with('success', $message);
    }

    public function storeClientReceipt(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|integer|exists:clients,id',
            'receipt_number' => 'nullable|string|max:255',
            'receipt_date' => 'required|date',
            'payment_method' => 'required|string|in:Cash,Bank Transfer,Cheque,Credit Card',
            'lines' => 'required|array|min:1',
            'lines.*.description' => 'required|string',
            'lines.*.invoice_id' => 'nullable|integer|exists:client_invoices,id',
            'lines.*.account_id' => 'required|integer|exists:general_ledger_accounts,account_id',
            'lines.*.amount' => 'required|numeric|min:0.01',
            'lines.*.reference' => 'nullable|string|max:255',
        ]);

        // Generate receipt number if not provided
        if (empty($validated['receipt_number'])) {
            $validated['receipt_number'] = 'RCP-' . date('Y') . '-' . str_pad(\App\Models\ClientReceipt::count() + 1, 4, '0', STR_PAD_LEFT);
        }

        // Calculate total amount from lines
        $totalAmount = collect($validated['lines'])->sum('amount');

        // Create receipt
        $receipt = \App\Models\ClientReceipt::create([
            'client_id' => $validated['customer_id'],
            'receipt_number' => $validated['receipt_number'],
            'receipt_date' => $validated['receipt_date'],
            'payment_method' => $validated['payment_method'],
            'total_amount' => $totalAmount,
            'status' => 'Pending',
        ]);

        // Create receipt lines and update invoices
        foreach ($validated['lines'] as $lineData) {
            $line = $receipt->lines()->create([
                'description' => $lineData['description'],
                'invoice_id' => $lineData['invoice_id'] ?? null,
                'account_id' => $lineData['account_id'],
                'amount' => $lineData['amount'],
                'reference' => $lineData['reference'] ?? null,
            ]);

            // If linked to invoice, update balance
            if ($lineData['invoice_id']) {
                $invoice = \App\Models\ClientInvoice::find($lineData['invoice_id']);
                if ($invoice) {
                    $currentBalance = $invoice->balance_due ?? $invoice->total;
                    $newBalance = max(0, $currentBalance - $lineData['amount']);
                    $invoice->update(['balance_due' => $newBalance]);
                }
            }
        }

        return redirect()->back()->with('success', 'Client receipt created successfully.');
    }

    public function updateClientReceipt(Request $request, $id)
    {
        $receipt = \App\Models\ClientReceipt::findOrFail($id);

        $validated = $request->validate([
            'customer_id' => 'required|integer|exists:clients,id',
            'receipt_number' => 'nullable|string|max:255',
            'receipt_date' => 'required|date',
            'payment_method' => 'required|string|in:Cash,Bank Transfer,Cheque,Credit Card',
            'lines' => 'required|array|min:1',
            'lines.*.description' => 'required|string',
            'lines.*.invoice_id' => 'nullable|integer|exists:client_invoices,id',
            'lines.*.account_id' => 'required|integer|exists:general_ledger_accounts,account_id',
            'lines.*.amount' => 'required|numeric|min:0.01',
            'lines.*.reference' => 'nullable|string|max:255',
        ]);

        // Generate receipt number if not provided
        if (empty($validated['receipt_number'])) {
            $validated['receipt_number'] = 'RCP-' . date('Y') . '-' . str_pad(\App\Models\ClientReceipt::count() + 1, 4, '0', STR_PAD_LEFT);
        }

        // Calculate total amount from lines
        $totalAmount = collect($validated['lines'])->sum('amount');

        // Update receipt
        $receipt->update([
            'client_id' => $validated['customer_id'],
            'receipt_number' => $validated['receipt_number'],
            'receipt_date' => $validated['receipt_date'],
            'payment_method' => $validated['payment_method'],
            'total_amount' => $totalAmount,
        ]);

        // Delete old lines
        $receipt->lines()->delete();

        // Create new lines
        foreach ($validated['lines'] as $lineData) {
            $receipt->lines()->create([
                'description' => $lineData['description'],
                'invoice_id' => $lineData['invoice_id'] ?? null,
                'account_id' => $lineData['account_id'],
                'amount' => $lineData['amount'],
                'reference' => $lineData['reference'] ?? null,
            ]);
        }

        return redirect()->back()->with('success', 'Client receipt updated successfully.');
    }

    public function destroyClientReceipt($id)
    {
        $receipt = \App\Models\ClientReceipt::findOrFail($id);
        $receipt->delete();

        return redirect()->back()->with('success', 'Client receipt deleted successfully.');
    }

    public function processClientReceipt($receipt = null)
    {
        // Process client receipt(s)
        if ($receipt) {
            $receipt = \App\Models\ClientReceipt::findOrFail($receipt);
            $receipt->update(['status' => 'Processed']);
            $message = 'Client receipt processed successfully.';
        } else {
            $receipts = \App\Models\ClientReceipt::where('status', '!=', 'Processed')->get();
            foreach ($receipts as $rec) {
                $rec->update(['status' => 'Processed']);
            }
            $message = 'All client receipts processed successfully.';
        }

        return redirect()->back()->with('success', $message);
    }

    public function supplierInvoice()
    {
        $suppliers = \App\Models\Supplier::all();
        $accounts = \App\Models\MainAccount::with('accountType')->get();
        $glAccounts = \App\Models\MainAccount::with('accountType')->get();
        $taxes = \App\Models\Tax::all();
        $currencies = \App\Models\Currency::all();
        $supplierTransactions = \App\Models\SupplierInvoice::with(['supplier', 'currency'])->get();
        $homeCurrency = \App\Models\HomeCurrency::with('currency')->first();

        return view('modules.processing.supplier-invoice', compact('suppliers', 'accounts', 'glAccounts', 'taxes', 'currencies', 'supplierTransactions', 'homeCurrency'));
    }

    public function storeSupplierInvoice(Request $request)
    {
        $validated = $request->validate([
            'supplier_id' => 'required|integer|exists:suppliers,id',
            'invoice_number' => 'required|string|max:255',
            'invoice_date' => 'required|date',
            'due_date' => 'nullable|date',
            'amount' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'account' => 'nullable|string|max:255',
            'narration' => 'nullable|string',
            'reference' => 'nullable|string|max:255',
            'gl_account' => 'nullable|string|max:255',
            'tax_id' => 'nullable|integer|exists:taxes,id',
            'discount' => 'nullable|numeric|min:0|max:100',
            'price' => 'required|numeric|min:0',
            'quantity' => 'nullable|integer|min:1',
            'currency_id' => 'nullable|integer|exists:currencies,id',
            'operation_cost' => 'nullable|boolean',
            'tax_amount' => 'nullable|numeric|min:0',
            'total' => 'required|numeric|min:0',
        ]);

        \App\Models\SupplierInvoice::create($validated);

        return redirect()->back()->with('success', 'Supplier invoice created successfully.');
    }

    public function updateSupplierInvoice(Request $request, $id)
    {
        $invoice = \App\Models\SupplierInvoice::findOrFail($id);

        $validated = $request->validate([
            'supplier_id' => 'required|integer|exists:suppliers,id',
            'invoice_number' => 'required|string|max:255',
            'invoice_date' => 'required|date',
            'due_date' => 'nullable|date',
            'amount' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'account' => 'nullable|string|max:255',
            'narration' => 'nullable|string',
            'reference' => 'nullable|string|max:255',
            'gl_account' => 'nullable|string|max:255',
            'tax_id' => 'nullable|integer|exists:taxes,id',
            'discount' => 'nullable|numeric|min:0|max:100',
            'price' => 'required|numeric|min:0',
            'quantity' => 'nullable|integer|min:1',
            'currency_id' => 'nullable|integer|exists:currencies,id',
            'operation_cost' => 'nullable|boolean',
            'tax_amount' => 'nullable|numeric|min:0',
            'total' => 'required|numeric|min:0',
        ]);

        $invoice->update($validated);

        return redirect()->back()->with('success', 'Supplier invoice updated successfully.');
    }

    public function destroySupplierInvoice($id)
    {
        $invoice = \App\Models\SupplierInvoice::findOrFail($id);
        $invoice->delete();

        return redirect()->back()->with('success', 'Supplier invoice deleted successfully.');
    }

    public function processSupplierInvoice($invoice = null)
    {
        // Process supplier invoice(s)
        if ($invoice) {
            $invoice = \App\Models\SupplierInvoice::findOrFail($invoice);
            $invoice->update(['status' => 'Processed']);
            $message = 'Supplier invoice processed successfully.';
        } else {
            $invoices = \App\Models\SupplierInvoice::where('status', '!=', 'Processed')->get();
            foreach ($invoices as $inv) {
                $inv->update(['status' => 'Processed']);
            }
            $message = 'All supplier invoices processed successfully.';
        }

        return redirect()->back()->with('success', $message);
    }

    public function cashBook()
    {
        $mainAccounts = MainAccount::with('accountType')->get();
        $currencies = \App\Models\Currency::all();
        $cashbookEntries = \App\Models\CashbookEntry::with(['cashbook', 'account', 'currency'])->get();

        return view('modules.processing.cash-book', compact('mainAccounts', 'currencies', 'cashbookEntries'));
    }

    public function journal()
    {
        // Fetch recent client journal entries
        $clientEntries = \App\Models\ClientJournal::latest('date')->take(10)->get()->map(function($entry) {
            return [
                'date' => $entry->date,
                'type' => 'Client',
                'account_code' => $entry->client_code ?? '1100',
                'description' => $entry->narration,
                'reference' => $entry->transaction_code,
                'debit' => $entry->dr ? number_format($entry->dr, 2) : '-',
                'credit' => $entry->cr ? number_format($entry->cr, 2) : '-',
                'status' => $entry->status,
            ];
        });

        // Fetch recent general journal entries
        $generalEntries = \App\Models\GeneralJournal::latest('date')->take(10)->get()->map(function($entry) {
            return [
                'date' => $entry->date,
                'type' => 'General',
                'account_code' => $entry->account_id ?? 'N/A',
                'description' => $entry->narration,
                'reference' => $entry->reference ?? '',
                'debit' => ($entry->tr_code == 'DR') ? number_format($entry->amount, 2) : '-',
                'credit' => ($entry->tr_code == 'CR') ? number_format($entry->amount, 2) : '-',
                'status' => $entry->status,
            ];
        });

        // Combine and sort entries by date descending
        $entries = $clientEntries->concat($generalEntries)->sortByDesc('date')->take(10)->values();

        return view('modules.processing.journal', compact('entries'));
    }

    public function chartOfAccounts()
    {
        $accounts = AccountType::all();
        $records = MainAccount::with('accountType')->get();
        return view('modules.edit.general-ledger', compact('accounts', 'records'));
    }

    public function chartOfAccountsOptions()
    {
        return view('modules.edit.chart-of-accounts');
    }

    public function cashbooks()
    {
        $currencies = \App\Models\Currency::all();
        $cashbooks = \App\Models\Cashbook::with(['currency'])->get();

        // Generate next code for auto-generation
        $lastCashbook = \App\Models\Cashbook::orderBy('id', 'desc')->first();
        $nextNumber = $lastCashbook ? intval(substr($lastCashbook->code, 2)) + 1 : 1;
        $nextCode = 'CB' . str_pad($nextNumber, 3, '0', STR_PAD_LEFT);

        return view('modules.edit.cashbooks', compact('currencies', 'cashbooks', 'nextCode'));
    }

    public function accountTypes()
    {
        $account_types = AccountType::all();
        return view('modules.edit.account-types', compact('account_types'));
    }

    public function multiCurrency()
    {
        $currencies = \App\Models\Currency::all();
        $homeCurrencies = \App\Models\HomeCurrency::with('currency')->get();
        $rates = \App\Models\Rate::with('baseCurrency', 'targetCurrency')->get();
        $allCurrencies = \App\Models\Currency::all();
        $homeCurrency = \App\Models\HomeCurrency::with('currency')->first();

        return view('modules.edit.multi-currency', compact('currencies', 'homeCurrencies', 'rates', 'allCurrencies', 'homeCurrency'));
    }

    public function storeCurrency(Request $request)
    {
        $request->validate([
            'currency_code' => 'required|string|max:10|unique:currencies,currency_code',
            'currency_name' => 'required|string|max:100',
            'symbol' => 'required|string|max:10',
            'country' => 'nullable|string|max:100',
        ]);

        \App\Models\Currency::create([
            'currency_code' => $request->currency_code,
            'currency_name' => $request->currency_name,
            'symbol' => $request->symbol,
            'country' => $request->country,
            'is_active' => true,
        ]);

        return redirect()->back()->with('success', 'Currency added successfully')->with('active_tab', 'currency');
    }

    public function updateCurrency(Request $request)
    {
        $request->validate([
            'id' => 'required|integer',
            'currency_code' => 'required|string|max:10|unique:currencies,currency_code,' . $request->id . ',currency_id',
            'currency_name' => 'required|string|max:100',
            'symbol' => 'required|string|max:10',
            'country' => 'nullable|string|max:100',
        ]);

        $currency = \App\Models\Currency::findOrFail($request->id);
        $currency->update([
            'currency_code' => $request->currency_code,
            'currency_name' => $request->currency_name,
            'symbol' => $request->symbol,
            'country' => $request->country,
        ]);

        return redirect()->back()->with('success', 'Currency updated successfully')->with('active_tab', 'currency');
    }

    public function deleteCurrency($id)
    {
        $currency = \App\Models\Currency::findOrFail($id);
        $currency->delete();

        return redirect()->back()->with('success', 'Currency deleted successfully')->with('active_tab', 'currency');
    }

    public function storeHomeCurrency(Request $request)
    {
        $request->validate([
            'currency' => 'required|integer|exists:currencies,currency_id',
        ]);

        // Check if a home currency already exists
        $existingHomeCurrency = \App\Models\HomeCurrency::first();

        if ($existingHomeCurrency) {
            // Store the old home currency in history
            \App\Models\HomeCurrencyHistory::create([
                'currency_id' => $existingHomeCurrency->currency_id,
                'effective_date' => $existingHomeCurrency->effective_date,
                'deactivated_date' => now(),
                'changed_by' => auth()->id(),
            ]);

            // Update existing home currency
            $existingHomeCurrency->update([
                'currency_id' => $request->currency,
                'effective_date' => now(),
                'is_active' => true,
            ]);
            $message = 'Home Currency updated successfully';
        } else {
            // Create new home currency
            \App\Models\HomeCurrency::create([
                'currency_id' => $request->currency,
                'effective_date' => now(),
                'is_active' => true,
            ]);
            $message = 'Home Currency set successfully';
        }

        return redirect()->back()->with('home_success', $message)->with('active_tab', 'home-currency');
    }

    public function updateHomeCurrency(Request $request)
{
    $request->validate([
        'id' => 'required|integer|exists:home_currency,home_currency_id',
        'currency' => 'required|integer|exists:currencies,currency_id',
    ]);

    $homeCurrency = \App\Models\HomeCurrency::findOrFail($request->id);

    // Archive the old home currency
    \App\Models\HomeCurrencyHistory::create([
        'currency_id' => $homeCurrency->currency_id,
        'effective_date' => $homeCurrency->effective_date,
        'deactivated_date' => now(),
        'changed_by' => auth()->id(),
    ]);

    // Update the current home currency
    $homeCurrency->update([
        'currency_id' => $request->currency,
        'effective_date' => now(),
    ]);

    return redirect()->back()
        ->with('home_success', 'Home Currency updated successfully')
        ->with('active_tab', 'home-currency');
}

    public function deleteHomeCurrency($id)
    {
        $homeCurrency = \App\Models\HomeCurrency::findOrFail($id);
        $homeCurrency->delete();

        return redirect()->back()->with('home_success', 'Home Currency deleted successfully')->with('active_tab', 'home-currency');
    }

    public function storeRate(Request $request)
{
    $request->validate([
        'base_currency' => 'required|integer|exists:currencies,currency_id',
        'target_currency' => 'required|integer|exists:currencies,currency_id',
        'rate_date' => 'required|date',
        'exchange_rate' => 'required|numeric|min:0.0001',
        'source' => 'nullable|string|max:100',
    ]);

    \App\Models\Rate::create([
        'base_currency_id' => $request->base_currency,
        'target_currency_id' => $request->target_currency,
        'rate_date' => $request->rate_date,
        'exchange_rate' => $request->exchange_rate,
        'source' => $request->source,
    ]);

    return redirect()->back()
        ->with('rate_success', 'Rate added successfully')
        ->with('active_tab', 'rates');
}

public function updateRate(Request $request)
{
    $request->validate([
        'id' => 'required|integer|exists:currency_rates,rate_id',
        'base_currency' => 'required|integer|exists:currencies,currency_id',
        'target_currency' => 'required|integer|exists:currencies,currency_id',
        'rate_date' => 'required|date',
        'exchange_rate' => 'required|numeric|min:0.0001',
        'source' => 'nullable|string|max:100',
    ]);

    $rate = \App\Models\Rate::findOrFail($request->id);

    $rate->update([
        'base_currency_id' => $request->base_currency,
        'target_currency_id' => $request->target_currency,
        'rate_date' => $request->rate_date,
        'exchange_rate' => $request->exchange_rate,
        'source' => $request->source,
    ]);

    return redirect()->back()
        ->with('rate_success', 'Rate updated successfully')
        ->with('active_tab', 'rates');
}


    public function deleteRate($id)
    {
        $rate = \App\Models\Rate::findOrFail($id);
        $rate->delete();

        return redirect()->back()->with('rate_success', 'Rate deleted successfully')->with('active_tab', 'rates');
    }

    public function tax()
    {
        return view('modules.edit.tax');
    }

    public function suppliers()
    {
        return view('modules.edit.suppliers');
    }

    public function supplierAccounts()
    {
        $suppliers = \App\Models\Supplier::with('currency', 'category', 'ageAnalysis', 'area')->paginate(10);
        $categories = \App\Models\SupplierCategory::all();
        $ageAnalyses = \App\Models\AgeingAnalysis::all();
        $areas = \App\Models\SupplierArea::all();
        return view('modules.edit.supplier-accs', compact('suppliers', 'categories', 'ageAnalyses', 'areas'));
    }

    public function createSupplierAccount()
    {
        $categories = \App\Models\SupplierCategory::all();
        $ageAnalyses = \App\Models\AgeingAnalysis::all();
        $areas = \App\Models\SupplierArea::all();
        return view('modules.edit.supplier-accs-create', compact('categories', 'ageAnalyses', 'areas'));
    }

    public function storeSupplierAccount(Request $request)
    {
        $validated = $request->validate([
            'supplier_code' => 'required|string|max:255|unique:suppliers,supplier_code',
            'supplier_name' => 'required|string|max:255',
            'full_name' => 'nullable|string|max:255',
            'contact_person' => 'nullable|string|max:255',
            'contact_1' => 'nullable|string|max:255',
            'contact_2' => 'nullable|string|max:255',
            'email' => 'required|string|email|max:255|unique:suppliers,email',
            'physical_address' => 'nullable|string',
            'category_id' => 'nullable|integer|exists:supplier_categories,id',
            'age_analysis_id' => 'nullable|integer|exists:ageing_analyses,id',
            'area_id' => 'nullable|integer|exists:supplier_areas,id',
        ]);

        \App\Models\Supplier::create([
            'supplier_code' => $validated['supplier_code'],
            'supplier_name' => $validated['supplier_name'],
            'full_name' => $request->input('full_name'),
            'contact_person' => $validated['contact_person'] ?? null,
            'contact_1' => $validated['contact_1'] ?? null,
            'contact_2' => $validated['contact_2'] ?? null,
            'email' => $validated['email'],
            'address' => $validated['physical_address'] ?? null,
            'category_id' => $validated['category_id'] ?? null,
            'age_analysis_id' => $validated['age_analysis_id'] ?? null,
            'area_id' => $validated['area_id'] ?? null,
        ]);

        return redirect()->route('modules.edit.suppliers.accounts')->with('success', 'Supplier account added successfully.');
    }

    public function showSupplierAccount($id)
    {
        $supplier = \App\Models\Supplier::with('currency', 'category', 'ageAnalysis', 'area')->findOrFail($id);
        return view('modules.edit.supplier-accs-show', compact('supplier'));
    }

    public function editSupplierAccount($id)
    {
        $supplier = \App\Models\Supplier::findOrFail($id);
        $categories = \App\Models\SupplierCategory::all();
        $ageAnalyses = \App\Models\AgeingAnalysis::all();
        $areas = \App\Models\SupplierArea::all();
        return view('modules.edit.edit-supplier-accs', compact('supplier', 'categories', 'ageAnalyses', 'areas'));
    }

    public function updateSupplierAccount(Request $request, $id)
    {
        $supplier = \App\Models\Supplier::findOrFail($id);

        $validated = $request->validate([
            'supplier_code' => 'required|string|max:255|unique:suppliers,supplier_code,' . $supplier->id,
            'supplier_name' => 'required|string|max:255',
            'full_name' => 'nullable|string|max:255',
            'contact_person' => 'nullable|string|max:255',
            'contact_1' => 'nullable|string|max:255',
            'contact_2' => 'nullable|string|max:255',
            'email' => 'required|string|email|max:255|unique:suppliers,email,' . $supplier->id,
            'physical_address' => 'nullable|string',
            'category_id' => 'nullable|integer|exists:supplier_categories,id',
            'age_analysis_id' => 'nullable|integer|exists:ageing_analyses,id',
            'area_id' => 'nullable|integer|exists:supplier_areas,id',
        ]);

        $supplier->update([
            'supplier_code' => $validated['supplier_code'],
            'supplier_name' => $validated['supplier_name'],
            'full_name' => $validated['full_name'] ?? null,
            'contact_person' => $validated['contact_person'] ?? null,
            'contact_1' => $validated['contact_1'] ?? null,
            'contact_2' => $validated['contact_2'] ?? null,
            'email' => $validated['email'],
            'address' => $validated['physical_address'] ?? null,
            'category_id' => $validated['category_id'] ?? null,
            'age_analysis_id' => $validated['age_analysis_id'] ?? null,
            'area_id' => $validated['area_id'] ?? null,
        ]);

        return redirect()->route('modules.edit.suppliers.accounts')->with('success', 'Supplier account updated successfully.');
    }

    public function destroySupplierAccount($id)
    {
        $supplier = \App\Models\Supplier::findOrFail($id);
        $supplier->delete();

        return redirect()->route('modules.edit.suppliers.accounts')->with('success', 'Supplier account deleted successfully.');
    }

    public function supplierAgeAnalysis()
    {
        return view('modules.edit.add-age-analysis-new');
    }

    public function supplierArea()
    {
        $supplierAreas = \App\Models\SupplierArea::paginate(10);
        return view('modules.edit.supplier-areas', compact('supplierAreas'));
    }

    public function createSupplierArea()
    {
        return view('modules.edit.supplier-areas-create');
    }

    public function storeSupplierArea(Request $request)
    {
        $validated = $request->validate([
            'area_code' => 'required|string|max:255|unique:supplier_areas,area_code',
            'area_name' => 'required|string|max:255',
            'city' => 'required|string|max:255',
            'country' => 'required|string|max:255',
            'specific_location' => 'nullable|string',
        ]);

        \App\Models\SupplierArea::create($validated);

        return redirect()->route('modules.edit.suppliers.area')->with('success', 'Supplier area added successfully.');
    }

    public function showSupplierArea($id)
    {
        $supplierArea = \App\Models\SupplierArea::findOrFail($id);
        return view('modules.edit.supplier-areas-show', compact('supplierArea'));
    }

    public function editSupplierArea($id)
    {
        $supplierArea = \App\Models\SupplierArea::findOrFail($id);
        return view('modules.edit.edit-supplier-areas', compact('supplierArea'));
    }

    public function updateSupplierArea(Request $request, $id)
    {
        $supplierArea = \App\Models\SupplierArea::findOrFail($id);

        $validated = $request->validate([
            'area_code' => 'required|string|max:255|unique:supplier_areas,area_code,' . $supplierArea->id,
            'area_name' => 'required|string|max:255',
            'city' => 'required|string|max:255',
            'country' => 'required|string|max:255',
            'specific_location' => 'nullable|string',
        ]);

        $supplierArea->update($validated);

        return redirect()->route('modules.edit.suppliers.area')->with('success', 'Supplier area updated successfully.');
    }

    public function destroySupplierArea($id)
    {
        $supplierArea = \App\Models\SupplierArea::findOrFail($id);
        $supplierArea->delete();

        return redirect()->route('modules.edit.suppliers.area')->with('success', 'Supplier area deleted successfully.');
    }

    public function supplierCategories()
    {
        $supplierCategories = \App\Models\SupplierCategory::paginate(10);
        return view('modules.edit.supplier-categories', compact('supplierCategories'));
    }

    public function createSupplierCategory()
    {
        return view('modules.edit.supplier-categories-create');
    }

    public function storeSupplierCategory(Request $request)
    {
        $validated = $request->validate([
            'category_code' => 'required|string|max:255|unique:supplier_categories,category_code',
            'category_name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        \App\Models\SupplierCategory::create($validated);

        return redirect()->route('modules.edit.suppliers.categories')->with('success', 'Supplier category added successfully.');
    }

    public function showSupplierCategory($category_code)
    {
        $supplierCategory = \App\Models\SupplierCategory::where('category_code', $category_code)->firstOrFail();
        return view('modules.edit.supplier-categories-show', compact('supplierCategory'));
    }

    public function editSupplierCategory($category_code)
    {
        $supplierCategory = \App\Models\SupplierCategory::where('category_code', $category_code)->firstOrFail();
        return view('modules.edit.edit-supplier-categories', compact('supplierCategory'));
    }

    public function updateSupplierCategory(Request $request, $category_code)
    {
        $supplierCategory = \App\Models\SupplierCategory::where('category_code', $category_code)->firstOrFail();

        $validated = $request->validate([
            'category_code' => 'required|string|max:255|unique:supplier_categories,category_code,' . $supplierCategory->category_code . ',category_code',
            'category_name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        $supplierCategory->update($validated);

        return redirect()->route('modules.edit.suppliers.categories')->with('success', 'Supplier category updated successfully.');
    }

    public function destroySupplierCategory($category_code)
    {
        $supplierCategory = \App\Models\SupplierCategory::where('category_code', $category_code)->firstOrFail();
        $supplierCategory->delete();

        return redirect()->route('modules.edit.suppliers.categories')->with('success', 'Supplier category deleted successfully.');
    }

    // Diary Calendar Methods
    public function diaryCalendar(Request $request)
    {
        $month = $request->get('month', now()->month);
        $year = $request->get('year', now()->year);

        // Create Carbon instances for the requested month
        $startOfMonth = Carbon::create($year, $month, 1);
        $endOfMonth = $startOfMonth->copy()->endOfMonth();

        // Get all diary entries for the month
        $diaryEntries = \App\Models\Diary::whereBetween('date_time', [
            $startOfMonth->startOfDay(),
            $endOfMonth->endOfDay()
        ])->orderBy('date_time', 'asc')->get();

        // Group entries by date for easier calendar display
        $entriesByDate = $diaryEntries->groupBy(function($entry) {
            return $entry->date_time->format('Y-m-d');
        });

        // Get calendar data
        $calendarDays = [];
        $currentDate = $startOfMonth->copy()->startOfWeek();

        // Generate 6 weeks of calendar
        for ($week = 0; $week < 6; $week++) {
            $weekDays = [];
            for ($day = 0; $day < 7; $day++) {
                $dateKey = $currentDate->format('Y-m-d');
                $weekDays[] = [
                    'date' => $currentDate->copy(),
                    'isCurrentMonth' => $currentDate->month == $month,
                    'isToday' => $currentDate->isToday(),
                    'entries' => $entriesByDate->get($dateKey, collect())
                ];
                $currentDate->addDay();
            }
            $calendarDays[] = $weekDays;
        }

        // Get previous and next month for navigation
        $previousMonth = $startOfMonth->copy()->subMonth();
        $nextMonth = $startOfMonth->copy()->addMonth();

        return view('modules.processing.diary-calendar', compact(
            'calendarDays',
            'startOfMonth',
            'entriesByDate',
            'previousMonth',
            'nextMonth',
            'diaryEntries'
        ));
    }

    public function diaryList(Request $request)
    {
        $query = \App\Models\Diary::query();

        // Apply filters
        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('priority') && $request->priority) {
            $query->where('priority', $request->priority);
        }

        if ($request->has('task_type') && $request->task_type) {
            $query->where('task_type', $request->task_type);
        }

        if ($request->has('start_date') && $request->start_date) {
            $query->whereDate('date_time', '>=', $request->start_date);
        }

        if ($request->has('end_date') && $request->end_date) {
            $query->whereDate('date_time', '<=', $request->end_date);
        }

        $diaryEntries = $query->orderBy('date_time', 'asc')->paginate(15);

        return view('modules.processing.diary-list', compact('diaryEntries'));
    }

    public function diaryAdd()
    {
        return view('modules.processing.add-diary');
    }

    public function diaryStore(Request $request)
    {
        $validated = $request->validate([
            'date_time' => 'required|date',
            'task_title' => 'required|string|max:255',
            'client_case' => 'nullable|string|max:255',
            'task_type' => 'required|string|max:50',
            'priority' => 'required|in:low,medium,high',
            'location' => 'nullable|string|max:255',
            'duration' => 'nullable|string|max:50',
            'status' => 'required|in:pending,in-progress,completed,cancelled',
            'notes' => 'nullable|string',
        ]);

        // Combine date and time if time is provided separately
        if ($request->has('time') && $request->time) {
            $date = Carbon::parse($request->input('date_time'));
            $validated['date_time'] = $date->format('Y-m-d') . ' ' . $request->time;
        }

        \App\Models\Diary::create($validated);

        return redirect()->route('modules.processing.diary.list')->with('success', 'Diary entry created successfully.');
    }

    public function diaryShow($id)
    {
        $diaryEntry = \App\Models\Diary::findOrFail($id);
        return view('modules.processing.diary-show', compact('diaryEntry'));
    }

    public function diaryEdit($id)
    {
        $diaryEntry = \App\Models\Diary::findOrFail($id);
        return view('modules.processing.edit-diary', compact('diaryEntry'));
    }

    public function diaryUpdate(Request $request, $id)
    {
        $diaryEntry = \App\Models\Diary::findOrFail($id);

        $validated = $request->validate([
            'date_time' => 'required|date',
            'task_title' => 'required|string|max:255',
            'client_case' => 'nullable|string|max:255',
            'task_type' => 'required|string|max:50',
            'priority' => 'required|in:low,medium,high',
            'location' => 'nullable|string|max:255',
            'duration' => 'nullable|string|max:50',
            'status' => 'required|in:pending,in-progress,completed,cancelled',
            'notes' => 'nullable|string',
        ]);

        // Combine date and time if time is provided separately
        if ($request->has('time') && $request->time) {
            $date = Carbon::parse($request->input('date_time'));
            $validated['date_time'] = $date->format('Y-m-d') . ' ' . $request->time;
        }

        $diaryEntry->update($validated);

        return redirect()->route('modules.processing.diary.list')->with('success', 'Diary entry updated successfully.');
    }

    public function diaryDelete($id)
    {
        $diaryEntry = \App\Models\Diary::findOrFail($id);
        $diaryEntry->delete();

        return redirect()->route('modules.processing.diary.list')->with('success', 'Diary entry deleted successfully.');
    }


    public function clients()
    {
        return view('modules.edit.client');
    }

    public function addClients()
    {
        return view('modules.edit.add-clients');
    }

    public function case()
    {
        return view('modules.edit.case');
    }

    public function casesOptions()
    {
        return view('modules.edit.cases-options');
    }

    public function caseParties()
    {
        return view('modules.edit.case-parties');
    }

    public function caseRoles()
    {
        return view('modules.edit.case-roles');
    }

    public function casePriorities()
    {
        return view('modules.edit.case-priorities');
    }

    public function casePrioritiesAdd()
    {
        return view('modules.edit.case-priorities-add');
    }

    public function options()
    {
        return view('modules.processing.options');
    }

    public function client()
    {
        return view('modules.processing.client');
    }

    public function clientQuotation()
    {
        $clients = \App\Models\Client::all();
        $clientQuotations = \App\Models\ClientQuotation::with(['client', 'lines'])->get();

        return view('modules.processing.client-quotation', compact('clients', 'clientQuotations'));
    }

    public function storeClientQuotation(Request $request)
    {
        $validated = $request->validate([
            'client_id' => 'required|integer|exists:clients,id',
            'quotation_date' => 'required|date',
            'terms' => 'nullable|string',
            'validity_period' => 'nullable|integer|min:1|max:365',
            'notes' => 'nullable|string',
            'services' => 'required|array|min:1',
            'services.*.description' => 'required|string',
            'services.*.quantity' => 'required|integer|min:1',
            'services.*.unit_price' => 'required|numeric|min:0',
        ]);

        // Generate quotation number
        $quotationNumber = 'QUO-' . date('Y') . '-' . str_pad(\App\Models\ClientQuotation::count() + 1, 4, '0', STR_PAD_LEFT);

        // Calculate totals
        $totalAmount = 0;
        foreach ($validated['services'] as $service) {
            $totalAmount += $service['quantity'] * $service['unit_price'];
        }

        // Create quotation
        $quotation = \App\Models\ClientQuotation::create([
            'client_id' => $validated['client_id'],
            'quotation_number' => $quotationNumber,
            'quotation_date' => $validated['quotation_date'],
            'terms' => $validated['terms'] ?? null,
            'validity_period' => $validated['validity_period'] ?? 30,
            'notes' => $validated['notes'] ?? null,
            'status' => 'Draft',
        ]);

        // Create quotation lines
        foreach ($validated['services'] as $service) {
            $quotation->lines()->create([
                'description' => $service['description'],
                'quantity' => $service['quantity'],
                'unit_price' => $service['unit_price'],
                'total' => $service['quantity'] * $service['unit_price'],
            ]);
        }

        return redirect()->back()->with('success', 'Client quotation created successfully.');
    }

    public function updateClientQuotation(Request $request, $id)
    {
        $quotation = \App\Models\ClientQuotation::findOrFail($id);

        $validated = $request->validate([
            'client_id' => 'required|integer|exists:clients,id',
            'quotation_date' => 'required|date',
            'terms' => 'nullable|string',
            'validity_period' => 'nullable|integer|min:1|max:365',
            'notes' => 'nullable|string',
            'services' => 'required|array|min:1',
            'services.*.description' => 'required|string',
            'services.*.quantity' => 'required|integer|min:1',
            'services.*.unit_price' => 'required|numeric|min:0',
        ]);

        // Calculate totals
        $totalAmount = 0;
        foreach ($validated['services'] as $service) {
            $totalAmount += $service['quantity'] * $service['unit_price'];
        }

        // Update quotation
        $quotation->update([
            'client_id' => $validated['client_id'],
            'quotation_date' => $validated['quotation_date'],
            'terms' => $validated['terms'] ?? null,
            'validity_period' => $validated['validity_period'] ?? 30,
            'notes' => $validated['notes'] ?? null,
        ]);

        // Delete old lines and create new ones
        $quotation->lines()->delete();
        foreach ($validated['services'] as $service) {
            $quotation->lines()->create([
                'description' => $service['description'],
                'quantity' => $service['quantity'],
                'unit_price' => $service['unit_price'],
                'total' => $service['quantity'] * $service['unit_price'],
            ]);
        }

        return redirect()->back()->with('success', 'Client quotation updated successfully.');
    }

    public function destroyClientQuotation($id)
    {
        $quotation = \App\Models\ClientQuotation::findOrFail($id);
        $quotation->delete();

        return redirect()->back()->with('success', 'Client quotation deleted successfully.');
    }

    public function processClientQuotation($quotation = null)
    {
        if ($quotation) {
            $quotation = \App\Models\ClientQuotation::findOrFail($quotation);
            $quotation->update(['status' => 'Sent']);
            $message = 'Client quotation sent successfully.';
        } else {
            $quotations = \App\Models\ClientQuotation::where('status', 'Draft')->get();
            foreach ($quotations as $quo) {
                $quo->update(['status' => 'Sent']);
            }
            $message = 'All draft quotations sent successfully.';
        }

        return redirect()->back()->with('success', $message);
    }

    public function editClientQuotation($id)
    {
        $quotation = \App\Models\ClientQuotation::with('lines')->findOrFail($id);
        $clients = \App\Models\Client::all();
        $clientQuotations = \App\Models\ClientQuotation::with(['client', 'lines'])->get();

        return view('modules.processing.client-quotation', compact('clients', 'clientQuotations', 'quotation'));
    }

    public function defendant()
    {
        return view('modules.edit.defendant');
    }

    public function lawyer()
    {
        return view('modules.edit.lawyer');
    }

    // Case Reasons Management Methods
    public function caseReasons()
    {
        return view('modules.edit.case-reasons');
    }

    public function caseReasonsList()
    {
        return view('modules.edit.case-reasons-list');
    }

    public function caseReasonsAdd()
    {
        return view('modules.edit.add-case-reasons');
    }

    public function caseReasonsShow($id)
    {
        return view('modules.edit.case-reasons-show');
    }

    public function caseReasonsEdit($id)
    {
        return view('modules.edit.edit-case-reasons');
    }

    public function clientJournal()
    {
        $clientAccount = \App\Models\Client::first() ?? (object)['client_id' => '1'];
        $clientAccounts = \App\Models\Client::all();
        $currencies = \App\Models\Currency::all();
        $tempEntries = DB::table('journal_transactions')
            ->where('journal_type', 'client')
            ->orderBy('created_at', 'desc')
            ->get();

        return view('modules.processing.journals.client', compact('clientAccount', 'clientAccounts', 'currencies', 'tempEntries'));
    }

    public function storeClientJournal(Request $request)
    {
        $validated = $request->validate([
            'client_account_id' => 'required|integer|exists:client_accounts,id',
            'date' => 'required|date',
            'narration' => 'required|string',
            'tr_code' => 'required|in:DR,CR',
            'currency_id' => 'required|integer|exists:currencies,id',
            'amount' => 'required|numeric|min:0',
            'reference' => 'nullable|string|max:255',
        ]);

        \App\Models\ClientJournal::create($validated);

        return redirect()->back()->with('success', 'Client journal entry added successfully.');
    }

    public function updateClientJournal(Request $request, $id)
    {
        $entry = \App\Models\ClientJournal::findOrFail($id);

        $validated = $request->validate([
            'client_account_id' => 'required|integer|exists:client_accounts,id',
            'date' => 'required|date',
            'narration' => 'required|string',
            'tr_code' => 'required|in:DR,CR',
            'currency_id' => 'required|integer|exists:currencies,id',
            'amount' => 'required|numeric|min:0',
            'reference' => 'nullable|string|max:255',
        ]);

        $entry->update($validated);

        return redirect()->back()->with('success', 'Client journal entry updated successfully.');
    }

    public function processClientJournal(Request $request)
    {
        // Process all client journal entries
        $entries = \App\Models\ClientJournal::where('status', '!=', 'Processed')->get();
        foreach ($entries as $entry) {
            $entry->update(['status' => 'Processed']);
        }

        return redirect()->back()->with('success', 'All client journal entries processed successfully.');
    }

    public function processSingleClientJournal($id)
    {
        $entry = \App\Models\ClientJournal::findOrFail($id);
        $entry->update(['status' => 'Processed']);

        return redirect()->back()->with('success', 'Client journal entry processed successfully.');
    }

    public function deleteClientJournal($id)
    {
        $entry = \App\Models\ClientJournal::findOrFail($id);
        $entry->delete();

        return redirect()->back()->with('success', 'Client journal entry deleted successfully.');
    }

    public function generalJournal()
    {
        $accounts = \App\Models\MainAccount::with('accountType')->get();
        $currencies = \App\Models\Currency::all();
        $tempEntries = \App\Models\GeneralJournal::with('account', 'currency')->get();

        return view('modules.processing.journals.general', compact('accounts', 'currencies', 'tempEntries'));
    }

    public function supplierJournal()
    {
        $supplierAccount = \App\Models\Supplier::first() ?? (object)['code' => '2100'];
        $suppliers = \App\Models\Supplier::all();
        $currencies = \App\Models\Currency::all();
        $tempEntries = collect(); // Assuming no specific model, using empty collection for now

        return view('modules.processing.journals.supplier', compact('supplierAccount', 'suppliers', 'currencies', 'tempEntries'));
    }

    // Case Status Management Methods
    public function caseStatus()
    {
        return view('modules.edit.case-status');
    }

    public function caseStatusList()
    {
        $caseStatuses = \App\Models\CaseStatus::all();
        return view('modules.edit.case-status.list', compact('caseStatuses'));
    }

    public function caseStatusAdd()
    {
        return view('modules.edit.case-status.add');
    }

    public function caseStatusShow($status_code)
    {
        $caseStatus = \App\Models\CaseStatus::findOrFail($status_code);
        return view('modules.edit.case-status.show', compact('caseStatus'));
    }

    public function caseStatusEdit($status_code)
    {
        $caseStatus = \App\Models\CaseStatus::findOrFail($status_code);
        return view('modules.edit.case-status.edit', compact('caseStatus'));
    }

    public function caseStatusStore(Request $request)
    {
        $validated = $request->validate([
            'status_code' => 'required|string|max:20|unique:case_statuses,status_code',
            'status_name' => 'required|string|max:255|unique:case_statuses,status_name',
            'status_description' => 'nullable|string',
        ]);

        $validated['is_active'] = true; // Default to active
        $validated['sort_order'] = 0; // Default sort order
        $validated['color_code'] = '#6b7280'; // Default color

        \App\Models\CaseStatus::create($validated);

        return redirect()->route('modules.edit.case-status.list')->with('success', 'Case status added successfully.');
    }

    public function caseStatusUpdate(Request $request, $status_code)
    {
        $caseStatus = \App\Models\CaseStatus::findOrFail($status_code);

        $validated = $request->validate([
            'status_code' => 'required|string|max:20|unique:case_statuses,status_code,' . $caseStatus->status_code . ',status_code',
            'status_name' => 'required|string|max:255|unique:case_statuses,status_name,' . $caseStatus->status_code . ',status_code',
            'status_description' => 'nullable|string',
        ]);

        $caseStatus->update($validated);

        return redirect()->route('modules.edit.case-status.list')->with('success', 'Case status updated successfully.');
    }

    public function caseStatusDelete($status_code)
    {
        $caseStatus = \App\Models\CaseStatus::findOrFail($status_code);
        $caseStatus->delete();

        return redirect()->route('modules.edit.case-status.list')->with('success', 'Case status deleted successfully.');
    }

    // General Journal Methods
    public function storeGeneralJournal(Request $request)
    {
        // Basic implementation - you may need to adjust based on your model
        $validated = $request->validate([
            'account_id' => 'required|integer',
            'date' => 'required|date',
            'narration' => 'required|string',
            'tr_code' => 'required|in:DR,CR',
            'currency_id' => 'required|integer',
            'amount' => 'required|numeric|min:0',
        ]);

        // Assuming you have a GeneralJournal model
        \App\Models\GeneralJournal::create($validated);

        return redirect()->back()->with('success', 'General journal entry added successfully.');
    }

    public function processGeneralJournal(Request $request)
    {
        // Process all general journal entries
        $entries = \App\Models\GeneralJournal::where('status', '!=', 'Processed')->get();
        foreach ($entries as $entry) {
            $entry->update(['status' => 'Processed']);
        }

        return redirect()->back()->with('success', 'All general journal entries processed successfully.');
    }

    public function processSingleGeneralJournal($id)
    {
        $entry = \App\Models\GeneralJournal::findOrFail($id);
        $entry->update(['status' => 'Processed']);

        return redirect()->back()->with('success', 'General journal entry processed successfully.');
    }

    public function updateGeneralJournal(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required|integer',
            'account_id' => 'required|integer',
            'date' => 'required|date',
            'narration' => 'required|string',
            'tr_code' => 'required|in:DR,CR',
            'currency_id' => 'required|integer',
            'amount' => 'required|numeric|min:0',
        ]);

        $entry = \App\Models\GeneralJournal::findOrFail($validated['id']);
        $entry->update($validated);

        return redirect()->back()->with('success', 'General journal entry updated successfully.');
    }

    public function deleteGeneralJournal($id)
    {
        $entry = \App\Models\GeneralJournal::findOrFail($id);
        $entry->delete();

        return redirect()->back()->with('success', 'General journal entry deleted successfully.');
    }

    // Cashbook Methods
    public function storeCashbookEntry(Request $request)
    {
        $validated = $request->validate([
            'cashbook_id' => 'required|integer|exists:main_accounts,id',
            'entry_date' => 'required|date',
            'module' => 'required|in:AR,AP,GL,CB',
            'account_id' => 'required|integer|exists:main_accounts,id',
            'reference' => 'nullable|string|max:255',
            'description' => 'required|string',
            'tr_code' => 'required|in:DR,CR',
            'currency_id' => 'required|integer|exists:currencies,id',
            'amount' => 'required|numeric|min:0',
        ]);

        \App\Models\CashbookEntry::create($validated);

        return redirect()->back()->with('success', 'Cashbook entry added successfully.');
    }

    public function editCashbookEntry($id)
    {
        $entry = \App\Models\CashbookEntry::with(['cashbook', 'account', 'currency'])->findOrFail($id);
        $mainAccounts = MainAccount::with('accountType')->get();
        $currencies = \App\Models\Currency::all();

        return view('modules.processing.cash-book', compact('entry', 'mainAccounts', 'currencies'));
    }

    public function updateCashbookEntry(Request $request, $id)
    {
        $entry = \App\Models\CashbookEntry::findOrFail($id);

        $validated = $request->validate([
            'cashbook_id' => 'required|integer|exists:main_accounts,id',
            'entry_date' => 'required|date',
            'module' => 'required|in:AR,AP,GL,CB',
            'account_id' => 'required|integer|exists:main_accounts,id',
            'reference' => 'nullable|string|max:255',
            'description' => 'required|string',
            'tr_code' => 'required|in:DR,CR',
            'currency_id' => 'required|integer|exists:currencies,id',
            'amount' => 'required|numeric|min:0',
        ]);

        $entry->update($validated);

        return redirect()->back()->with('success', 'Cashbook entry updated successfully.');
    }

    public function deleteCashbookEntry($id)
    {
        $entry = \App\Models\CashbookEntry::findOrFail($id);
        $entry->delete();

        return redirect()->back()->with('success', 'Cashbook entry deleted successfully.');
    }

    // Assignment Management Methods
    public function createAssignment()
    {
        $lawyers = \App\Models\Lawyer::all();
        $clients = \App\Models\Client::all();
        $cases = \App\Models\LegalCase::all();
        $caseTypes = \App\Models\CaseType::all();
        $caseCategories = \App\Models\CaseCategory::all();
        $caseStatuses = \App\Models\CaseStatus::all();
        $priorities = ['low', 'medium', 'high'];
        $taskTypes = ['document_review', 'client_meeting', 'court_appearance', 'research', 'filing', 'correspondence', 'administrative'];

        return view('modules.processing.create-assignment', compact(
            'lawyers', 'clients', 'cases', 'caseTypes', 'caseCategories', 'caseStatuses', 'priorities', 'taskTypes'
        ));
    }

    public function storeAssignment(Request $request)
    {
        $validated = $request->validate([
            'assignment_title' => 'required|string|max:255',
            'lawyer_id' => 'required|integer|exists:lawyers,id',
            'client_id' => 'nullable|integer|exists:clients,id',
            'case_id' => 'nullable|integer|exists:legal_cases,id',
            'case_type_id' => 'nullable|integer|exists:case_types,id',
            'case_category_id' => 'nullable|integer|exists:case_categories,id',
            'case_status_id' => 'nullable|integer|exists:case_statuses,id',
            'task_type' => 'required|string|max:50',
            'priority' => 'required|in:low,medium,high',
            'due_date' => 'required|date',
            'due_time' => 'nullable|date_format:H:i',
            'estimated_hours' => 'nullable|numeric|min:0.5|max:100',
            'description' => 'required|string',
            'instructions' => 'nullable|string',
            'documents_required' => 'nullable|string',
            'court_date' => 'nullable|date',
            'court_time' => 'nullable|date_format:H:i',
            'court_location' => 'nullable|string|max:255',
            'opposing_counsel' => 'nullable|string|max:255',
            'client_contact_info' => 'nullable|string',
            'billing_notes' => 'nullable|string',
            'status' => 'required|in:pending,in_progress,completed,cancelled',
        ]);

        // Combine date and time for due_datetime
        if ($request->has('due_time') && $request->due_time) {
            $validated['due_datetime'] = $request->due_date . ' ' . $request->due_time;
        } else {
            $validated['due_datetime'] = $request->due_date . ' 23:59:59';
        }

        // Combine court date and time if provided
        if ($request->has('court_date') && $request->court_date && $request->has('court_time') && $request->court_time) {
            $validated['court_datetime'] = $request->court_date . ' ' . $request->court_time;
        }

        // Set default values
        $validated['created_by'] = auth()->id() ?? 1;
        $validated['assigned_date'] = now();
        $validated['progress_percentage'] = 0;
        $validated['is_billable'] = true;

        /** @var Assignment $assignment */
        $assignment = Assignment::create($validated);

        // Log assignment creation
        \App\Models\AssignmentHistory::create([
            'assignment_id' => $assignment->id,
            'user_id' => auth()->id(),
            'action' => 'created',
            'description' => 'Assignment created',
        ]);

        return redirect()->route('modules.processing.assignment.list')->with('success', 'Assignment created successfully.');
    }

    public function listAssignments(Request $request)
    {
        $query = \App\Models\Assignment::with(['lawyer', 'client', 'legalCase', 'caseType', 'caseCategory', 'caseStatus']);

        // Apply filters
        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        if ($request->has('priority') && $request->priority) {
            $query->where('priority', $request->priority);
        }

        if ($request->has('lawyer_id') && $request->lawyer_id) {
            $query->where('lawyer_id', $request->lawyer_id);
        }

        if ($request->has('task_type') && $request->task_type) {
            $query->where('task_type', $request->task_type);
        }

        if ($request->has('start_date') && $request->start_date) {
            $query->whereDate('due_datetime', '>=', $request->start_date);
        }

        if ($request->has('end_date') && $request->end_date) {
            $query->whereDate('due_datetime', '<=', $request->end_date);
        }

        $assignments = $query->orderBy('due_datetime', 'asc')->paginate(15);
        $lawyers = \App\Models\Lawyer::all();
        $taskTypes = ['document_review', 'client_meeting', 'court_appearance', 'research', 'filing', 'correspondence', 'administrative'];

        return view('modules.processing.assignment-list', compact('assignments', 'lawyers', 'taskTypes'));
    }

    public function showAssignment($id)
    {
        $assignment = \App\Models\Assignment::with(['lawyer', 'client', 'legalCase', 'caseType', 'caseCategory', 'caseStatus'])->findOrFail($id);
        return view('modules.processing.assignment-show', compact('assignment'));
    }

    public function editAssignment($id)
    {
        $assignment = \App\Models\Assignment::findOrFail($id);
        $lawyers = \App\Models\Lawyer::all();
        $clients = \App\Models\Client::all();
        $cases = \App\Models\LegalCase::all();
        $caseTypes = \App\Models\CaseType::all();
        $caseCategories = \App\Models\CaseCategory::all();
        $caseStatuses = \App\Models\CaseStatus::all();
        $priorities = ['low', 'medium', 'high'];
        $taskTypes = ['document_review', 'client_meeting', 'court_appearance', 'research', 'filing', 'correspondence', 'administrative'];

        return view('modules.processing.edit-assignment', compact(
            'assignment', 'lawyers', 'clients', 'cases', 'caseTypes', 'caseCategories', 'caseStatuses', 'priorities', 'taskTypes'
        ));
    }

    public function updateAssignment(Request $request, $id)
    {
        /** @var \App\Models\Assignment $assignment */
        $assignment = \App\Models\Assignment::findOrFail($id);

        $validated = $request->validate([
            'assignment_title' => 'required|string|max:255',
            'lawyer_id' => 'required|integer|exists:lawyers,id',
            'client_id' => 'nullable|integer|exists:clients,id',
            'case_id' => 'nullable|integer|exists:legal_cases,id',
            'case_type_id' => 'nullable|integer|exists:case_types,id',
            'case_category_id' => 'nullable|integer|exists:case_categories,id',
            'case_status_id' => 'nullable|integer|exists:case_statuses,id',
            'task_type' => 'required|string|max:50',
            'priority' => 'required|in:low,medium,high',
            'due_date' => 'required|date',
            'due_time' => 'nullable|date_format:H:i',
            'estimated_hours' => 'nullable|numeric|min:0.5|max:100',
            'description' => 'required|string',
            'instructions' => 'nullable|string',
            'documents_required' => 'nullable|string',
            'court_date' => 'nullable|date',
            'court_time' => 'nullable|date_format:H:i',
            'court_location' => 'nullable|string|max:255',
            'opposing_counsel' => 'nullable|string|max:255',
            'client_contact_info' => 'nullable|string',
            'billing_notes' => 'nullable|string',
            'status' => 'required|in:pending,in_progress,completed,cancelled',
        ]);

        // Combine date and time for due_datetime
        if ($request->has('due_time') && $request->due_time) {
            $validated['due_datetime'] = $request->due_date . ' ' . $request->due_time;
        } else {
            $validated['due_datetime'] = $request->due_date . ' 23:59:59';
        }

        // Combine court date and time if provided
        if ($request->has('court_date') && $request->court_date && $request->has('court_time') && $request->court_time) {
            $validated['court_datetime'] = $request->court_date . ' ' . $request->court_time;
        }

        $assignment->update($validated);

        // Log assignment update
        \App\Models\AssignmentHistory::create([
            'assignment_id' => $assignment->id,
            'user_id' => auth()->id(),
            'action' => 'updated',
            'description' => 'Assignment updated',
        ]);

        return redirect()->route('modules.processing.assignment.list')->with('success', 'Assignment updated successfully.');
    }

    public function deleteAssignment($id)
    {
        /** @var \App\Models\Assignment $assignment */
        $assignment = \App\Models\Assignment::findOrFail($id);

        // Log assignment deletion
        \App\Models\AssignmentHistory::create([
            'assignment_id' => $assignment->id,
            'user_id' => auth()->id(),
            'action' => 'deleted',
            'description' => 'Assignment deleted',
        ]);

        $assignment->delete();

        return redirect()->route('modules.processing.assignment.list')->with('success', 'Assignment deleted successfully.');
    }

    public function reassignAssignment(Request $request, $id)
    {
        $validated = $request->validate([
            'new_lawyer_id' => 'required|integer|exists:lawyers,id',
            'reassignment_reason' => 'nullable|string|max:255',
        ]);

        /** @var \App\Models\Assignment $assignment */
        $assignment = \App\Models\Assignment::findOrFail($id);

        // Store old lawyer for logging
        $oldLawyer = $assignment->lawyer;
        $oldLawyerName = $oldLawyer ? $oldLawyer->first_name . ' ' . $oldLawyer->last_name : 'No Lawyer Assigned';

        $newLawyer = \App\Models\Lawyer::find($validated['new_lawyer_id']);

        // Update assignment with new lawyer
        $assignment->update([
            'lawyer_id' => $validated['new_lawyer_id'],
            'assigned_date' => now(), // Reset assigned date
        ]);

        // Log reassignment
        \App\Models\AssignmentHistory::create([
            'assignment_id' => $assignment->id,
            'user_id' => auth()->id(),
            'action' => 'reassigned',
            'description' => 'Assignment reassigned from ' . $oldLawyerName . ' to ' . $newLawyer->first_name . ' ' . $newLawyer->last_name,
        ]);

        return redirect()->back()->with('success', 'Assignment reassigned successfully from ' . $oldLawyerName . ' to ' . $newLawyer->first_name . ' ' . $newLawyer->last_name . '.');
    }

    public function assignmentReports(Request $request)
    {
        // Build query with eager loading
        $query = \App\Models\Assignment::with([
            'lawyer', 'client', 'legalCase', 'caseType', 'caseCategory', 
            'caseStatus', 'creator', 'histories'
        ]);
    
        // Apply filters at the query builder level
        if ($request->filled('date_from')) {
            $query->whereDate('due_datetime', '>=', $request->date_from);
        }
    
        if ($request->filled('date_to')) {
            $query->whereDate('due_datetime', '<=', $request->date_to);
        }
    
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
    
        if ($request->filled('priority')) {
            $query->where('priority', $request->priority);
        }
    
        if ($request->filled('lawyer_id')) {
            $query->where('lawyer_id', $request->lawyer_id);
        }
    
        if ($request->filled('task_type')) {
            $query->where('task_type', $request->task_type);
        }
    
        // Execute query
        $assignments = $query->get();
        $lawyers = \App\Models\Lawyer::all();
    
        // Compute stats safely
        $stats = $this->calculateStats($assignments);
    
        // Data for charts - Collection-safe
        $statusData = [
            'labels' => ['Pending', 'In Progress', 'Completed', 'Cancelled'],
            'data' => [
                $assignments->where('status', 'pending')->count(),
                $assignments->where('status', 'in_progress')->count(),
                $assignments->where('status', 'completed')->count(),
                $assignments->where('status', 'cancelled')->count(),
            ]
        ];
    
        $priorityData = [
            'labels' => ['Low', 'Medium', 'High'],
            'data' => [
                $assignments->where('priority', 'low')->count(),
                $assignments->where('priority', 'medium')->count(),
                $assignments->where('priority', 'high')->count(),
            ]
        ];
    
    // Productivity per lawyer
    $productivityData = [
        'labels' => $lawyers->map(function($l) { return $l->first_name . ' ' . $l->last_name; })->toArray(),
        'data' => $lawyers->map(function($l) { return $assignments->where('lawyer_id', $l->id)->where('status', 'completed')->count(); })->toArray()
    ];
    
        // Deadline analysis
        $deadlineData = [
            'labels' => ['Overdue', 'Due Today', 'Due This Week', 'Due This Month'],
            'data' => [
                $assignments->filter(fn($a) => $a->due_datetime && $a->due_datetime < now() && $a->status !== 'completed')->count(),
                $assignments->filter(fn($a) => $a->due_datetime && $a->due_datetime->isToday() && $a->status !== 'completed')->count(),
                $assignments->filter(fn($a) => $a->due_datetime && $a->due_datetime->between(now(), now()->addDays(7)) && $a->status !== 'completed')->count(),
                $assignments->filter(fn($a) => $a->due_datetime && $a->due_datetime->between(now(), now()->addDays(30)) && $a->status !== 'completed')->count(),
            ]
        ];
    
        // Task type distribution
        $taskTypes = $assignments->groupBy('task_type')->map->count();
        $taskTypeData = [
            'labels' => $taskTypes->keys()->toArray(),
            'data' => $taskTypes->values()->toArray()
        ];
    
        // Monthly trend (last 12 months)
        $monthlyData = $this->getMonthlyAssignmentData();
    
        // Detailed assignments for table
        $detailedAssignments = $assignments->map(fn($a) => [
            'id' => $a->id,
            'title' => $a->assignment_title,
            'lawyer' => $a->lawyer ? $a->lawyer->first_name . ' ' . $a->lawyer->last_name : 'N/A',
            'client' => $a->client?->client_name ?? 'N/A',
            'case' => $a->legalCase?->case_title ?? 'N/A',
            'priority' => $a->priority,
            'status' => $a->status,
            'due_date' => $a->due_datetime?->format('M d, Y H:i') ?? 'N/A',
            'progress' => $a->progress_percentage,
            'estimated_hours' => $a->estimated_hours,
            'created_by' => $a->creator?->name ?? 'N/A',
            'assigned_date' => $a->assigned_date?->format('M d, Y') ?? 'N/A',
            'created_date' => $a->created_at?->format('M d, Y') ?? 'N/A',
            'is_billable' => $a->is_billable,
            'description' => $a->description,
            'instructions' => $a->instructions,
            'documents_required' => $a->documents_required,
            'court_date' => $a->court_datetime?->format('M d, Y H:i') ?? 'N/A',
            'court_location' => $a->court_location,
            'opposing_counsel' => $a->opposing_counsel,
            'client_contact_info' => $a->client_contact_info,
            'billing_notes' => $a->billing_notes,
            'histories' => $a->histories->map(fn($h) => [
                'action' => $h->action,
                'description' => $h->description,
                'created_at' => $h->created_at->format('M d, Y H:i')
            ])
        ]);
    
        return view('modules.processing.assignment-reports', compact(
            'assignments',
            'detailedAssignments',
            'lawyers',
            'stats',
            'statusData',
            'priorityData',
            'productivityData',
            'deadlineData',
            'taskTypeData',
            'monthlyData'
        ));
    }
    
    // Helper for average completion time
    private function calculateAverageCompletionTime($assignments)
    {
        $completedAssignments = $assignments->where('status', 'completed');
        if ($completedAssignments->isEmpty()) {
            return 0;
        }
    
        $totalDays = $completedAssignments->sum(fn($a) => 
            $a->assigned_date && $a->due_datetime ? $a->assigned_date->diffInDays($a->due_datetime) : 0
        );
    
        return round($totalDays / $completedAssignments->count(), 1);
    }
    
    // Monthly assignment data for last 12 months
    private function getMonthlyAssignmentData()
    {
        $data = ['labels' => [], 'data' => []];
        for ($i = 11; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $monthName = $date->format('M Y');
            $count = \App\Models\Assignment::whereYear('created_at', $date->year)
                ->whereMonth('created_at', $date->month)
                ->count();
            $data['labels'][] = $monthName;
            $data['data'][] = $count;
        }
        return $data;
    }
    
    // Collection-safe stats helper
    private function calculateStats($assignments)
    {
        return [
            'total_assignments' => $assignments->count(),
            'pending_assignments' => $assignments->where('status', 'pending')->count(),
            'in_progress_assignments' => $assignments->where('status', 'in_progress')->count(),
            'completed_assignments' => $assignments->where('status', 'completed')->count(),
            'overdue_assignments' => $assignments->filter(fn($a) => $a->due_datetime && $a->due_datetime < now() && $a->status !== 'completed')->count(),
            'cancelled_assignments' => $assignments->where('status', 'cancelled')->count(),
            'average_completion_time' => $this->calculateAverageCompletionTime($assignments),
            'total_estimated_hours' => $assignments->sum('estimated_hours'),
            'total_actual_hours' => $assignments->where('status', 'completed')->sum('estimated_hours'),
            'billable_assignments' => $assignments->where('is_billable', true)->count(),
            'high_priority_assignments' => $assignments->where('priority', 'high')->count(),
        ];
    }
    

    private function exportAssignmentReportsExcel($assignments, $request)
    {
        $filename = 'assignment-reports-' . now()->format('Y-m-d') . '.xlsx';

        // Create CSV content that Excel can open
        $headers = [
            'ID',
            'Title',
            'Lawyer',
            'Client',
            'Case',
            'Priority',
            'Status',
            'Due Date',
            'Progress (%)',
            'Estimated Hours',
            'Created Date',
            'Assigned Date'
        ];

        $rows = [];
        foreach ($assignments as $assignment) {
            $rows[] = [
                $assignment->id,
                $assignment->assignment_title,
                $assignment->lawyer ? $assignment->lawyer->first_name . ' ' . $assignment->lawyer->last_name : 'N/A',
                $assignment->client ? $assignment->client->client_name : 'N/A',
                $assignment->legalCase ? $assignment->legalCase->case_title : 'N/A',
                ucfirst($assignment->priority),
                ucfirst(str_replace('_', ' ', $assignment->status)),
                $assignment->due_datetime ? $assignment->due_datetime->format('M d, Y H:i') : 'N/A',
                $assignment->progress_percentage,
                $assignment->estimated_hours,
                $assignment->created_at ? $assignment->created_at->format('M d, Y') : 'N/A',
                $assignment->assigned_date ? $assignment->assigned_date->format('M d, Y') : 'N/A'
            ];
        }

        $content = implode("\n", array_map(function($row) {
            return implode(',', array_map(function($cell) {
                return '"' . str_replace('"', '""', $cell) . '"';
            }, $row));
        }, array_merge([$headers], $rows)));

        return response($content)
            ->header('Content-Type', 'text/csv')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
    }

    private function exportAssignmentReportsCSV($assignments, $request)
    {
        $filename = 'assignment-reports-' . now()->format('Y-m-d') . '.csv';

        $headers = [
            'ID',
            'Title',
            'Lawyer',
            'Client',
            'Case',
            'Priority',
            'Status',
            'Due Date',
            'Progress (%)',
            'Estimated Hours',
            'Created Date',
            'Assigned Date',
            'Description',
            'Instructions'
        ];

        $rows = [];
        foreach ($assignments as $assignment) {
            $rows[] = [
                $assignment->id,
                $assignment->assignment_title,
                $assignment->lawyer ? $assignment->lawyer->first_name . ' ' . $assignment->lawyer->last_name : 'N/A',
                $assignment->client ? $assignment->client->client_name : 'N/A',
                $assignment->legalCase ? $assignment->legalCase->case_title : 'N/A',
                ucfirst($assignment->priority),
                ucfirst(str_replace('_', ' ', $assignment->status)),
                $assignment->due_datetime ? $assignment->due_datetime->format('M d, Y H:i') : 'N/A',
                $assignment->progress_percentage,
                $assignment->estimated_hours,
                $assignment->created_at ? $assignment->created_at->format('M d, Y') : 'N/A',
                $assignment->assigned_date ? $assignment->assigned_date->format('M d, Y') : 'N/A',
                $assignment->description,
                $assignment->instructions
            ];
        }

        $content = implode("\n", array_map(function($row) {
            return implode(',', array_map(function($cell) {
                return '"' . str_replace('"', '""', $cell) . '"';
            }, $row));
        }, array_merge([$headers], $rows)));

        return response($content)
            ->header('Content-Type', 'text/csv')
            ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
    }

    // Additional assignment methods to match routes/task-management.php
    public function assignmentAdd()
    {
        return $this->createAssignment();
    }

    public function assignmentList(Request $request)
    {
        return $this->listAssignments($request);
    }

    public function assignmentShow($id)
    {
        return $this->showAssignment($id);
    }

    public function assignmentEdit($id)
    {
        return $this->editAssignment($id);
    }

    public function assignmentUpdate(Request $request, $id)
    {
        return $this->updateAssignment($request, $id);
    }

    public function assignmentDelete($id)
    {
        return $this->deleteAssignment($id);
    }

    public function assignmentStore(Request $request)
    {
        return $this->storeAssignment($request);
    }

    // Court Details Options Page
    public function courtDetails()
    {
        return view('modules.edit.court-details');
    }

    public function editClientReceipt($id)
    {
        $receipt = \App\Models\ClientReceipt::with('lines')->findOrFail($id);
        $clients = \App\Models\Client::all();
        $accounts = \App\Models\MainAccount::with('accountType')->get();
        $glAccounts = \App\Models\MainAccount::with('accountType')->get();
        $taxes = \App\Models\Tax::all();
        $currencies = \App\Models\Currency::all();
        $paymentMethods = ['Cash', 'Bank Transfer', 'Cheque', 'Credit Card'];

        return view('modules.processing.client-receipt', compact('clients', 'accounts', 'glAccounts', 'taxes', 'currencies', 'paymentMethods', 'receipt'));
    }

    public function getCustomerInvoices($id)
    {
        $invoices = \App\Models\ClientInvoice::where('client_id', $id)
            ->select('id', 'invoice_number', 'total', 'balance_due')
            ->get();

        return response()->json($invoices);
    }

    public function tariff()
    {
        $tariffs = \App\Models\Tariff::all();
        $lawyers = \App\Models\Lawyer::all();
        $lawyerTiers = \App\Models\LawyerTier::all();
        $lawyerGroups = \App\Models\LawyerGroup::all();
        $lawyerTypes = \App\Models\LawyerType::all();
        $currencies = \App\Models\Currency::all();
        return view('modules.edit.tariff-list', compact('tariffs', 'lawyers', 'lawyerTiers', 'lawyerGroups', 'lawyerTypes', 'currencies'));
    }

    public function storeCashbook(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'cashbook_type' => 'nullable|string|in:bank,cash,petty_cash',
            'account_id' => 'nullable|integer|exists:main_accounts,id',
            'currency_id' => 'nullable|integer|exists:currencies,currency_id',
            'is_trust' => 'boolean',
            'opening_balance' => 'nullable|numeric|min:0',
            'opening_balance_date' => 'nullable|date',
            'bank_name' => 'nullable|string|max:255',
            'branch_name' => 'nullable|string|max:255',
            'account_number' => 'nullable|string|max:255',
            'swift_code' => 'nullable|string|max:255',
            'is_reconcilable' => 'boolean',
            'is_active' => 'boolean',
        ]);

        // Auto-generate code
        $lastCashbook = \App\Models\Cashbook::orderBy('id', 'desc')->first();
        $nextNumber = $lastCashbook ? intval(substr($lastCashbook->code, 2)) + 1 : 1;
        $validated['code'] = 'CB' . str_pad($nextNumber, 3, '0', STR_PAD_LEFT);

        \App\Models\Cashbook::create($validated);

        return redirect()->back()->with('success', 'Cashbook created successfully.');
    }

    public function editCashbook($id)
    {
        $cashbook = \App\Models\Cashbook::findOrFail($id);
        $currencies = \App\Models\Currency::all();
        $cashbooks = \App\Models\Cashbook::with(['account', 'currency'])->get();

        return view('modules.edit.cashbooks', compact('cashbook', 'currencies', 'cashbooks'));
    }

    public function updateCashbook(Request $request, $id)
    {
        $cashbook = \App\Models\Cashbook::findOrFail($id);

        $validated = $request->validate([
            'code' => 'required|string|max:255|unique:cashbooks,code,' . $cashbook->id,
            'name' => 'required|string|max:255',
            'cashbook_type' => 'nullable|string|in:bank,cash,petty_cash',
            'account_id' => 'nullable|integer|exists:main_accounts,id',
            'currency_id' => 'nullable|integer|exists:currencies,currency_id',
            'is_trust' => 'boolean',
            'opening_balance' => 'nullable|numeric|min:0',
            'opening_balance_date' => 'nullable|date',
            'bank_name' => 'nullable|string|max:255',
            'branch_name' => 'nullable|string|max:255',
            'account_number' => 'nullable|string|max:255',
            'swift_code' => 'nullable|string|max:255',
            'is_reconcilable' => 'boolean',
            'is_active' => 'boolean',
        ]);

        $cashbook->update($validated);

        return redirect()->back()->with('success', 'Cashbook updated successfully.');
    }

    public function destroyCashbook($id)
    {
        $cashbook = \App\Models\Cashbook::findOrFail($id);
        $cashbook->delete();

        return redirect()->back()->with('success', 'Cashbook deleted successfully.');
    }

    // Case Parties Methods
    public function addCaseParties()
    {
        $caseRoles = \App\Models\CaseRole::all();
        $legalCases = \App\Models\LegalCase::all();
        return view('modules.edit.case-parties-add', compact('caseRoles', 'legalCases'));
    }

    public function storeCaseParties(Request $request)
    {
        $validated = $request->validate([
            'case_id' => 'required|integer|exists:legal_cases,case_id',
            'party_name' => 'required|string|max:255',
            'party_type' => 'required|string|in:individual,company,organization',
            'contact_info' => 'nullable|string',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'role_id' => 'required|integer|exists:case_roles,role_id',
            'is_active' => 'boolean',
        ]);

        CaseParty::create($validated);

        return redirect()->route('modules.edit.view-parties')->with('success', 'Case party added successfully.');
    }

    public function listCaseParties()
    {
        $caseParties = CaseParty::with(['legalCase', 'caseRole'])->paginate(10);
        return view('modules.edit.case-parties-list', compact('caseParties'));
    }

    public function editCaseParties($id)
    {
        $caseParty = CaseParty::findOrFail($id);
        $caseRoles = CaseRole::all();
        $legalCases = LegalCase::all();
        return view('modules.edit.case-parties-edit', compact('caseParty', 'caseRoles', 'legalCases'));
    }

    public function updateCaseParties(Request $request, $id)
    {
        $caseParty = CaseParty::findOrFail($id);

        $validated = $request->validate([
            'case_id' => 'required|integer|exists:legal_cases,case_id',
            'party_name' => 'required|string|max:255',
            'party_type' => 'required|string|in:individual,company,organization',
            'contact_info' => 'nullable|string',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'role_id' => 'required|integer|exists:case_roles,role_id',
            'is_active' => 'boolean',
        ]);

        $caseParty->update($validated);

        return redirect()->route('modules.edit.view-parties')->with('success', 'Case party updated successfully.');
    }

    public function destroyCaseParties($id)
    {
        $caseParty = CaseParty::findOrFail($id);
        $caseParty->delete();

        return redirect()->route('modules.edit.view-parties')->with('success', 'Case party deleted successfully.');
    }

    public function lawyerRoles()
    {
        return view('modules.edit.lawyer-roles');
    }

    public function createLawyerRole()
    {
        return view('modules.edit.lawyer-roles-add');
    }

    public function viewLawyerRoles()
    {
        $lawyerRoles = \App\Models\LawyerRole::orderBy('sort_order')->orderBy('name')->paginate(15);
        return view('modules.edit.lawyer-roles-view', compact('lawyerRoles'));
    }

    public function storeLawyerRole(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string|max:255|unique:lawyer_roles,code|regex:/^[a-zA-Z_]+$/',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'side' => 'required|in:plaintiff,defendant,prosecution,defense,neutral',
            'is_primary' => 'boolean',
            'requires_bar_status' => 'boolean',
            'jurisdiction_scope' => 'in:US,UK,GLOBAL',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        // Set defaults
        $validated['is_primary'] = $request->has('is_primary');
        $validated['requires_bar_status'] = $request->has('requires_bar_status') ? true : false;
        $validated['jurisdiction_scope'] = $validated['jurisdiction_scope'] ?? 'GLOBAL';
        $validated['is_active'] = $request->has('is_active') ? true : false;
        $validated['sort_order'] = $validated['sort_order'] ?? 0;

        // Assuming you have a LawyerRole model
        \App\Models\LawyerRole::create($validated);

        return redirect()->route('modules.edit.lawyer-roles.view')->with('success', 'Lawyer role created successfully.');
    }

    public function editLawyerRole($id)
    {
        $lawyerRole = \App\Models\LawyerRole::findOrFail($id);
        return view('modules.edit.lawyer-roles-edit', compact('lawyerRole'));
    }

    public function updateLawyerRole(Request $request, $id)
    {
        $lawyerRole = \App\Models\LawyerRole::findOrFail($id);

        $validated = $request->validate([
            'code' => 'required|string|max:255|unique:lawyer_roles,code,' . $lawyerRole->lawyer_role_id . ',lawyer_role_id|regex:/^[a-zA-Z_]+$/',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'side' => 'required|in:plaintiff,defendant,prosecution,defense,neutral',
            'is_primary' => 'boolean',
            'requires_bar_status' => 'boolean',
            'jurisdiction_scope' => 'in:US,UK,GLOBAL',
            'is_active' => 'boolean',
            'sort_order' => 'nullable|integer|min:0',
        ]);

        // Set defaults
        $validated['is_primary'] = $request->has('is_primary');
        $validated['requires_bar_status'] = $request->has('requires_bar_status') ? true : false;
        $validated['jurisdiction_scope'] = $validated['jurisdiction_scope'] ?? 'GLOBAL';
        $validated['is_active'] = $request->has('is_active') ? true : false;
        $validated['sort_order'] = $validated['sort_order'] ?? 0;

        $lawyerRole->update($validated);

        return redirect()->route('modules.edit.lawyer-roles.view')->with('success', 'Lawyer role updated successfully.');
    }

    public function destroyLawyerRole($id)
    {
        $lawyerRole = \App\Models\LawyerRole::findOrFail($id);
        $lawyerRole->delete();

        return redirect()->route('modules.edit.lawyer-roles.view')->with('success', 'Lawyer role deleted successfully.');
    }
}
