<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Tax;
use App\Models\Currency;
use PDF;

class ReportController extends Controller
{
    public function chartOfAccounts()
    {
        // Fetch all main accounts with their account types
        $accounts = \App\Models\MainAccount::with('accountType')->get();

        return view('modules.reports.chart-of-accounts', compact('accounts'));
    }

    public function generalLedger()
    {
        // Fetch real GL accounts from database
        $glAccounts = \App\Models\MainAccount::with('accountType')->get()->map(function ($account) {
            // Calculate debit, credit, and balance for each account from GeneralJournal transactions
            $transactions = \App\Models\GeneralJournal::where('account_id', $account->id)->get();

            $debit = $transactions->where('tr_code', 'DR')->sum('amount');
            $credit = $transactions->where('tr_code', 'CR')->sum('amount');
            $balance = $debit - $credit;

            // Get last transaction date
            $lastTransaction = $transactions->sortByDesc('date')->first();
            $lastTransactionDate = $lastTransaction ? $lastTransaction->date : null;

            // Get currency (assuming default or from account)
            $currency = $account->currency ?? 'USD';

            return [
                'glcode' => $account->code,
                'name' => $account->name,
                'description' => $account->description ?? '',
                'accounttype' => $account->accountType->name ?? 'Unknown',
                'debit' => $debit,
                'credit' => $credit,
                'balance' => $balance,
                'currency' => $currency,
                'last_transaction_date' => $lastTransactionDate,
            ];
        });

        return view('modules.reports.general-ledger', compact('glAccounts'));
    }

    public function downloadGeneralLedgerPdf(Request $request)
    {
        $glAccounts = \App\Models\MainAccount::with('accountType')->get()->map(function ($account) {
            $transactions = \App\Models\GeneralJournal::where('account_id', $account->id)->get();

            $debit = $transactions->where('tr_code', 'DR')->sum('amount');
            $credit = $transactions->where('tr_code', 'CR')->sum('amount');
            $balance = $debit - $credit;

            $lastTransaction = $transactions->sortByDesc('date')->first();
            $lastTransactionDate = $lastTransaction ? $lastTransaction->date : null;

            $currency = $account->currency ?? 'USD';

            return [
                'glcode' => $account->code,
                'name' => $account->name,
                'description' => $account->description ?? '',
                'accounttype' => $account->accountType->name ?? 'Unknown',
                'debit' => $debit,
                'credit' => $credit,
                'balance' => $balance,
                'currency' => $currency,
                'last_transaction_date' => $lastTransactionDate,
            ];
        });

        $pdf = \PDF::loadView('modules.reports.general-ledger-pdf', compact('glAccounts'));
        return $pdf->download('general_ledger_report.pdf');
    }

    public function glAccountDetail($glcode)
    {
        // Find the GL account by code
        $account = \App\Models\MainAccount::where('code', $glcode)->first();

        if (!$account) {
            abort(404, 'GL Account not found');
        }

        // Fetch real transactions from GeneralJournal for this account
        $journalEntries = \App\Models\GeneralJournal::where('account_id', $account->id)
            ->with('currency')
            ->orderBy('date')
            ->get();

        // Calculate running balance for each transaction
        $runningBalance = 0;
        $transactions = $journalEntries->map(function ($entry) use (&$runningBalance) {
            if ($entry->tr_code === 'DR') {
                $runningBalance += $entry->amount;
            } else {
                $runningBalance -= $entry->amount;
            }

            return [
                'date' => $entry->date,
                'reference' => $entry->id, // Using ID as reference, could be customized
                'description' => $entry->narration,
                'trcode' => $entry->tr_code,
                'amount' => $entry->amount,
                'converted_amount' => $entry->amount, // Assuming same currency for now
                'balance_after' => $runningBalance,
            ];
        });

        return view('modules.reports.gl-account-detail', compact('glcode', 'transactions'));
    }

    public function clientStatement()
    {
        return view('modules.reports.placeholder', ['report' => 'Client Statement']);
    }

    public function caseStatus()
    {
        $cases = \App\Models\LegalCase::with(['lawyer', 'caseCategory', 'caseStatus'])->get();

        return view('case-status', compact('cases'));
    }

    public function lawyerStatement()
    {
        return view('modules.reports.lawyer-statement');
    }

    public function supplierStatement()
    {
        // Fetch suppliers, currencies, and initial data for the report
        $suppliers = \App\Models\Supplier::all();
        $currencies = \App\Models\Currency::all();

        // Initialize variables for view
        $supplierId = request()->input('supplier_account', 'all');
        $dateFrom = request()->input('date_from', date('Y-m-d'));
        $dateTo = request()->input('date_to', date('Y-m-d'));
        $currencyCode = request()->input('currency', '');

        // Placeholder for statements data
        $allStatements = [];
        $statementData = [];

        // Logic to fetch statement data based on filters can be added here
        // For now, just pass empty data to view

        return view('modules.reports.supplier-statement', compact(
            'suppliers',
            'currencies',
            'supplierId',
            'dateFrom',
            'dateTo',
            'currencyCode',
            'allStatements',
            'statementData'
        ));
    }

    public function cashBook()
    {
        // Fetch cashbooks for the dropdown
        $cashbooks = collect([
            (object)['id' => 1, 'code' => 'CB001', 'description' => 'Main Cash Book'],
            (object)['id' => 2, 'code' => 'CB002', 'description' => 'Petty Cash Book'],
        ]);

        $currencies = Currency::all();

        return view('modules.reports.cashbook', compact('cashbooks', 'currencies'));
    }

    public function taxReports()
    {
        $taxes = Tax::all();
        $currencies = Currency::all();
        $currnames = $currencies->pluck('currency_name', 'currency_id')->toArray();

        return view('tax-reports', compact('taxes', 'currencies', 'currnames'));
    }

    public function generateTaxReport(Request $request)
    {
        $request->validate([
            'tax' => 'required',
            'pfrom' => 'required|date',
            'pto' => 'required|date|after_or_equal:pfrom',
            'curr' => 'required|exists:currencies,id',
            'from' => 'required|date',
            'to' => 'required|date|after_or_equal:from',
        ]);

        $taxes = Tax::all();
        $taxId = $request->input('tax');
        $pfrom = $request->input('pfrom');
        $pto = $request->input('pto');
        $from = $request->input('from');
        $to = $request->input('to');
        $currencyId = $request->input('curr');

        $currency = Currency::find($currencyId);
        $currnames = $currency->currency_name;

        if ($taxId == 'All') {
            $taxRates = $taxes->pluck('tax_rate')->toArray();
        } else {
            $tax = $taxes->find($taxId);
            $taxRates = [$tax->tax_rate];
        }

        // Fetch output tax transactions (client invoices)
        $outputTransactions = \App\Models\ClientInvoice::with('customer')
            ->whereIn('tax_rate', $taxRates)
            ->whereBetween('invoice_date', [$from, $to])
            ->where('currency_id', $currencyId)
            ->get();

        // Fetch input tax transactions (supplier invoices)
        $inputTransactions = \App\Models\SupplierInvoice::with('supplier', 'tax')
            ->whereHas('tax', function($q) use ($taxRates) {
                $q->whereIn('rate', $taxRates);
            })
            ->whereBetween('invoice_date', [$from, $to])
            ->where('currency_id', $currencyId)
            ->get();

        // Calculate output totals
        $outputTotals = ['invoice' => 0, 'tax' => 0, 'incl' => 0];
        foreach ($outputTransactions as $transaction) {
            $invoiceAmount = $transaction->price * $transaction->quantity;
            $taxAmount = $invoiceAmount * ($transaction->tax_rate / 100);
            $outputTotals['invoice'] += $invoiceAmount;
            $outputTotals['tax'] += $taxAmount;
            $outputTotals['incl'] += $invoiceAmount + $taxAmount;
        }

        // Calculate input totals
        $inputTotals = ['invoice' => 0, 'tax' => 0, 'incl' => 0];
        foreach ($inputTransactions as $transaction) {
            $invoiceAmount = $transaction->price * $transaction->quantity;
            $taxAmount = $invoiceAmount * ($transaction->tax_rate / 100);
            $inputTotals['invoice'] += $invoiceAmount;
            $inputTotals['tax'] += $taxAmount;
            $inputTotals['incl'] += $invoiceAmount + $taxAmount;
        }

        $currencies = Currency::all();

        return view('tax-reports', compact(
            'taxes', 'currencies', 'currnames', 'outputTransactions', 'inputTransactions',
            'outputTotals', 'inputTotals', 'taxId', 'pfrom', 'pto', 'from', 'to', 'currencyId'
        ));
    }

    public function downloadTaxReportPdf(Request $request)
    {
        $request->validate([
            'taxId' => 'required',
            'pfrom' => 'required|date',
            'pto' => 'required|date|after_or_equal:pfrom',
            'from' => 'required|date',
            'to' => 'required|date|after_or_equal:from',
            'currencyId' => 'required|exists:currencies,id',
        ]);

        $taxes = Tax::all();
        $taxId = $request->input('taxId');
        $pfrom = $request->input('pfrom');
        $pto = $request->input('pto');
        $from = $request->input('from');
        $to = $request->input('to');
        $currencyId = $request->input('currencyId');

        $currency = Currency::find($currencyId);
        $currnames = $currency->name;

        if ($taxId == 'All') {
            $taxRates = $taxes->pluck('tax_rate')->toArray();
        } else {
            $tax = $taxes->find($taxId);
            $taxRates = [$tax->tax_rate];
        }

        // Fetch output tax transactions (client invoices)
        $outputTransactions = \App\Models\ClientInvoice::with('customer')
            ->whereIn('tax_rate', $taxRates)
            ->whereBetween('invoice_date', [$from, $to])
            ->where('currency_id', $currencyId)
            ->get();

        // Fetch input tax transactions (supplier invoices)
        $inputTransactions = \App\Models\SupplierInvoice::with('supplier')
            ->whereIn('tax_rate', $taxRates)
            ->whereBetween('invoice_date', [$from, $to])
            ->where('currency_id', $currencyId)
            ->get();

        // Calculate output totals
        $outputTotals = ['invoice' => 0, 'tax' => 0, 'incl' => 0];
        foreach ($outputTransactions as $transaction) {
            $invoiceAmount = $transaction->price * $transaction->quantity;
            $taxAmount = $invoiceAmount * ($transaction->tax_rate / 100);
            $outputTotals['invoice'] += $invoiceAmount;
            $outputTotals['tax'] += $taxAmount;
            $outputTotals['incl'] += $invoiceAmount + $taxAmount;
        }

        // Calculate input totals
        $inputTotals = ['invoice' => 0, 'tax' => 0, 'incl' => 0];
        foreach ($inputTransactions as $transaction) {
            $invoiceAmount = $transaction->price * $transaction->quantity;
            $taxAmount = $invoiceAmount * ($transaction->tax_rate / 100);
            $inputTotals['invoice'] += $invoiceAmount;
            $inputTotals['tax'] += $taxAmount;
            $inputTotals['incl'] += $invoiceAmount + $taxAmount;
        }

        $pdf = \PDF::loadView('reports.tax-reports-pdf', compact(
            'currnames', 'outputTransactions', 'inputTransactions',
            'outputTotals', 'inputTotals', 'from', 'to'
        ));

        return $pdf->download('tax_report_' . $from . '_to_' . $to . '.pdf');
    }

    public function generateCashbookReport(Request $request)
    {
        // Placeholder implementation - in a real app, this would process the form data and generate the report
        $cashbookId = $request->input('cashbook_account');
        $dateFrom = $request->input('date_from');
        $dateTo = $request->input('date_to');
        $currencyCode = $request->input('currency');

        // Placeholder data - in a real implementation, this would fetch from database
        $cashbooks = collect([
            (object)['id' => 1, 'code' => 'CB001', 'description' => 'Main Cash Book'],
            (object)['id' => 2, 'code' => 'CB002', 'description' => 'Petty Cash Book'],
        ]);

        $cashbook = $cashbooks->where('id', $cashbookId)->first();

        // Sample transaction data
        $transactionData = [
            [
                'date' => '2025-09-01',
                'description' => 'Opening Balance',
                'reference' => 'OP-001',
                'dr' => 0.00,
                'cr' => 0.00,
                'balance' => 1000.00,
            ],
            [
                'date' => '2025-09-02',
                'description' => 'Client Payment - Case #123',
                'reference' => 'RCPT-001',
                'dr' => 500.00,
                'cr' => 0.00,
                'balance' => 1500.00,
            ],
            [
                'date' => '2025-09-03',
                'description' => 'Office Supplies',
                'reference' => 'EXP-001',
                'dr' => 0.00,
                'cr' => 150.00,
                'balance' => 1350.00,
            ],
        ];

        $currencies = Currency::all();

        return view('modules.reports.cashbook', compact('cashbooks', 'currencies', 'transactionData', 'cashbook', 'cashbookId', 'dateFrom', 'dateTo', 'currencyCode'));
    }

    public function downloadCashbookPdf(Request $request)
    {
        // Placeholder implementation - in a real app, this would generate and download a PDF
        return redirect()->back()->with('success', 'PDF download not implemented yet.');
    }

    public function feeCollection()
    {
        return view('modules.reports.placeholder', ['report' => 'Fee Collection']);
    }



    public function downloadOutstandingFeesPdf(Request $request)
    {
        // Static data for design purposes (same as outstandingFees method)
        $outstandingInvoices = collect([
            [
                'client_name' => 'ABC Corporation',
                'invoice_number' => 'INV-001',
                'invoice_date' => '2024-09-01',
                'due_date' => '2024-09-30',
                'total_amount' => 5000.00,
                'paid_amount' => 2000.00,
                'balance_due' => 3000.00,
                'currency' => 'USD',
                'status' => 'partially_paid',
                'days_overdue' => 0,
            ],
            [
                'client_name' => 'XYZ Ltd',
                'invoice_number' => 'INV-002',
                'invoice_date' => '2024-08-15',
                'due_date' => '2024-09-15',
                'total_amount' => 7500.00,
                'paid_amount' => 0.00,
                'balance_due' => 7500.00,
                'currency' => 'USD',
                'status' => 'unpaid',
                'days_overdue' => 15,
            ],
            [
                'client_name' => 'Global Services Inc',
                'invoice_number' => 'INV-003',
                'invoice_date' => '2024-07-20',
                'due_date' => '2024-08-20',
                'total_amount' => 3200.00,
                'paid_amount' => 3200.00,
                'balance_due' => 0.00,
                'currency' => 'USD',
                'status' => 'paid',
                'days_overdue' => 0,
            ],
        ]);

        $pdf = \PDF::loadView('modules.reports.outstanding-fees-pdf', compact('outstandingInvoices'));
        return $pdf->download('outstanding_fees_report.pdf');
    }

    public function totalFees()
    {
        return view('modules.reports.total-fees');
    }

    public function incomeStatements()
    {
        $currencies = Currency::all();
        $currnames = $currencies->pluck('name', 'id')->toArray();

        return view('modules.reports.income-statements', compact('currencies', 'currnames'));
    }

    public function generateIncomeStatements(Request $request)
    {
        // Placeholder implementation - in a real app, this would process the form data and generate the report
        return redirect()->back()->with('success', 'Income statements report generated successfully.');
    }

    public function downloadIncomeStatementsPdf(Request $request)
    {
        // Placeholder implementation - in a real app, this would generate and download a PDF
        return redirect()->back()->with('success', 'PDF download not implemented yet.');
    }

    public function balanceSheet()
    {
        $currencies = Currency::all();
        $currnames = $currencies->pluck('name', 'id')->toArray();

        return view('balance-sheet', compact('currencies', 'currnames'));
    }

    public function generateBalanceSheet(Request $request)
    {
        // Placeholder implementation - in a real app, this would process the form data and generate the report
        return redirect()->back()->with('success', 'Balance sheet generated successfully.');
    }

    public function downloadBalanceSheetPdf(Request $request)
    {
        // Placeholder implementation - in a real app, this would generate and download a PDF
        return redirect()->back()->with('success', 'PDF download not implemented yet.');
    }

    public function generateCaseStatus(Request $request)
    {
        $request->validate([
            'case_id' => 'required|exists:legal_cases,id',
        ]);

        // Get date filters
        $dateFrom = $request->input('date_from');
        $dateTo = $request->input('date_to');

        // Filter cases by date range if provided
        $casesQuery = \App\Models\LegalCase::with(['lawyer', 'caseCategory', 'caseStatus', 'caseReason', 'judgement']);
        if ($dateFrom && $dateTo) {
            $casesQuery->whereBetween('created_at', [$dateFrom, $dateTo]);
        }
        $cases = $casesQuery->get();

        $selectedCase = \App\Models\LegalCase::with(['lawyer', 'caseCategory', 'caseStatus', 'caseReason', 'judgement'])
            ->find($request->case_id);

        if (!$selectedCase) {
            return redirect()->back()->with('error', 'Case not found.');
        }

        // Check if selected case is within date range
        if ($dateFrom && $dateTo && ($selectedCase->created_at < $dateFrom || $selectedCase->created_at > $dateTo)) {
            return redirect()->back()->with('error', 'Selected case is not within the specified date range.');
        }

        return view('case-status', compact('cases', 'selectedCase'));
    }

    public function downloadCaseStatusPdf(Request $request)
    {
        $request->validate([
            'case_id' => 'required|exists:legal_cases,id',
        ]);

        $selectedCase = \App\Models\LegalCase::with(['lawyer', 'caseCategory', 'caseStatus', 'caseReason', 'judgement'])
            ->find($request->case_id);

        if (!$selectedCase) {
            return redirect()->back()->with('error', 'Case not found.');
        }

        $pdf = \PDF::loadView('modules.reports.case-status-pdf', compact('selectedCase'));
        return $pdf->download('case_status_report_' . $selectedCase->case_id . '.pdf');
    }

    public function trialBalance(Request $request)
    {
        $currencies = Currency::all();
        $currnames = $currencies->pluck('name', 'id')->toArray();

        $pfrom = $request->input('pfrom');
        $pto = $request->input('pto');
        $currencyId = $request->input('curr');

        if ($pfrom && $pto && $currencyId) {
            // Fetch GL accounts and calculate trial balance
            $glAccounts = \App\Models\MainAccount::with('accountType')->get()->map(function ($account) use ($pfrom, $pto) {
                // Calculate debit, credit, and balance for each account from GeneralJournal transactions within the period
                $transactions = \App\Models\GeneralJournal::where('account_id', $account->id)
                    ->whereBetween('date', [$pfrom, $pto])
                    ->get();

                $debit = $transactions->where('tr_code', 'DR')->sum('amount');
                $credit = $transactions->where('tr_code', 'CR')->sum('amount');
                $balance = $debit - $credit;

                return [
                    'code' => $account->code,
                    'name' => $account->name,
                    'debit' => $debit,
                    'credit' => $credit,
                    'balance' => $balance,
                ];
            });

            $totalDebit = $glAccounts->sum('debit');
            $totalCredit = $glAccounts->sum('credit');
            $totalBalance = $glAccounts->sum('balance');

            return view('modules.reports.trial-balance', compact('currencies', 'currnames', 'glAccounts', 'pfrom', 'pto', 'currencyId', 'totalDebit', 'totalCredit', 'totalBalance'));
        }

        return view('modules.reports.trial-balance', compact('currencies', 'currnames'));
    }

    public function journals()
    {
        return view('modules.reports.journals');
    }

    public function generalJournalReport(Request $request)
    {
        $currencies = Currency::all();
        $currnames = $currencies->pluck('name', 'id')->toArray();

        $pfrom = $request->input('pfrom');
        $pto = $request->input('pto');
        $currencyId = $request->input('curr');

        if ($pfrom && $pto && $currencyId) {
            // Fetch general journal entries within the period
            $journalEntries = \App\Models\GeneralJournal::with(['account', 'currency'])
                ->whereBetween('date', [$pfrom, $pto])
                ->orderBy('date')
                ->get();

            return view('modules.reports.general-journal', compact('currencies', 'currnames', 'journalEntries', 'pfrom', 'pto', 'currencyId'));
        }

        return view('modules.reports.general-journal', compact('currencies', 'currnames'));
    }

    public function customerJournalReport(Request $request)
    {
        $currencies = Currency::all();
        $currnames = $currencies->pluck('name', 'id')->toArray();

        $pfrom = $request->input('pfrom');
        $pto = $request->input('pto');
        $currencyId = $request->input('curr');

        if ($pfrom && $pto && $currencyId) {
            // Fetch customer journal entries within the period
            $journalEntries = \App\Models\ClientJournal::with(['clientAccount', 'currency'])
                ->whereBetween('date', [$pfrom, $pto])
                ->orderBy('date')
                ->get();

            return view('modules.reports.customer-journal', compact('currencies', 'currnames', 'journalEntries', 'pfrom', 'pto', 'currencyId'));
        }

        return view('modules.reports.customer-journal', compact('currencies', 'currnames'));
    }

    public function supplierJournalReport(Request $request)
    {
        $currencies = Currency::all();
        $currnames = $currencies->pluck('name', 'id')->toArray();

        $pfrom = $request->input('pfrom');
        $pto = $request->input('pto');
        $currencyId = $request->input('curr');

        if ($pfrom && $pto && $currencyId) {
            // Fetch supplier journal entries within the period (assuming supplier journals are in GeneralJournal or separate model)
            // For now, using GeneralJournal as placeholder
            $journalEntries = \App\Models\GeneralJournal::with(['account', 'currency'])
                ->whereBetween('date', [$pfrom, $pto])
                ->orderBy('date')
                ->get();

            return view('modules.reports.supplier-journal', compact('currencies', 'currnames', 'journalEntries', 'pfrom', 'pto', 'currencyId'));
        }

        return view('modules.reports.supplier-journal', compact('currencies', 'currnames'));
    }

    public function courtFilings()
    {
        return view('modules.reports.court-filings');
    }

    public function generateCourtFilings(Request $request)
    {
        $request->validate([
            'pfrom' => 'required|date',
            'pto' => 'required|date|after_or_equal:pfrom',
        ]);

        $pfrom = $request->input('pfrom');
        $pto = $request->input('pto');

        // Fetch court filings within the date range
        $courtFilings = \App\Models\CourtFiling::with(['case', 'lawyer', 'courtArea', 'courtType', 'judge'])
            ->whereBetween('filing_date', [$pfrom, $pto])
            ->orderBy('filing_date')
            ->get();

        return view('modules.reports.court-filings', compact('courtFilings', 'pfrom', 'pto'));
    }

    public function downloadCourtFilingsPdf(Request $request)
    {
        $request->validate([
            'pfrom' => 'required|date',
            'pto' => 'required|date|after_or_equal:pfrom',
        ]);

        $pfrom = $request->input('pfrom');
        $pto = $request->input('pto');

        // Fetch court filings within the date range
        $courtFilings = \App\Models\CourtFiling::with(['case', 'lawyer', 'courtArea', 'courtType', 'judge'])
            ->whereBetween('filing_date', [$pfrom, $pto])
            ->orderBy('filing_date')
            ->get();

        $pdf = \PDF::loadView('modules.reports.court-filings-pdf', compact('courtFilings', 'pfrom', 'pto'));
        return $pdf->download('court_filings_report_' . $pfrom . '_to_' . $pto . '.pdf');
    }
}
