<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

class SupplierInvoiceController extends Controller
{
    public function supplierInvoice()
    {
        $suppliers = \App\Models\Supplier::all();
        $accounts = \App\Models\MainAccount::with('accountType')->get();
        $glAccounts = \App\Models\MainAccount::with('accountType')->get();
        $taxes = \App\Models\Tax::all();
        $currencies = \App\Models\Currency::all();
        $supplierTransactions = \App\Models\SupplierInvoice::with(['supplier', 'currency'])->get();
        $homeCurrency = \App\Models\HomeCurrency::with('currency')->first();

        return view('modules.processing.supplier-invoice', compact('suppliers', 'accounts', 'glAccounts', 'taxes', 'currencies', 'supplierTransactions', 'homeCurrency'));
    }

    public function storeSupplierInvoice(Request $request)
    {
        $validated = $request->validate([
            'supplier_id' => 'required|integer|exists:suppliers,id',
            'invoice_number' => 'required|string|max:255',
            'invoice_date' => 'required|date',
            'due_date' => 'nullable|date',
            'amount' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'account' => 'nullable|string|max:255',
            'narration' => 'nullable|string',
            'reference' => 'nullable|string|max:255',
            'gl_account' => 'nullable|string|max:255',
            'tax_id' => 'nullable|integer|exists:taxes,id',
            'discount' => 'nullable|numeric|min:0|max:100',
            'price' => 'required|numeric|min:0',
            'quantity' => 'nullable|integer|min:1',
            'currency_id' => 'nullable|integer|exists:currencies,id',
            'operation_cost' => 'nullable|boolean',
            'tax_amount' => 'nullable|numeric|min:0',
            'total' => 'required|numeric|min:0',
        ]);

        \App\Models\SupplierInvoice::create($validated);

        return redirect()->back()->with('success', 'Supplier invoice created successfully.');
    }

    public function updateSupplierInvoice(Request $request, $id)
    {
        $invoice = \App\Models\SupplierInvoice::findOrFail($id);

        $validated = $request->validate([
            'supplier_id' => 'required|integer|exists:suppliers,id',
            'invoice_number' => 'required|string|max:255',
            'invoice_date' => 'required|date',
            'due_date' => 'nullable|date',
            'amount' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'account' => 'nullable|string|max:255',
            'narration' => 'nullable|string',
            'reference' => 'nullable|string|max:255',
            'gl_account' => 'nullable|string|max:255',
            'tax_id' => 'nullable|integer|exists:taxes,id',
            'discount' => 'nullable|numeric|min:0|max:100',
            'price' => 'required|numeric|min:0',
            'quantity' => 'nullable|integer|min:1',
            'currency_id' => 'nullable|integer|exists:currencies,id',
            'operation_cost' => 'nullable|boolean',
            'tax_amount' => 'nullable|numeric|min:0',
            'total' => 'required|numeric|min:0',
        ]);

        $invoice->update($validated);

        return redirect()->back()->with('success', 'Supplier invoice updated successfully.');
    }

    public function destroySupplierInvoice($id)
    {
        $invoice = \App\Models\SupplierInvoice::findOrFail($id);
        $invoice->delete();

        return redirect()->back()->with('success', 'Supplier invoice deleted successfully.');
    }

    public function processSupplierInvoice($invoice = null)
    {
        // Process supplier invoice(s)
        if ($invoice) {
            $invoice = \App\Models\SupplierInvoice::findOrFail($invoice);
            $invoice->update(['status' => 'Processed']);
            $message = 'Supplier invoice processed successfully.';
        } else {
            $invoices = \App\Models\SupplierInvoice::where('status', '!=', 'Processed')->get();
            foreach ($invoices as $inv) {
                $inv->update(['status' => 'Processed']);
            }
            $message = 'All supplier invoices processed successfully.';
        }

        return redirect()->back()->with('success', $message);
    }
}
