<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Models\Session;
use Symfony\Component\HttpFoundation\Response;

class Authenticate
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $sessionToken = $request->cookie('session_token');

        if (!$sessionToken) {
            Log::warning('Access denied: No session token', [
                'ip' => $request->ip(),
                'url' => $request->fullUrl(),
                'user_agent' => $request->userAgent()
            ]);
            return redirect()->route('login')->withErrors(['auth' => 'Please log in to access this page.']);
        }

        $session = Session::where('session_token', $sessionToken)
            ->where('status', 'active')
            ->where('expires_at', '>', now())
            ->first();

        if (!$session) {
            Log::warning('Access denied: Invalid or expired session', [
                'session_token' => substr($sessionToken, 0, 10) . '...', // Log partial token for security
                'ip' => $request->ip(),
                'url' => $request->fullUrl(),
                'user_agent' => $request->userAgent()
            ]);

            // Clear invalid cookie
            return redirect()->route('login')
                ->cookie('session_token', '', -1)
                ->withErrors(['auth' => 'Your session has expired. Please log in again.']);
        }

        // Update last activity
        $session->update(['last_activity' => now()]);

        // Set the authenticated user for Laravel auth
        Auth::login($session->user);

        // Add user to request for convenience
        $request->merge(['authenticated_user' => $session->user]);

        return $next($request);
    }
}
