<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Hash;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password_hash',
        'role_id',
        'group_id',
        'tier_id',
        'status',
        'location',
        'permissions',
        'failed_login_attempts',
        'account_locked_until',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the password for authentication.
     *
     * @return string
     */
    public function getAuthPassword()
    {
        return $this->password_hash;
    }

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password_hash' => 'hashed',
            'permissions' => 'array',
            'locked_until' => 'datetime',
        ];
    }

    /**
     * Relationships
     */
    public function role()
    {
        // Changed to use UserRole model and user_role_id primary key
        return $this->belongsTo(UserRole::class, 'role_id', 'user_role_id');
    }

    public function group()
    {
        return $this->belongsTo(Group::class, 'group_id', 'user_group_id');
    }

    public function tier()
    {
        return $this->belongsTo(UserTier::class, 'tier_id');
    }

    public function phones()
    {
        return $this->hasMany(UserPhone::class, 'user_id');
    }

    public function userRoles()
    {
        return $this->hasMany(UserRole::class, 'user_id');
    }

    /*
    public function userGroups()
    {
        return $this->hasMany(UserGroup::class, 'user_id');
    }
    */

    /**
     * Check if the user account is locked
     */
    public function isLocked()
    {
        if (!$this->account_locked_until) {
            return false;
        }

        $lockedUntil = $this->account_locked_until;

        // Handle string datetime from database
        if (is_string($lockedUntil)) {
            $lockedUntil = \Carbon\Carbon::parse($lockedUntil);
        }

        return $lockedUntil->isFuture();
    }

    /**
     * Increment failed login attempts
     */
    public function incrementFailedAttempts()
    {
        $this->increment('failed_login_attempts');

        // Lock account after 5 failed attempts for 15 minutes
        if ($this->failed_login_attempts >= 5) {
            $this->locked_until = now()->addMinutes(15);
            $this->save();
        }
    }

    /**
     * Reset failed login attempts
     */
    public function resetFailedAttempts()
    {
        $this->update([
            'failed_login_attempts' => 0,
            'locked_until' => null,
        ]);
    }

    /**
     * Check if user has a specific permission
     */
    public function hasPermission($permission)
    {
        $permissions = $this->permissions ?? [];
        return in_array($permission, $permissions);
    }
}
