<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class GeminiService
{
    protected $apiKey;
    protected $baseUrl;
    protected $model;

    public function __construct()
    {
        $this->apiKey = config('services.gemini.api_key');
        $this->baseUrl = config('services.gemini.base_url', 'https://generativelanguage.googleapis.com/v1beta');
        $this->model = config('services.gemini.model', 'gemini-1.5-flash-latest');
    }

    public function generateText(string $prompt): array
    {
        // Create cache key from prompt hash
        $cacheKey = 'gemini_' . md5($prompt);

        // Check cache first
        if (Cache::has($cacheKey)) {
            return Cache::get($cacheKey);
        }

        try {
            $response = Http::timeout(30)->post("{$this->baseUrl}/models/{$this->model}:generateContent?key={$this->apiKey}", [
                'contents' => [
                    [
                        'parts' => [
                            [
                                'text' => $prompt
                            ]
                        ]
                    ]
                ],
                'generationConfig' => [
                    'temperature' => 0.7,
                    'topK' => 40,
                    'topP' => 0.95,
                    'maxOutputTokens' => 1024,
                ]
            ]);

            if ($response->successful()) {
                $data = $response->json();

                $result = [
                    'success' => true,
                    'text' => $data['candidates'][0]['content']['parts'][0]['text'] ?? 'No response generated',
                    'model' => $this->model,
                    'usage' => [
                        'promptTokenCount' => $data['usageMetadata']['promptTokenCount'] ?? 0,
                        'candidatesTokenCount' => $data['usageMetadata']['candidatesTokenCount'] ?? 0,
                        'totalTokens' => $data['usageMetadata']['totalTokenCount'] ?? 0
                    ]
                ];

                // Cache the result for 1 hour
                Cache::put($cacheKey, $result, 3600);

                return $result;
            } else {
                Log::error('Gemini API error: ' . $response->body());

                return [
                    'success' => false,
                    'error' => 'API request failed: ' . $response->status(),
                    'text' => ''
                ];
            }
        } catch (\Exception $e) {
            Log::error('Gemini service exception: ' . $e->getMessage());

            return [
                'success' => false,
                'error' => $e->getMessage(),
                'text' => ''
            ];
        }
    }
}
