<?php

namespace App\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class RateLimiterService
{
    private $config;

    public function __construct()
    {
        $this->config = config('services.gemini.rate_limits');
    }

    public function canMakeRequest(string $model): bool
    {
        $modelType = $this->getModelType($model);
        $limits = $this->config[$modelType] ?? $this->config['flash'];

        $minuteKey = "gemini:rate:{$modelType}:minute:" . floor(time() / 60);
        $secondKey = "gemini:rate:{$modelType}:second:" . time();

        $minuteCount = Cache::get($minuteKey, 0);
        $secondCount = Cache::get($secondKey, 0);

        if ($minuteCount >= $limits['requests_per_minute']) {
            Log::warning('Gemini minute rate limit exceeded', ['model' => $model]);
            return false;
        }

        if ($secondCount >= $limits['requests_per_second']) {
            Log::warning('Gemini second rate limit exceeded', ['model' => $model]);
            return false;
        }

        return true;
    }

    public function recordRequest(string $model): void
    {
        $modelType = $this->getModelType($model);

        $minuteKey = "gemini:rate:{$modelType}:minute:" . floor(time() / 60);
        $secondKey = "gemini:rate:{$modelType}:second:" . time();

        Cache::increment($minuteKey);
        Cache::increment($secondKey);

        // Set expiration
        Cache::expire($minuteKey, 120); // 2 minutes
        Cache::expire($secondKey, 2);   // 2 seconds
    }

    private function getModelType(string $model): string
    {
        if (str_contains($model, 'flash')) return 'flash';
        if (str_contains($model, 'pro')) return 'pro';
        return 'flash';
    }
}
