// Court Filings JavaScript - Add your JS code here
/**
 * Court Filing Form JavaScript
 * Handles form interactions, validation, and dynamic updates
 */

class CourtFilingForm {
    constructor(formId) {
        this.form = document.getElementById(formId);
        if (!this.form) return;
        
        this.init();
    }

    init() {
        this.setupFormDefaults();
        this.attachEventListeners();
        this.updateProgress();
        this.initializeStatusBadge();
    }

    // =======================
    // Setup & Initialization
    // =======================

    setupFormDefaults() {
        // Set today's date as default filing date
        const filingDate = document.getElementById('filing_date');
        if (filingDate && !filingDate.value) {
            filingDate.value = this.getTodayDate();
        }

        // Initialize character counters
        document.querySelectorAll('[maxlength]').forEach(element => {
            this.updateCharCounter(element);
        });
    }

    initializeStatusBadge() {
        const statusSelect = document.getElementById('filing_status_id');
        if (statusSelect?.value) {
            this.updateStatusBadge(statusSelect);
        }
    }

    // =======================
    // Event Listeners
    // =======================

    attachEventListeners() {
        // Progress tracking
        const requiredInputs = this.form.querySelectorAll('input[required], select[required], textarea[required]');
        requiredInputs.forEach(input => {
            input.addEventListener('input', () => this.updateProgress());
            input.addEventListener('change', () => this.updateProgress());
        });

        // Card collapse toggles
        document.querySelectorAll('[data-toggle-card]').forEach(toggle => {
            toggle.addEventListener('click', (e) => this.toggleCard(e.currentTarget));
        });

        // Character counters
        document.querySelectorAll('[maxlength]').forEach(element => {
            element.addEventListener('input', () => this.updateCharCounter(element));
        });

        // Deadline checker
        const dueDate = document.getElementById('due_date');
        if (dueDate) {
            dueDate.addEventListener('change', () => this.checkDeadline());
        }

        const filingDate = document.getElementById('filing_date');
        if (filingDate) {
            filingDate.addEventListener('change', () => this.checkDeadline());
        }

        // Status badge updater
        const statusSelect = document.getElementById('filing_status_id');
        if (statusSelect) {
            statusSelect.addEventListener('change', (e) => this.updateStatusBadge(e.target));
        }

        // File upload handler
        const fileInput = document.querySelector('[data-file-upload]');
        if (fileInput) {
            fileInput.addEventListener('change', (e) => this.handleFileUpload(e.target));
        }

        // Reset form button
        const resetBtn = document.querySelector('[data-reset-form]');
        if (resetBtn) {
            resetBtn.addEventListener('click', () => this.resetForm());
        }

        // Form submission
        this.form.addEventListener('submit', (e) => this.validateForm(e));
    }

    // =======================
    // Progress Tracking
    // =======================

    updateProgress() {
        const requiredInputs = this.form.querySelectorAll('input[required], select[required], textarea[required]');
        let completed = 0;

        requiredInputs.forEach(input => {
            if (input.value?.trim()) {
                completed++;
            }
        });

        const percentage = Math.round((completed / requiredInputs.length) * 100);
        const progressBar = document.getElementById('progress-bar');
        const progressText = document.getElementById('progress-percentage');

        if (progressBar) progressBar.style.width = `${percentage}%`;
        if (progressText) progressText.textContent = `${percentage}%`;
    }

    // =======================
    // Card Management
    // =======================

    toggleCard(header) {
        const card = header.closest('.form-card');
        if (card) {
            card.classList.toggle('collapsed');
        }
    }

    // =======================
    // Character Counter
    // =======================

    updateCharCounter(element) {
        const maxLength = parseInt(element.getAttribute('maxlength'));
        const currentLength = element.value.length;
        const counter = document.querySelector(`[data-field="${element.id}"]`);

        if (counter) {
            counter.textContent = `${currentLength} / ${maxLength}`;
            
            if (currentLength > maxLength * 0.9) {
                counter.classList.add('warning');
            } else {
                counter.classList.remove('warning');
            }
        }
    }

    // =======================
    // Deadline Checker
    // =======================

    checkDeadline() {
        const filingDate = document.getElementById('filing_date');
        const dueDate = document.getElementById('due_date');
        const warningDiv = document.getElementById('deadline-warning');

        if (!filingDate?.value || !dueDate?.value || !warningDiv) {
            warningDiv?.classList.add('hidden');
            return;
        }

        const filing = new Date(filingDate.value);
        const due = new Date(dueDate.value);

        if (due < filing) {
            warningDiv.className = 'warning-message error';
            warningDiv.textContent = '⚠️ Due date is before filing date!';
            warningDiv.classList.remove('hidden');
        } else {
            const diffDays = Math.ceil((due - filing) / (1000 * 60 * 60 * 24));
            
            if (diffDays <= 7 && diffDays >= 0) {
                warningDiv.className = 'warning-message';
                warningDiv.textContent = `⚠️ Deadline approaching in ${diffDays} day${diffDays !== 1 ? 's' : ''}!`;
                warningDiv.classList.remove('hidden');
            } else {
                warningDiv.classList.add('hidden');
            }
        }
    }

    // =======================
    // Status Badge
    // =======================

    updateStatusBadge(select) {
        const badgeContainer = document.getElementById('status-badge');
        if (!badgeContainer) return;

        const selectedOption = select.options[select.selectedIndex];
        const statusName = selectedOption?.text || '';

        const statusConfig = {
            'Draft': { class: 'draft', icon: '📝' },
            'Ready for Filing': { class: 'ready', icon: '📋' },
            'Filed': { class: 'filed', icon: '✅' },
            'Rejected': { class: 'rejected', icon: '❌' },
            'Awaiting Response': { class: 'awaiting', icon: '⏳' }
        };

        const config = statusConfig[statusName];

        if (config) {
            badgeContainer.innerHTML = `
                <div class="status-badge ${config.class}">
                    <span>${config.icon}</span>
                    ${statusName}
                </div>
            `;
        } else {
            badgeContainer.innerHTML = '';
        }
    }

    // =======================
    // File Upload
    // =======================

    handleFileUpload(input) {
        const fileInfo = document.getElementById('file-info');
        if (!fileInfo) return;

        const file = input.files?.[0];
        
        if (!file) {
            fileInfo.classList.add('hidden');
            return;
        }

        // Validation
        const allowedTypes = [
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
        ];
        const maxSize = 10 * 1024 * 1024; // 10MB

        // Check file type
        if (!allowedTypes.includes(file.type)) {
            this.showFileError(fileInfo, 'Invalid file type. Please upload PDF, DOC, or DOCX.');
            input.value = '';
            return;
        }

        // Check file size
        if (file.size > maxSize) {
            this.showFileError(fileInfo, 'File size exceeds 10MB limit.');
            input.value = '';
            return;
        }

        // Show success
        const fileSize = (file.size / 1024 / 1024).toFixed(2);
        fileInfo.className = 'file-info';
        fileInfo.textContent = `✅ ${file.name} (${fileSize} MB)`;
        fileInfo.classList.remove('hidden');
    }

    showFileError(element, message) {
        element.className = 'file-info error';
        element.textContent = `❌ ${message}`;
        element.classList.remove('hidden');
    }

    // =======================
    // Form Reset
    // =======================

    resetForm() {
        if (!confirm('Are you sure you want to reset the form? All entered data will be lost.')) {
            return;
        }

        this.form.reset();

        // Reset character counters
        document.querySelectorAll('.char-counter').forEach(counter => {
            const maxLength = counter.textContent.split(' / ')[1];
            counter.textContent = `0 / ${maxLength}`;
            counter.classList.remove('warning');
        });

        // Reset dynamic elements
        const statusBadge = document.getElementById('status-badge');
        const deadlineWarning = document.getElementById('deadline-warning');
        const fileInfo = document.getElementById('file-info');

        if (statusBadge) statusBadge.innerHTML = '';
        if (deadlineWarning) deadlineWarning.classList.add('hidden');
        if (fileInfo) fileInfo.classList.add('hidden');

        // Reset filing date to today
        const filingDate = document.getElementById('filing_date');
        if (filingDate) {
            filingDate.value = this.getTodayDate();
        }

        this.updateProgress();
    }

    // =======================
    // Form Validation
    // =======================

    validateForm(e) {
        const requiredInputs = this.form.querySelectorAll('input[required], select[required], textarea[required]');
        let isValid = true;
        let firstInvalid = null;

        requiredInputs.forEach(input => {
            if (!input.value?.trim()) {
                isValid = false;
                input.style.borderColor = '#ef4444';
                
                if (!firstInvalid) {
                    firstInvalid = input;
                }
            } else {
                input.style.borderColor = '';
            }
        });

        if (!isValid) {
            e.preventDefault();
            alert('Please fill in all required fields marked with *');
            
            if (firstInvalid) {
                firstInvalid.scrollIntoView({ behavior: 'smooth', block: 'center' });
                firstInvalid.focus();
            }
        }
    }

    // =======================
    // Utilities
    // =======================

    getTodayDate() {
        return new Date().toISOString().split('T')[0];
    }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    new CourtFilingForm('filing-form');
});