/**
 * Dashboard Charts Module
 * Handles all Chart.js visualizations with lazy loading and performance optimization
 */

const DashboardCharts = (function() {
    'use strict';

    // Chart color schemes
    const chartColors = {
        primary: 'rgba(14, 165, 233, 0.8)',
        primaryLight: 'rgba(14, 165, 233, 0.2)',
        success: 'rgba(34, 197, 94, 0.8)',
        successLight: 'rgba(34, 197, 94, 0.2)',
        warning: 'rgba(245, 158, 11, 0.8)',
        warningLight: 'rgba(245, 158, 11, 0.2)',
        danger: 'rgba(239, 68, 68, 0.8)',
        dangerLight: 'rgba(239, 68, 68, 0.2)',
        purple: 'rgba(139, 92, 246, 0.8)',
        purpleLight: 'rgba(139, 92, 246, 0.2)',
        cyan: 'rgba(6, 182, 212, 0.8)',
        pink: 'rgba(236, 72, 153, 0.8)',
        orange: 'rgba(249, 115, 22, 0.8)',
        teal: 'rgba(20, 184, 166, 0.8)'
    };

    // Common chart options
    const commonOptions = {
        responsive: true,
        maintainAspectRatio: false,
        animation: {
            duration: 2000,
            easing: 'easeInOutQuart'
        },
        plugins: {
            legend: {
                display: true,
                position: 'bottom',
                labels: {
                    padding: 15,
                    usePointStyle: true,
                    font: {
                        size: 12,
                        weight: '600'
                    }
                }
            },
            tooltip: {
                backgroundColor: 'rgba(30, 41, 59, 0.95)',
                padding: 12,
                titleFont: {
                    size: 14,
                    weight: '700'
                },
                bodyFont: {
                    size: 13
                },
                borderColor: 'rgba(148, 163, 184, 0.2)',
                borderWidth: 1
            }
        }
    };

    // Store chart instances for cleanup
    const chartInstances = {};

    /**
     * Initialize Revenue Trend Chart
     */
    function initRevenueChart() {
        const ctx = document.getElementById('revenueChart');
        if (!ctx) return;

        const chart = new Chart(ctx.getContext('2d'), {
            type: 'line',
            data: {
                labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun'],
                datasets: [
                    {
                        label: 'Revenue',
                        data: [385000, 420000, 398000, 455000, 472000, 485000],
                        borderColor: chartColors.primary,
                        backgroundColor: chartColors.primaryLight,
                        tension: 0.4,
                        fill: true,
                        pointRadius: 5,
                        pointHoverRadius: 7,
                        pointBackgroundColor: chartColors.primary,
                        pointBorderColor: '#ffffff',
                        pointBorderWidth: 2
                    },
                    {
                        label: 'Target',
                        data: [400000, 410000, 420000, 430000, 440000, 450000],
                        borderColor: chartColors.success,
                        backgroundColor: 'transparent',
                        borderDash: [10, 5],
                        tension: 0.4,
                        fill: false,
                        pointRadius: 4,
                        pointHoverRadius: 6
                    }
                ]
            },
            options: {
                ...commonOptions,
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: { 
                            color: 'rgba(0,0,0,0.05)',
                            drawBorder: false
                        },
                        ticks: {
                            callback: value => '$' + (value / 1000) + 'K'
                        }
                    },
                    x: { 
                        grid: { display: false }
                    }
                },
                interaction: { 
                    intersect: false, 
                    mode: 'index' 
                }
            }
        });

        chartInstances.revenue = chart;
    }

    /**
     * Initialize Practice Area Distribution Chart
     */
    function initPracticeAreaChart() {
        const ctx = document.getElementById('practiceAreaChart');
        if (!ctx) return;

        const chart = new Chart(ctx.getContext('2d'), {
            type: 'pie',
            data: {
                labels: ['Corporate Law', 'Litigation', 'Real Estate', 'Family Law', 'IP Law'],
                datasets: [{
                    data: [35, 28, 18, 12, 7],
                    backgroundColor: [
                        chartColors.primary,
                        chartColors.success,
                        chartColors.warning,
                        chartColors.purple,
                        chartColors.danger
                    ],
                    borderWidth: 3,
                    borderColor: '#ffffff',
                    hoverOffset: 15
                }]
            },
            options: {
                ...commonOptions,
                plugins: {
                    ...commonOptions.plugins,
                    tooltip: {
                        ...commonOptions.plugins.tooltip,
                        callbacks: {
                            label: ctx => ctx.label + ': ' + ctx.parsed + '%'
                        }
                    }
                }
            }
        });

        chartInstances.practiceArea = chart;
    }

    /**
     * Initialize Attorney Utilization Chart
     */
    function initUtilizationChart() {
        const ctx = document.getElementById('utilizationChart');
        if (!ctx) return;

        const chart = new Chart(ctx.getContext('2d'), {
            type: 'bar',
            data: {
                labels: ['Week 1', 'Week 2', 'Week 3', 'Week 4'],
                datasets: [
                    {
                        label: 'Billable Hours',
                        data: [285, 310, 298, 354],
                        backgroundColor: chartColors.primary,
                        borderRadius: 8,
                        borderWidth: 0
                    },
                    {
                        label: 'Non-Billable Hours',
                        data: [95, 82, 88, 76],
                        backgroundColor: chartColors.warningLight,
                        borderRadius: 8,
                        borderWidth: 0
                    }
                ]
            },
            options: {
                ...commonOptions,
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: { 
                            color: 'rgba(0,0,0,0.05)',
                            drawBorder: false
                        },
                        stacked: true
                    },
                    x: {
                        grid: { display: false },
                        stacked: true
                    }
                }
            }
        });

        chartInstances.utilization = chart;
    }

    /**
     * Initialize AR Aging Chart
     */
    function initARAgingChart() {
        const ctx = document.getElementById('arAgingChart');
        if (!ctx) return;

        const chart = new Chart(ctx.getContext('2d'), {
            type: 'bar',
            data: {
                labels: ['0-30 Days', '31-60 Days', '61-90 Days', '90+ Days'],
                datasets: [{
                    label: 'Amount ($)',
                    data: [85000, 52000, 28500, 22000],
                    backgroundColor: [
                        chartColors.success,
                        chartColors.primary,
                        chartColors.warning,
                        chartColors.danger
                    ],
                    borderRadius: 8,
                    borderWidth: 0
                }]
            },
            options: {
                ...commonOptions,
                indexAxis: 'y',
                scales: {
                    x: {
                        beginAtZero: true,
                        grid: { 
                            color: 'rgba(0,0,0,0.05)',
                            drawBorder: false
                        },
                        ticks: {
                            callback: value => '$' + (value / 1000) + 'K'
                        }
                    },
                    y: { 
                        grid: { display: false }
                    }
                }
            }
        });

        chartInstances.arAging = chart;
    }

    /**
     * Initialize Matter Performance Chart
     */
    function initMatterPerformanceChart() {
        const ctx = document.getElementById('matterPerformanceChart');
        if (!ctx) return;

        const chart = new Chart(ctx.getContext('2d'), {
            type: 'bar',
            data: {
                labels: ['Matter A', 'Matter B', 'Matter C', 'Matter D', 'Matter E', 'Matter F'],
                datasets: [
                    {
                        label: 'Budget',
                        data: [45000, 62000, 38000, 55000, 42000, 58000],
                        backgroundColor: chartColors.primaryLight,
                        borderColor: chartColors.primary,
                        borderWidth: 2,
                        borderRadius: 6
                    },
                    {
                        label: 'Actual Spend',
                        data: [42500, 58000, 41000, 52000, 39000, 61000],
                        backgroundColor: chartColors.successLight,
                        borderColor: chartColors.success,
                        borderWidth: 2,
                        borderRadius: 6
                    }
                ]
            },
            options: {
                ...commonOptions,
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: { 
                            color: 'rgba(0,0,0,0.05)',
                            drawBorder: false
                        },
                        ticks: {
                            callback: value => '$' + (value / 1000) + 'K'
                        }
                    },
                    x: { 
                        grid: { display: false }
                    }
                }
            }
        });

        chartInstances.matterPerformance = chart;
    }

    /**
     * Initialize Time Tracking Chart
     */
    function initTimeTrackingChart(data) {
        const ctx = document.getElementById('timeTrackingChart');
        if (!ctx) return;

        const chart = new Chart(ctx.getContext('2d'), {
            type: 'line',
            data: {
                labels: ['Week 1', 'Week 2', 'Week 3', 'Week 4', 'Week 5', 'Week 6'],
                datasets: [{
                    label: 'Billable Hours',
                    data: data || [285, 310, 298, 354, 342, 368],
                    borderColor: chartColors.primary,
                    backgroundColor: chartColors.primaryLight,
                    tension: 0.4,
                    fill: true,
                    pointBackgroundColor: chartColors.primary,
                    pointBorderColor: '#ffffff',
                    pointBorderWidth: 3,
                    pointRadius: 6,
                    pointHoverRadius: 8
                }]
            },
            options: {
                ...commonOptions,
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: { 
                            color: 'rgba(0,0,0,0.05)',
                            drawBorder: false
                        },
                        ticks: {
                            callback: value => value + 'h'
                        }
                    },
                    x: { 
                        grid: { display: false }
                    }
                },
                interaction: { 
                    intersect: false, 
                    mode: 'index' 
                }
            }
        });

        chartInstances.timeTracking = chart;
    }

    /**
     * Initialize Financial Overview Chart
     */
    function initFinancialChart(data) {
        const ctx = document.getElementById('financialChart');
        if (!ctx) return;

        const financialData = data || {
            revenue: 485000,
            expenses: 312000,
            profit: 173000
        };

        const chart = new Chart(ctx.getContext('2d'), {
            type: 'doughnut',
            data: {
                labels: ['Revenue', 'Expenses', 'Profit'],
                datasets: [{
                    data: [
                        financialData.revenue,
                        financialData.expenses,
                        financialData.profit
                    ],
                    backgroundColor: [
                        chartColors.success,
                        chartColors.danger,
                        chartColors.primary
                    ],
                    borderWidth: 3,
                    borderColor: '#ffffff',
                    hoverOffset: 20
                }]
            },
            options: {
                ...commonOptions,
                cutout: '65%',
                plugins: {
                    ...commonOptions.plugins,
                    tooltip: {
                        ...commonOptions.plugins.tooltip,
                        callbacks: {
                            label: ctx => {
                                const label = ctx.label || '';
                                const value = ctx.parsed;
                                const total = ctx.dataset.data.reduce((a, b) => a + b, 0);
                                const percentage = Math.round((value / total) * 100);
                                return label + ': $' + value.toLocaleString() + ' (' + percentage + '%)';
                            }
                        }
                    }
                }
            }
        });

        chartInstances.financial = chart;
    }

    /**
     * Lazy load charts using Intersection Observer
     */
    function lazyLoadCharts(timeTrackingData, financialData) {
        const chartCards = document.querySelectorAll('.chart-card');
        
        const observer = new IntersectionObserver((entries) => {
            entries.forEach((entry, index) => {
                if (entry.isIntersecting) {
                    // Add visible class for animation
                    entry.target.classList.add('visible');
                    entry.target.style.transitionDelay = `${index * 0.1}s`;
                    
                    // Initialize chart based on canvas ID
                    const canvas = entry.target.querySelector('canvas');
                    if (!canvas) return;

                    const chartId = canvas.id;
                    
                    // Prevent re-initialization
                    if (chartInstances[chartId]) return;

                    // Initialize appropriate chart
                    switch(chartId) {
                        case 'revenueChart':
                            initRevenueChart();
                            break;
                        case 'practiceAreaChart':
                            initPracticeAreaChart();
                            break;
                        case 'utilizationChart':
                            initUtilizationChart();
                            break;
                        case 'arAgingChart':
                            initARAgingChart();
                            break;
                        case 'matterPerformanceChart':
                            initMatterPerformanceChart();
                            break;
                        case 'timeTrackingChart':
                            initTimeTrackingChart(timeTrackingData);
                            break;
                        case 'financialChart':
                            initFinancialChart(financialData);
                            break;
                    }
                    
                    // Stop observing after chart is loaded
                    observer.unobserve(entry.target);
                }
            });
        }, {
            threshold: 0.1,
            rootMargin: '50px'
        });

        chartCards.forEach(card => observer.observe(card));
    }

    /**
     * Destroy all chart instances (cleanup)
     */
    function destroyAll() {
        Object.values(chartInstances).forEach(chart => {
            if (chart && typeof chart.destroy === 'function') {
                chart.destroy();
            }
        });
        
        // Clear the instances object
        for (const key in chartInstances) {
            delete chartInstances[key];
        }
    }

    /**
     * Update chart data dynamically
     */
    function updateChart(chartName, newData) {
        const chart = chartInstances[chartName];
        if (!chart) {
            console.warn(`Chart "${chartName}" not found`);
            return;
        }

        chart.data.datasets.forEach((dataset, index) => {
            if (newData.datasets && newData.datasets[index]) {
                dataset.data = newData.datasets[index].data;
            }
        });

        if (newData.labels) {
            chart.data.labels = newData.labels;
        }

        chart.update('none'); // Update without animation for performance
    }

    /**
     * Initialize Case Status Distribution Chart
     */
    function initCaseStatusChart() {
        const ctx = document.getElementById('caseStatusChart');
        if (!ctx) return;

        const chart = new Chart(ctx.getContext('2d'), {
            type: 'doughnut',
            data: {
                labels: ['Active', 'Settled', 'Pending', 'Closed'],
                datasets: [{
                    data: [32, 15, 8, 5],
                    backgroundColor: [
                        chartColors.primary,
                        chartColors.success,
                        chartColors.warning,
                        chartColors.danger
                    ],
                    borderWidth: 3,
                    borderColor: '#ffffff',
                    hoverOffset: 15
                }]
            },
            options: {
                ...commonOptions,
                cutout: '70%',
                plugins: {
                    ...commonOptions.plugins,
                    tooltip: {
                        ...commonOptions.plugins.tooltip,
                        callbacks: {
                            label: ctx => {
                                const label = ctx.label || '';
                                const value = ctx.parsed;
                                const total = ctx.dataset.data.reduce((a, b) => a + b, 0);
                                const percentage = Math.round((value / total) * 100);
                                return label + ': ' + value + ' cases (' + percentage + '%)';
                            }
                        }
                    }
                }
            }
        });

        chartInstances.caseStatus = chart;
    }

    /**
     * Initialize Client Growth Chart
     */
    function initClientGrowthChart() {
        const ctx = document.getElementById('clientGrowthChart');
        if (!ctx) return;

        const chart = new Chart(ctx.getContext('2d'), {
            type: 'line',
            data: {
                labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun'],
                datasets: [{
                    label: 'New Clients',
                    data: [12, 18, 15, 22, 28, 35],
                    borderColor: chartColors.success,
                    backgroundColor: chartColors.successLight,
                    tension: 0.4,
                    fill: true,
                    pointBackgroundColor: chartColors.success,
                    pointBorderColor: '#ffffff',
                    pointBorderWidth: 2,
                    pointRadius: 5,
                    pointHoverRadius: 7
                }]
            },
            options: {
                ...commonOptions,
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0,0,0,0.05)',
                            drawBorder: false
                        }
                    },
                    x: {
                        grid: { display: false }
                    }
                }
            }
        });

        chartInstances.clientGrowth = chart;
    }

    /**
     * Initialize Revenue by Practice Area Chart
     */
    function initRevenueByPracticeChart() {
        const ctx = document.getElementById('revenueByPracticeChart');
        if (!ctx) return;

        const chart = new Chart(ctx.getContext('2d'), {
            type: 'horizontalBar',
            data: {
                labels: ['Corporate Law', 'Litigation', 'Real Estate', 'Family Law', 'IP Law', 'Employment'],
                datasets: [{
                    label: 'Revenue ($)',
                    data: [185000, 142000, 98000, 65000, 45000, 38000],
                    backgroundColor: [
                        chartColors.primary,
                        chartColors.success,
                        chartColors.warning,
                        chartColors.danger,
                        chartColors.purple,
                        chartColors.cyan
                    ],
                    borderRadius: 6,
                    borderWidth: 0
                }]
            },
            options: {
                ...commonOptions,
                indexAxis: 'y',
                scales: {
                    x: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0,0,0,0.05)',
                            drawBorder: false
                        },
                        ticks: {
                            callback: value => '$' + (value / 1000) + 'K'
                        }
                    },
                    y: {
                        grid: { display: false }
                    }
                }
            }
        });

        chartInstances.revenueByPractice = chart;
    }

    /**
     * Initialize Monthly Expenses Chart
     */
    function initMonthlyExpensesChart() {
        const ctx = document.getElementById('monthlyExpensesChart');
        if (!ctx) return;

        const chart = new Chart(ctx.getContext('2d'), {
            type: 'bar',
            data: {
                labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun'],
                datasets: [
                    {
                        label: 'Operating Expenses',
                        data: [45000, 48000, 42000, 51000, 47000, 49000],
                        backgroundColor: chartColors.danger,
                        borderRadius: 6,
                        borderWidth: 0
                    },
                    {
                        label: 'Staff Costs',
                        data: [185000, 188000, 192000, 195000, 198000, 201000],
                        backgroundColor: chartColors.warning,
                        borderRadius: 6,
                        borderWidth: 0
                    }
                ]
            },
            options: {
                ...commonOptions,
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(0,0,0,0.05)',
                            drawBorder: false
                        },
                        stacked: true,
                        ticks: {
                            callback: value => '$' + (value / 1000) + 'K'
                        }
                    },
                    x: {
                        grid: { display: false },
                        stacked: true
                    }
                }
            }
        });

        chartInstances.monthlyExpenses = chart;
    }

    /**
     * Initialize Win Rate Trend Chart
     */
    function initWinRateChart() {
        const ctx = document.getElementById('winRateChart');
        if (!ctx) return;

        const chart = new Chart(ctx.getContext('2d'), {
            type: 'line',
            data: {
                labels: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun'],
                datasets: [{
                    label: 'Win Rate (%)',
                    data: [72, 75, 78, 76, 82, 85],
                    borderColor: chartColors.success,
                    backgroundColor: chartColors.successLight,
                    tension: 0.4,
                    fill: true,
                    pointBackgroundColor: chartColors.success,
                    pointBorderColor: '#ffffff',
                    pointBorderWidth: 2,
                    pointRadius: 5,
                    pointHoverRadius: 7
                }]
            },
            options: {
                ...commonOptions,
                scales: {
                    y: {
                        beginAtZero: false,
                        min: 60,
                        max: 100,
                        grid: {
                            color: 'rgba(0,0,0,0.05)',
                            drawBorder: false
                        },
                        ticks: {
                            callback: value => value + '%'
                        }
                    },
                    x: {
                        grid: { display: false }
                    }
                }
            }
        });

        chartInstances.winRate = chart;
    }

    /**
     * Initialize Court Filing Status Chart
     */
    function initCourtFilingChart() {
        const ctx = document.getElementById('courtFilingChart');
        if (!ctx) return;

        const chart = new Chart(ctx.getContext('2d'), {
            type: 'pie',
            data: {
                labels: ['Filed', 'Pending Review', 'Approved', 'Rejected'],
                datasets: [{
                    data: [45, 23, 18, 14],
                    backgroundColor: [
                        chartColors.success,
                        chartColors.warning,
                        chartColors.primary,
                        chartColors.danger
                    ],
                    borderWidth: 3,
                    borderColor: '#ffffff',
                    hoverOffset: 15
                }]
            },
            options: {
                ...commonOptions,
                plugins: {
                    ...commonOptions.plugins,
                    tooltip: {
                        ...commonOptions.plugins.tooltip,
                        callbacks: {
                            label: ctx => ctx.label + ': ' + ctx.parsed + ' filings'
                        }
                    }
                }
            }
        });

        chartInstances.courtFiling = chart;
    }

    /**
     * Create custom chart with dynamic configuration
     */
    function createCustomChart(canvasId, config) {
        const ctx = document.getElementById(canvasId);
        if (!ctx) return null;

        const defaultConfig = {
            type: 'bar',
            data: {
                labels: [],
                datasets: [{
                    label: 'Data',
                    data: [],
                    backgroundColor: chartColors.primary,
                    borderColor: chartColors.primary,
                    borderWidth: 1
                }]
            },
            options: {
                ...commonOptions
            }
        };

        const mergedConfig = {
            ...defaultConfig,
            ...config,
            options: {
                ...defaultConfig.options,
                ...config.options
            }
        };

        const chart = new Chart(ctx.getContext('2d'), mergedConfig);
        chartInstances[canvasId] = chart;
        return chart;
    }

    /**
     * Export chart as image
     */
    function exportChartAsImage(chartId, filename) {
        const canvas = document.getElementById(chartId);
        if (!canvas) {
            console.error(`Canvas with id "${chartId}" not found`);
            return;
        }

        try {
            const link = document.createElement('a');
            link.download = filename || `${chartId}-${new Date().toISOString().split('T')[0]}.png`;
            link.href = canvas.toDataURL('image/png');
            link.click();
        } catch (error) {
            console.error('Failed to export chart:', error);
        }
    }

    /**
     * Get chart data as CSV
     */
    function getChartDataAsCSV(chartId) {
        const chart = chartInstances[chartId];
        if (!chart) return null;

        let csv = 'Label,Value\n';
        chart.data.labels.forEach((label, index) => {
            chart.data.datasets.forEach(dataset => {
                csv += `"${label}","${dataset.data[index]}"\n`;
            });
        });

        return csv;
    }

    /**
     * Export chart data as CSV
     */
    function exportChartDataAsCSV(chartId, filename) {
        const csv = getChartDataAsCSV(chartId);
        if (!csv) return;

        try {
            const blob = new Blob([csv], { type: 'text/csv' });
            const url = URL.createObjectURL(blob);

            const link = document.createElement('a');
            link.href = url;
            link.download = filename || `${chartId}-data-${new Date().toISOString().split('T')[0]}.csv`;
            link.click();

            URL.revokeObjectURL(url);
        } catch (error) {
            console.error('Failed to export chart data:', error);
        }
    }

    /**
     * Apply theme to all charts
     */
    function applyThemeToCharts(theme) {
        const isDark = theme === 'dark';
        const textColor = isDark ? '#ffffff' : '#374151';
        const gridColor = isDark ? 'rgba(255,255,255,0.1)' : 'rgba(0,0,0,0.05)';

        Object.values(chartInstances).forEach(chart => {
            if (chart && chart.options) {
                // Update text colors
                if (chart.options.plugins && chart.options.plugins.legend) {
                    chart.options.plugins.legend.labels.color = textColor;
                }

                // Update grid colors
                if (chart.options.scales) {
                    Object.values(chart.options.scales).forEach(scale => {
                        if (scale.grid) {
                            scale.grid.color = gridColor;
                        }
                        if (scale.ticks) {
                            scale.ticks.color = textColor;
                        }
                    });
                }

                chart.update();
            }
        });
    }

    /**
     * Resize all charts
     */
    function resizeAllCharts() {
        Object.values(chartInstances).forEach(chart => {
            if (chart && typeof chart.resize === 'function') {
                chart.resize();
            }
        });
    }

    /**
     * Get chart performance metrics
     */
    function getChartMetrics() {
        const metrics = {
            totalCharts: Object.keys(chartInstances).length,
            chartTypes: {},
            memoryUsage: 0,
            renderTime: 0
        };

        Object.values(chartInstances).forEach(chart => {
            if (chart) {
                const type = chart.config.type;
                metrics.chartTypes[type] = (metrics.chartTypes[type] || 0) + 1;
            }
        });

        return metrics;
    }

    // Public API
    return {
        init: lazyLoadCharts,
        destroy: destroyAll,
        update: updateChart,
        instances: chartInstances,

        // Additional chart initializers
        initRevenueChart,
        initPracticeAreaChart,
        initUtilizationChart,
        initARAgingChart,
        initMatterPerformanceChart,
        initTimeTrackingChart,
        initFinancialChart,
        initCaseStatusChart,
        initClientGrowthChart,
        initRevenueByPracticeChart,
        initMonthlyExpensesChart,
        initWinRateChart,
        initCourtFilingChart,

        // Utility functions
        createCustomChart,
        exportChartAsImage,
        exportChartDataAsCSV,
        applyThemeToCharts,
        resizeAllCharts,
        getChartMetrics
    };
})();

// Auto-initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
        // Will be initialized from main app with data
    });
}