@extends('layouts.auth')

@section('page-title', 'Forgot Password - Prolex Legal System')

@section('content')
<div class="auth-container">
    <!-- Left Panel - Branding -->
    <div class="auth-branding-panel">
        <div class="branding-content">
            <div class="logo-container">
                <i class="fas fa-balance-scale logo-icon"></i>
            </div>
            <h1 class="brand-title">Prolex Legal System</h1>
            <p class="brand-tagline">Secure Legal Management Platform</p>
            
            <div class="brand-features">
                <div class="feature-item">
                    <i class="fas fa-shield-alt"></i>
                    <span>Bank-Level Security</span>
                </div>
                <div class="feature-item">
                    <i class="fas fa-paper-plane"></i>
                    <span>Instant Email Delivery</span>
                </div>
                <div class="feature-item">
                    <i class="fas fa-user-lock"></i>
                    <span>Secure Password Reset</span>
                </div>
            </div>

            <!-- Recovery Tips -->
            <div class="recovery-tips">
                <h4><i class="fas fa-lightbulb"></i> Password Recovery Tips</h4>
                <ul>
                    <li><i class="fas fa-check-circle"></i> Check your spam/junk folder</li>
                    <li><i class="fas fa-check-circle"></i> Use a secure email address</li>
                    <li><i class="fas fa-check-circle"></i> Reset link expires in 1 hour</li>
                    <li><i class="fas fa-check-circle"></i> Contact support if needed</li>
                </ul>
            </div>
        </div>
    </div>

    <!-- Right Panel - Forgot Password Form -->
    <div class="auth-form-panel">
        <div class="auth-form-wrapper">
            <div class="form-header">
                <div class="form-header-icon">
                    <i class="fas fa-unlock-alt"></i>
                </div>
                <h2 class="form-title">Reset Your Password</h2>
                <p class="form-subtitle">Enter your email to receive a password reset link</p>
            </div>

            @if(session('status'))
                <div class="alert alert-success fade-in">
                    <div class="alert-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="alert-content">
                        <strong>Success!</strong>
                        <p>{{ session('status') }}</p>
                        <p class="alert-note">Check your email inbox and spam folder for the reset link.</p>
                    </div>
                </div>
            @endif

            @if($errors->any())
                <div class="alert alert-danger fade-in">
                    <div class="alert-icon">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                    <div class="alert-content">
                        <strong>Request Failed</strong>
                        @foreach($errors->all() as $error)
                            <p>{{ $error }}</p>
                        @endforeach
                    </div>
                </div>
            @endif

            <form method="POST" action="{{ route('password.email') }}" class="forgot-form" id="forgotForm">
                @csrf

                <!-- Email Field -->
                <div class="form-group">
                    <label for="email" class="form-label">
                        <i class="fas fa-envelope"></i> Email Address
                    </label>
                    <div class="input-wrapper">
                        <i class="fas fa-user-circle input-icon"></i>
                        <input type="email"
                               class="form-input"
                               id="email"
                               name="email"
                               value="{{ old('email') }}"
                               placeholder="your.email@example.com"
                               autocomplete="email"
                               required
                               autofocus>
                        <div class="input-status" id="emailStatus">
                            <i class="fas fa-check"></i>
                        </div>
                    </div>
                    <div class="form-hint">
                        <i class="fas fa-info-circle"></i> Enter the email associated with your account
                    </div>
                </div>

                <!-- Email Domain Suggestions -->
                <div class="domain-suggestions" id="domainSuggestions">
                    <div class="suggestions-header">
                        <i class="fas fa-lightbulb"></i> Common Email Domains
                    </div>
                    <div class="suggestion-chips">
                        <button type="button" class="chip" data-domain="@gmail.com">@gmail.com</button>
                        <button type="button" class="chip" data-domain="@yahoo.com">@yahoo.com</button>
                        <button type="button" class="chip" data-domain="@outlook.com">@outlook.com</button>
                        <button type="button" class="chip" data-domain="@hotmail.com">@hotmail.com</button>
                    </div>
                </div>

                <!-- Security Verification -->
                <div class="security-verification">
                    <div class="verification-icon">
                        <i class="fas fa-user-shield"></i>
                    </div>
                    <div class="verification-content">
                        <h5>Security Verification</h5>
                        <p>A secure password reset link will be sent to your email address. This link expires in 60 minutes for your protection.</p>
                    </div>
                </div>

                <!-- Submit Button -->
                <button type="submit" class="btn-auth-primary" id="submitBtn">
                    <i class="fas fa-paper-plane"></i> Send Reset Link
                    <i class="fas fa-arrow-right btn-icon"></i>
                </button>

                <!-- Alternative Options -->
                <div class="alternative-options">
                    <div class="option-item">
                        <i class="fas fa-question-circle"></i>
                        <div>
                            <strong>Can't access your email?</strong>
                            <p>Contact our support team for assistance</p>
                        </div>
                    </div>
                </div>
            </form>

            <!-- Action Links -->
            <div class="auth-actions">
                <a href="{{ route('login') }}" class="auth-action-link">
                    <i class="fas fa-arrow-left"></i>
                    <span>Back to Login</span>
                </a>
            </div>
        </div>
    </div>
</div>

@push('styles')
<style>
:root {
    --primary-color: #87CEEB;
    --secondary-color: #87CEEB;
    --accent-color: #e74c3c;
    --success-color: #27ae60;
    --warning-color: #f39c12;
    --light-bg: #f8f9fa;
    --dark-text: #2c3e50;
    --light-text: #7f8c8d;
    --border-color: #e0e6ed;
    --shadow-light: 0 4px 6px rgba(0,0,0,0.1);
    --shadow-medium: 0 8px 25px rgba(0,0,0,0.15);
}

.auth-container {
    display: flex;
    min-height: 100vh;
    width: 100%;
    max-width: 968px;
    margin: 0 auto;
}

/* Left Panel Styles */
.auth-branding-panel {
    flex: 0 0 35%;
    background: linear-gradient(135deg, var(--primary-color) 0%, #1a252f 100%);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    position: relative;
    overflow: hidden;
    min-width: 450px;
}

.auth-branding-panel::before {
    content: '';
    position: absolute;
    top: -50%;
    right: -50%;
    width: 200%;
    height: 200%;
    background: radial-gradient(circle, rgba(255,255,255,0.1) 1px, transparent 1px);
    background-size: 50px 50px;
    animation: float 20s linear infinite;
}

@keyframes float {
    0% { transform: translate(0, 0) rotate(0deg); }
    100% { transform: translate(-50px, -50px) rotate(360deg); }
}

.branding-content {
    max-width: 480px;
    width: 90%;
    z-index: 1;
    text-align: center;
    padding: 2rem;
}

.logo-icon {
    font-size: 4rem;
}

.logo-animation {
    position: relative;
    width: 110px;
    height: 110px;
    margin: 0 auto 2rem;
}

.logo-circle {
    position: absolute;
    width: 100%;
    height: 100%;
    border: 3px solid rgba(255,255,255,0.2);
    border-radius: 50%;
    animation: pulse 2s ease-in-out infinite;
}

@keyframes pulse {
    0%, 100% { transform: scale(1); opacity: 0.5; }
    50% { transform: scale(1.1); opacity: 0.8; }
}

.logo-image {
    width: 70px;
    height: 70px;
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    filter: brightness(0) invert(1);
}

.brand-title {
    font-size: 2.3rem;
    font-weight: 700;
    margin-bottom: 0.8rem;
    background: linear-gradient(45deg, #fff, #bdc3c7);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    line-height: 1.2;
}

.brand-tagline {
    font-size: 1rem;
    color: rgba(255,255,255,0.85);
    margin-bottom: 3rem;
    line-height: 1.5;
}

.brand-features {
    display: flex;
    flex-direction: column;
    gap: 1rem;
    margin-bottom: 3rem;
}

.feature-item {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1rem 1.2rem;
    background: rgba(255,255,255,0.1);
    border-radius: 10px;
    backdrop-filter: blur(10px);
    transition: transform 0.3s ease;
    font-size: 0.95rem;
}

.feature-item:hover {
    transform: translateX(8px);
    background: rgba(255,255,255,0.15);
}

.feature-item i {
    font-size: 1.1rem;
    color: var(--secondary-color);
    min-width: 24px;
}

/* Recovery Tips */
.recovery-tips {
    background: rgba(255,255,255,0.08);
    border-radius: 12px;
    padding: 1.5rem;
    margin-top: 2rem;
    text-align: left;
    backdrop-filter: blur(10px);
    border: 1px solid rgba(255,255,255,0.1);
}

.recovery-tips h4 {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 1rem;
    margin-bottom: 1rem;
    color: var(--secondary-color);
}

.recovery-tips h4 i {
    font-size: 1.1rem;
}

.recovery-tips ul {
    list-style: none;
    padding: 0;
}

.recovery-tips li {
    display: flex;
    align-items: center;
    gap: 0.8rem;
    padding: 0.4rem 0;
    font-size: 0.9rem;
    color: rgba(255,255,255,0.9);
}

.recovery-tips li i {
    color: var(--success-color);
    font-size: 0.9rem;
}

/* Right Panel Styles */
.auth-form-panel {
    flex: 0 0 60%;
    background: white;
    display: flex;
    align-items: center;
    justify-content: center;
    padding: 2.5rem;
    box-shadow: var(--shadow-medium);
    min-width: 600px;
}

.auth-form-wrapper {
    width: 100%;
    max-width: 600px;
    padding: 1.5rem;
}

/* Form Header */
.form-header {
    text-align: center;
    margin-bottom: 2.5rem;
}

.form-header-icon {
    font-size: 3rem;
    color: var(--secondary-color);
    margin-bottom: 1rem;
    animation: bounce 2s ease infinite;
}

@keyframes bounce {
    0%, 100% { transform: translateY(0); }
    50% { transform: translateY(-10px); }
}

.form-title {
    font-size: 2rem;
    color: var(--dark-text);
    margin-bottom: 0.6rem;
    font-weight: 700;
}

.form-subtitle {
    color: var(--light-text);
    font-size: 1rem;
    line-height: 1.5;
}

/* Alert Styles */
.alert {
    padding: 1.2rem 1.5rem;
    border-radius: 12px;
    margin-bottom: 2rem;
    display: flex;
    align-items: flex-start;
    gap: 1.2rem;
    animation: slideIn 0.5s ease;
    border: none;
}

@keyframes slideIn {
    from { 
        opacity: 0; 
        transform: translateY(-20px); 
    }
    to { 
        opacity: 1; 
        transform: translateY(0); 
    }
}

.alert-success {
    background: linear-gradient(135deg, #d4ffde, #c3e6cb);
    border-left: 4px solid var(--success-color);
}

.alert-danger {
    background: linear-gradient(135deg, #ffeaea, #f5c6cb);
    border-left: 4px solid var(--accent-color);
}

.alert-icon {
    font-size: 1.5rem;
    margin-top: 0.2rem;
}

.alert-success .alert-icon {
    color: var(--success-color);
}

.alert-danger .alert-icon {
    color: var(--accent-color);
}

.alert-content {
    flex: 1;
}

.alert-content strong {
    display: block;
    margin-bottom: 0.3rem;
    font-size: 1rem;
    color: var(--dark-text);
}

.alert-content p {
    font-size: 0.9rem;
    margin: 0.2rem 0;
    color: var(--light-text);
}

.alert-note {
    color: var(--success-color) !important;
    font-weight: 500;
    margin-top: 0.5rem !important;
}

/* Form Styles */
.forgot-form {
    width: 100%;
}

.form-group {
    margin-bottom: 2rem;
}

.form-label {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 0.92rem;
    font-weight: 600;
    color: var(--dark-text);
    margin-bottom: 0.5rem;
}

.form-label i {
    font-size: 0.9rem;
}

.input-wrapper {
    position: relative;
}

.form-input {
    width: 100%;
    padding: 0.95rem 3rem 0.95rem 3rem;
    border: 2px solid var(--border-color);
    border-radius: 10px;
    font-size: 0.95rem;
    transition: all 0.3s ease;
    background: var(--light-bg);
    height: 52px;
}

.form-input:focus {
    outline: none;
    border-color: var(--secondary-color);
    background: white;
    box-shadow: 0 0 0 3px rgba(135, 206, 235, 0.1);
}

.input-icon {
    position: absolute;
    left: 1rem;
    top: 50%;
    transform: translateY(-50%);
    color: var(--light-text);
    transition: color 0.3s ease;
    font-size: 1.1rem;
}

.form-input:focus ~ .input-icon {
    color: var(--secondary-color);
}

.input-status {
    position: absolute;
    right: 1rem;
    top: 50%;
    transform: translateY(-50%);
    color: var(--success-color);
    font-size: 1rem;
    opacity: 0;
    transition: opacity 0.3s ease;
}

.input-status.valid {
    opacity: 1;
}

.form-hint {
    display: block;
    margin-top: 0.5rem;
    font-size: 0.85rem;
    color: var(--light-text);
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

/* Domain Suggestions */
.domain-suggestions {
    background: linear-gradient(135deg, #f8f9fa, #e9ecef);
    border-radius: 10px;
    padding: 1.5rem;
    margin: 1.5rem 0;
    border: 1px solid var(--border-color);
}

.suggestions-header {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 0.9rem;
    font-weight: 600;
    color: var(--dark-text);
    margin-bottom: 1rem;
}

.suggestions-header i {
    color: var(--warning-color);
}

.suggestion-chips {
    display: flex;
    flex-wrap: wrap;
    gap: 0.5rem;
}

.chip {
    padding: 0.5rem 1rem;
    background: white;
    border: 1px solid var(--border-color);
    border-radius: 20px;
    font-size: 0.85rem;
    color: var(--light-text);
    cursor: pointer;
    transition: all 0.3s ease;
}

.chip:hover {
    background: var(--secondary-color);
    color: white;
    border-color: var(--secondary-color);
    transform: translateY(-2px);
}

/* Security Verification */
.security-verification {
    display: flex;
    gap: 1rem;
    padding: 1.5rem;
    background: linear-gradient(135deg, #f0f8ff, #e6f2ff);
    border-radius: 10px;
    margin: 1.5rem 0;
    border-left: 4px solid var(--secondary-color);
}

.verification-icon {
    font-size: 1.8rem;
    color: var(--secondary-color);
    margin-top: 0.2rem;
}

.verification-content h5 {
    color: var(--dark-text);
    margin-bottom: 0.5rem;
    font-size: 1rem;
}

.verification-content p {
    color: var(--light-text);
    font-size: 0.9rem;
    line-height: 1.5;
    margin: 0;
}

/* Submit Button */
.btn-auth-primary {
    width: 100%;
    padding: 1rem 2rem;
    background: linear-gradient(135deg, var(--secondary-color), #2980b9);
    color: white;
    border: none;
    border-radius: 10px;
    font-size: 1rem;
    font-weight: 600;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.8rem;
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
    height: 54px;
    margin: 2rem 0;
}

.btn-auth-primary:hover:not(:disabled) {
    transform: translateY(-2px);
    box-shadow: 0 10px 20px rgba(52, 152, 219, 0.3);
}

.btn-auth-primary:disabled {
    opacity: 0.6;
    cursor: not-allowed;
}

.btn-auth-primary .btn-icon {
    transition: transform 0.3s ease;
    font-size: 0.95rem;
}

.btn-auth-primary:hover .btn-icon {
    transform: translateX(5px);
}

/* Alternative Options */
.alternative-options {
    margin: 1.5rem 0;
}

.option-item {
    display: flex;
    gap: 1rem;
    padding: 1.2rem;
    background: linear-gradient(135deg, #fff9e6, #fff3cd);
    border-radius: 10px;
    border-left: 4px solid var(--warning-color);
}

.option-item i {
    font-size: 1.5rem;
    color: var(--warning-color);
    margin-top: 0.2rem;
}

.option-item strong {
    display: block;
    color: var(--dark-text);
    font-size: 0.95rem;
    margin-bottom: 0.3rem;
}

.option-item p {
    color: var(--light-text);
    font-size: 0.85rem;
    margin: 0;
    line-height: 1.5;
}

/* Action Links */
.auth-actions {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding-top: 2rem;
    border-top: 1px solid var(--border-color);
}

.auth-action-link {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    color: var(--secondary-color);
    text-decoration: none;
    font-size: 0.9rem;
    font-weight: 500;
    transition: all 0.3s ease;
    padding: 0.5rem 1rem;
    border-radius: 6px;
}

.auth-action-link:hover {
    background: rgba(135, 206, 235, 0.1);
    transform: translateX(5px);
}

.auth-action-link i {
    font-size: 0.9rem;
}

.action-divider {
    width: 1px;
    height: 24px;
    background: var(--border-color);
}

/* Responsive Design */
@media (max-width: 1200px) {
    .auth-container {
        min-height: 850px;
    }
    
    .auth-branding-panel {
        flex: 0 0 40%;
        min-width: 450px;
    }
    
    .auth-form-panel {
        flex: 0 0 60%;
        min-width: 550px;
    }
    
    .auth-form-wrapper {
        max-width: 550px;
    }
}

@media (max-width: 1024px) {
    .auth-container {
        flex-direction: column;
        min-height: auto;
        height: auto;
        overflow-y: auto;
    }
    
    .auth-branding-panel {
        flex: 0 0 auto;
        min-width: 100%;
        padding: 2rem;
        min-height: 400px;
    }
    
    .auth-form-panel {
        flex: 0 0 auto;
        min-width: 100%;
        padding: 2rem;
    }

    .branding-content {
        max-width: 600px;
    }
}

@media (max-width: 768px) {
    .auth-form-wrapper {
        max-width: 100%;
        padding: 1rem;
    }

    .form-title {
        font-size: 1.8rem;
    }
    
    .auth-branding-panel {
        min-height: 350px;
    }
    
    .brand-title {
        font-size: 2rem;
    }
    
    .auth-actions {
        flex-direction: column;
        gap: 1rem;
    }
    
    .action-divider {
        width: 100%;
        height: 1px;
    }
    
    .suggestion-chips {
        justify-content: center;
    }
}

@media (max-width: 480px) {
    .auth-form-panel {
        padding: 1.5rem;
    }

    .auth-branding-panel {
        padding: 1.5rem;
    }

    .brand-title {
        font-size: 1.8rem;
    }
    
    .logo-animation {
        width: 100px;
        height: 100px;
    }
    
    .logo-image {
        width: 60px;
        height: 60px;
    }
}
</style>
@endpush

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const emailInput = document.getElementById('email');
    const emailStatus = document.getElementById('emailStatus');
    const domainSuggestions = document.getElementById('domainSuggestions');
    const submitBtn = document.getElementById('submitBtn');
    const forgotForm = document.getElementById('forgotForm');
    const chips = document.querySelectorAll('.chip');

    // Email validation
    function validateEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }

    // Update email status
    function updateEmailStatus(email) {
        if (validateEmail(email)) {
            emailStatus.classList.add('valid');
            emailStatus.innerHTML = '<i class="fas fa-check"></i>';
            submitBtn.disabled = false;
        } else {
            emailStatus.classList.remove('valid');
            submitBtn.disabled = true;
        }
    }

    // Handle domain suggestion chips
    chips.forEach(chip => {
        chip.addEventListener('click', function() {
            const domain = this.getAttribute('data-domain');
            const currentValue = emailInput.value;
            
            // Check if input already has @ symbol
            if (currentValue.includes('@')) {
                // Replace everything after @ with selected domain
                const username = currentValue.split('@')[0];
                emailInput.value = username + domain;
            } else {
                // Just append domain
                emailInput.value = currentValue + domain;
            }
            
            emailInput.focus();
            updateEmailStatus(emailInput.value);
        });
    });

    // Email input event listeners
    emailInput.addEventListener('input', function() {
        updateEmailStatus(this.value);
    });

    emailInput.addEventListener('blur', function() {
        if (this.value && !validateEmail(this.value)) {
            showNotification('Please enter a valid email address', 'error');
        }
    });

    // Show domain suggestions based on input
    emailInput.addEventListener('keyup', function() {
        const value = this.value.toLowerCase();
        
        if (value.includes('@')) {
            domainSuggestions.style.display = 'none';
        } else if (value.length > 0) {
            domainSuggestions.style.display = 'block';
        } else {
            domainSuggestions.style.display = 'none';
        }
    });

    // Form submission
    forgotForm.addEventListener('submit', function(e) {
        const email = emailInput.value;
        
        if (!validateEmail(email)) {
            e.preventDefault();
            showNotification('Please enter a valid email address', 'error');
            return;
        }
        
        // Show loading state
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Sending Reset Link...';
        
        // Simulate sending delay
        setTimeout(() => {
            // This would be handled by Laravel, just visual feedback
            submitBtn.innerHTML = '<i class="fas fa-check"></i> Reset Link Sent!';
            submitBtn.style.background = 'linear-gradient(135deg, var(--success-color), #27ae60)';
        }, 2000);
    });

    // Auto-focus on page load
    emailInput.focus();

    // Notification function
    function showNotification(message, type) {
        // Remove existing notifications
        const existingAlert = document.querySelector('.notification-alert');
        if (existingAlert) {
            existingAlert.remove();
        }
        
        // Create notification
        const alert = document.createElement('div');
        alert.className = `notification-alert alert alert-${type === 'error' ? 'danger' : 'success'}`;
        alert.innerHTML = `
            <div class="alert-icon">
                <i class="fas fa-${type === 'error' ? 'exclamation-triangle' : 'check-circle'}"></i>
            </div>
            <div class="alert-content">
                <strong>${type === 'error' ? 'Error' : 'Success'}</strong>
                <p>${message}</p>
            </div>
        `;
        
        // Add styles for notification
        alert.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            z-index: 9999;
            animation: slideIn 0.3s ease;
            min-width: 300px;
            max-width: 400px;
        `;
        
        document.body.appendChild(alert);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            alert.style.animation = 'slideOut 0.3s ease forwards';
            setTimeout(() => alert.remove(), 300);
        }, 5000);
    }

    // Initialize
    updateEmailStatus(emailInput.value);
});
</script>
@endpush
@endsection