@extends('layouts.auth')

@section('page-title', 'Reset Password - Prolex Legal System')

@section('content')
<div class="auth-container">
    <!-- Left Panel - Branding (Same as Login) -->
    <div class="auth-branding-panel">
        <div class="branding-content">
            <div class="logo-animation">
                <div class="logo-circle"></div>
                <img src="{{ asset('prolexlogo.png') }}" alt="Prolex Legal System" class="logo-image">
            </div>
            <h1 class="brand-title">Prolex Legal System</h1>
            <p class="brand-tagline">Secure Legal Management Platform</p>
            
            <div class="brand-features">
                <div class="feature-item">
                    <i class="fas fa-shield-alt"></i>
                    <span>Bank-Level Security</span>
                </div>
                <div class="feature-item">
                    <i class="fas fa-key"></i>
                    <span>Password Encryption</span>
                </div>
                <div class="feature-item">
                    <i class="fas fa-user-lock"></i>
                    <span>Secure Authentication</span>
                </div>
            </div>

            <!-- Password Tips -->
            <div class="password-tips">
                <h4><i class="fas fa-lightbulb"></i> Password Tips</h4>
                <ul>
                    <li><i class="fas fa-check-circle"></i> Use 8+ characters</li>
                    <li><i class="fas fa-check-circle"></i> Mix letters & numbers</li>
                    <li><i class="fas fa-check-circle"></i> Include special characters</li>
                    <li><i class="fas fa-check-circle"></i> Avoid common words</li>
                </ul>
            </div>
        </div>
    </div>

    <!-- Right Panel - Reset Form -->
    <div class="auth-form-panel">
        <div class="auth-form-wrapper">
            <div class="form-header">
                <div class="form-header-icon">
                    <i class="fas fa-key-circle"></i>
                </div>
                <h2 class="form-title">Reset Your Password</h2>
                <p class="form-subtitle">Create a new secure password for your account</p>
            </div>

            @if(session('status'))
                <div class="alert alert-success fade-in">
                    <div class="alert-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="alert-content">
                        <strong>Success!</strong>
                        <p>{{ session('status') }}</p>
                        <p class="alert-note">You can now login with your new password.</p>
                    </div>
                </div>
            @endif

            @if($errors->any())
                <div class="alert alert-danger fade-in">
                    <div class="alert-icon">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                    <div class="alert-content">
                        <strong>Reset Failed</strong>
                        @foreach($errors->all() as $error)
                            <p>{{ $error }}</p>
                        @endforeach
                    </div>
                </div>
            @endif

            <form method="POST" action="{{ route('password.update') }}" class="reset-form" id="resetForm">
                @csrf

                <!-- Email Field (Readonly if pre-filled) -->
                <div class="form-group">
                    <label for="email" class="form-label">
                        <i class="fas fa-envelope"></i> Email Address
                    </label>
                    <div class="input-wrapper">
                        <i class="fas fa-user-circle input-icon"></i>
                        <input type="email"
                               class="form-input"
                               id="email"
                               name="email"
                               value="{{ old('email', $email ?? '') }}"
                               placeholder="your.email@example.com"
                               autocomplete="email"
                               required
                               readonly>
                        <i class="fas fa-check email-verified"></i>
                    </div>
                    <small class="form-hint">Email verified and secured</small>
                </div>

                <!-- New Password Field -->
                <div class="form-group">
                    <label for="password" class="form-label">
                        <i class="fas fa-lock"></i> New Password
                    </label>
                    <div class="input-wrapper">
                        <i class="fas fa-key input-icon"></i>
                        <input type="password"
                               class="form-input"
                               id="password"
                               name="password"
                               placeholder="Create a strong password"
                               autocomplete="new-password"
                               required
                               minlength="8">
                        <button type="button" class="password-toggle" data-target="password">
                            <i class="fas fa-eye"></i>
                            <i class="fas fa-eye-slash d-none"></i>
                        </button>
                    </div>

                    <!-- Password Strength Meter -->
                    <div class="password-strength-container">
                        <div class="strength-labels">
                            <span>Password Strength:</span>
                            <span class="strength-text" id="strengthText">Weak</span>
                        </div>
                        <div class="strength-meter">
                            <div class="strength-bar" id="strengthBar"></div>
                        </div>
                        <div class="strength-requirements" id="passwordRequirements">
                            <div class="requirement" data-requirement="length">
                                <i class="fas fa-times requirement-icon"></i>
                                <span>At least 8 characters</span>
                            </div>
                            <div class="requirement" data-requirement="uppercase">
                                <i class="fas fa-times requirement-icon"></i>
                                <span>One uppercase letter</span>
                            </div>
                            <div class="requirement" data-requirement="lowercase">
                                <i class="fas fa-times requirement-icon"></i>
                                <span>One lowercase letter</span>
                            </div>
                            <div class="requirement" data-requirement="number">
                                <i class="fas fa-times requirement-icon"></i>
                                <span>One number</span>
                            </div>
                            <div class="requirement" data-requirement="special">
                                <i class="fas fa-times requirement-icon"></i>
                                <span>One special character</span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Confirm Password Field -->
                <div class="form-group">
                    <label for="password_confirmation" class="form-label">
                        <i class="fas fa-lock"></i> Confirm Password
                    </label>
                    <div class="input-wrapper">
                        <i class="fas fa-key input-icon"></i>
                        <input type="password"
                               class="form-input"
                               id="password_confirmation"
                               name="password_confirmation"
                               placeholder="Re-enter your new password"
                               autocomplete="new-password"
                               required>
                        <button type="button" class="password-toggle" data-target="password_confirmation">
                            <i class="fas fa-eye"></i>
                            <i class="fas fa-eye-slash d-none"></i>
                        </button>
                        <div class="match-indicator" id="matchIndicator">
                            <i class="fas fa-check"></i>
                            <span>Passwords match</span>
                        </div>
                    </div>
                    <div class="match-status" id="matchStatus"></div>
                </div>

                <!-- Submit Button -->
                <button type="submit" class="btn-auth-primary" id="submitBtn">
                    <i class="fas fa-key"></i> Reset Password
                    <i class="fas fa-arrow-right btn-icon"></i>
                </button>

                <!-- Security Badge -->
                <div class="security-badge reset-badge">
                    <i class="fas fa-shield-check"></i>
                    <span>Your new password will be encrypted with AES-256</span>
                </div>
            </form>

            <!-- Action Links -->
            <div class="auth-actions">
                <a href="{{ route('login') }}" class="auth-action-link">
                    <i class="fas fa-arrow-left"></i>
                    <span>Back to Login</span>
                </a>
            </div>
        </div>
    </div>
</div>

@push('styles')
<style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary-color: #87CEEB;
            --secondary-color: #87CEEB;
            --accent-color: #e74c3c;
            --success-color: #27ae60;
            --warning-color: #f39c12;
            --light-bg: #f8f9fa;
            --dark-text: #2c3e50;
            --light-text: #7f8c8d;
            --border-color: #e0e6ed;
            --shadow-light: 0 4px 6px rgba(0,0,0,0.1);
            --shadow-medium: 0 8px 25px rgba(0,0,0,0.15);
        }

        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
        }

        .auth-container {
            display: flex;
            min-height: 100vh;
            width: 100%;
            max-width: 968px;
            margin: 0 auto;
        }

        /* LEFT PANEL - Fixed */
        .auth-branding-panel {
            flex: 0 0 35%;
            background: linear-gradient(135deg, var(--primary-color) 0%, #1a252f 100%);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            position: relative;
            overflow: hidden;
            min-width: 450px;
        }

        .auth-branding-panel::before {
            content: '';
            position: absolute;
            top: -50%;
            right: -50%;
            width: 200%;
            height: 200%;
            background: radial-gradient(circle, rgba(255,255,255,0.1) 1px, transparent 1px);
            background-size: 50px 50px;
            animation: float 20s linear infinite;
        }

        @keyframes float {
            0% { transform: translate(0, 0) rotate(0deg); }
            100% { transform: translate(-50px, -50px) rotate(360deg); }
        }

        .branding-content {
            max-width: 90%;
            z-index: 1;
            text-align: center;
            padding: 2rem;
        }

        .logo-animation {
            position: relative;
            width: 110px;
            height: 110px;
            margin: 0 auto 2rem;
        }

        .logo-circle {
            position: absolute;
            width: 100%;
            height: 100%;
            border: 3px solid rgba(255,255,255,0.2);
            border-radius: 50%;
            animation: pulse 2s ease-in-out infinite;
        }

        @keyframes pulse {
            0%, 100% { transform: scale(1); opacity: 0.5; }
            50% { transform: scale(1.1); opacity: 0.8; }
        }

        .logo-image {
            width: 70px;
            height: 70px;
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: white;
            border-radius: 50%;
        }

        .brand-title {
            font-size: 2.3rem;
            font-weight: 700;
            margin-bottom: 0.8rem;
            background: linear-gradient(45deg, #fff, #bdc3c7);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            line-height: 1.2;
        }

        .brand-tagline {
            font-size: 1rem;
            color: rgba(255,255,255,0.85);
            margin-bottom: 3rem;
            line-height: 1.5;
        }

        .brand-features {
            display: flex;
            flex-direction: column;
            gap: 1rem;
            margin-bottom: 3rem;
        }

        .feature-item {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1rem 1.2rem;
            background: rgba(255,255,255,0.1);
            border-radius: 10px;
            backdrop-filter: blur(10px);
            transition: transform 0.3s ease;
            font-size: 0.95rem;
        }

        .feature-item:hover {
            transform: translateX(8px);
            background: rgba(255,255,255,0.15);
        }

        .feature-item i {
            font-size: 1.1rem;
            color: var(--secondary-color);
            min-width: 24px;
        }

        .password-tips {
            background: rgba(255,255,255,0.08);
            border-radius: 12px;
            padding: 1.5rem;
            margin-top: 2rem;
            text-align: left;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255,255,255,0.1);
        }

        .password-tips h4 {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 1rem;
            margin-bottom: 1rem;
            color: var(--secondary-color);
        }

        .password-tips ul {
            list-style: none;
            padding: 0;
        }

        .password-tips li {
            display: flex;
            align-items: center;
            gap: 0.8rem;
            padding: 0.4rem 0;
            font-size: 0.9rem;
            color: rgba(255,255,255,0.9);
        }

        .password-tips li i {
            color: var(--success-color);
            font-size: 0.9rem;
        }

        /* RIGHT PANEL - Fixed */
        .auth-form-panel {
            flex: 1;
            background: white;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 3rem 2rem;
            overflow-y: auto;
        }

        .auth-form-wrapper {
            width: 100%;
            max-width: 600px;
            padding: 0 1rem;
        }

        .form-header {
            text-align: center;
            margin-bottom: 2.5rem;
        }

        .form-header-icon {
            font-size: 3rem;
            color: var(--secondary-color);
            margin-bottom: 1rem;
            animation: bounce 2s ease infinite;
        }

        @keyframes bounce {
            0%, 100% { transform: translateY(0); }
            50% { transform: translateY(-10px); }
        }

        .form-title {
            font-size: 2rem;
            color: var(--dark-text);
            margin-bottom: 0.6rem;
            font-weight: 700;
        }

        .form-subtitle {
            color: var(--light-text);
            font-size: 1rem;
            line-height: 1.5;
        }

        .alert {
            padding: 1.2rem 1.5rem;
            border-radius: 12px;
            margin-bottom: 2rem;
            display: flex;
            align-items: flex-start;
            gap: 1.2rem;
        }

        .alert-success {
            background: linear-gradient(135deg, #d4ffde, #c3e6cb);
            border-left: 4px solid var(--success-color);
        }

        .alert-icon {
            font-size: 1.5rem;
            color: var(--success-color);
        }

        .reset-form {
            width: 100%;
        }

        .form-group {
            margin-bottom: 2rem;
        }

        .form-label {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.92rem;
            font-weight: 600;
            color: var(--dark-text);
            margin-bottom: 0.5rem;
        }

        .input-wrapper {
            position: relative;
        }

        .form-input {
            width: 100%;
            padding: 0.95rem 3rem;
            border: 2px solid var(--border-color);
            border-radius: 10px;
            font-size: 0.95rem;
            transition: all 0.3s ease;
            background: var(--light-bg);
            height: 52px;
        }

        .form-input:focus {
            outline: none;
            border-color: var(--secondary-color);
            background: white;
            box-shadow: 0 0 0 3px rgba(135, 206, 235, 0.1);
        }

        .form-input[readonly] {
            background: #f8f9fa;
            border-color: #e9ecef;
            cursor: not-allowed;
            color: #6c757d;
        }

        .input-icon {
            position: absolute;
            left: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: var(--light-text);
            font-size: 1.1rem;
        }

        .email-verified {
            position: absolute;
            right: 1rem;
            top: 50%;
            transform: translateY(-50%);
            color: var(--success-color);
            font-size: 1rem;
        }

        .password-toggle {
            position: absolute;
            right: 1rem;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: none;
            color: var(--light-text);
            cursor: pointer;
            padding: 0.4rem;
            font-size: 1rem;
        }

        .password-toggle:hover {
            color: var(--secondary-color);
        }

        .d-none {
            display: none;
        }

        .form-hint {
            display: block;
            margin-top: 0.5rem;
            font-size: 0.85rem;
            color: var(--success-color);
        }

        .password-strength-container {
            margin-top: 1rem;
        }

        .strength-labels {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 0.5rem;
            font-size: 0.9rem;
        }

        .strength-text {
            font-weight: 600;
        }

        .strength-meter {
            height: 6px;
            background: #e9ecef;
            border-radius: 3px;
            overflow: hidden;
            margin-bottom: 1rem;
        }

        .strength-bar {
            height: 100%;
            width: 0%;
            background: #dc3545;
            border-radius: 3px;
            transition: all 0.3s ease;
        }

        .strength-requirements {
            display: flex;
            flex-direction: column;
            gap: 0.5rem;
        }

        .requirement {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.85rem;
            color: var(--light-text);
        }

        .requirement.met {
            color: var(--success-color);
        }

        .match-indicator {
            position: absolute;
            right: 3.5rem;
            top: 50%;
            transform: translateY(-50%);
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.85rem;
            color: var(--success-color);
            opacity: 0;
        }

        .match-indicator.show {
            opacity: 1;
        }

        .match-status {
            margin-top: 0.5rem;
            font-size: 0.85rem;
            min-height: 1.2rem;
        }

        .match-status.matching {
            color: var(--success-color);
        }

        .match-status.not-matching {
            color: var(--accent-color);
        }

        .btn-auth-primary {
            width: 100%;
            padding: 1rem 2rem;
            background: linear-gradient(135deg, var(--secondary-color), #2980b9);
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.8rem;
            transition: all 0.3s ease;
            height: 54px;
            margin: 2rem 0;
        }

        .btn-auth-primary:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(52, 152, 219, 0.3);
        }

        .reset-badge {
            text-align: center;
            padding: 1rem;
            background: linear-gradient(135deg, #f8f9fa, #e9ecef);
            border-radius: 8px;
            margin-bottom: 2rem;
            color: var(--light-text);
            font-size: 0.9rem;
        }

        .reset-badge i {
            color: var(--success-color);
            margin-right: 0.5rem;
        }

        .auth-actions {
            display: flex;
            justify-content: center;
            padding-top: 2rem;
            border-top: 1px solid var(--border-color);
        }

        .auth-action-link {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            color: var(--secondary-color);
            text-decoration: none;
            font-size: 0.9rem;
            font-weight: 500;
            transition: all 0.3s ease;
            padding: 0.5rem 1rem;
            border-radius: 6px;
        }

        .auth-action-link:hover {
            background: rgba(135, 206, 235, 0.1);
        }

        /* RESPONSIVE FIXES */
        @media (max-width: 1200px) {
            .auth-branding-panel {
                flex: 0 0 38%;
                min-width: 400px;
            }
        }

        @media (max-width: 1024px) {
            .auth-container {
                flex-direction: column;
            }
            
            .auth-branding-panel {
                flex: none;
                min-width: 100%;
                min-height: 400px;
            }
            
            .auth-form-panel {
                flex: none;
                padding: 3rem 2rem;
            }
        }

        @media (max-width: 768px) {
            .auth-branding-panel {
                min-height: 350px;
            }
            
            .brand-title {
                font-size: 2rem;
            }
            
            .form-title {
                font-size: 1.8rem;
            }

            .auth-form-panel {
                padding: 2rem 1.5rem;
            }
        }

        @media (max-width: 480px) {
            .auth-form-panel {
                padding: 2rem 1rem;
            }

            .brand-title {
                font-size: 1.8rem;
            }
            
            .logo-animation {
                width: 100px;
                height: 100px;
            }
        }
    </style>
@endpush

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const passwordInput = document.getElementById('password');
    const confirmPasswordInput = document.getElementById('password_confirmation');
    const strengthBar = document.getElementById('strengthBar');
    const strengthText = document.getElementById('strengthText');
    const passwordRequirements = document.getElementById('passwordRequirements');
    const matchIndicator = document.getElementById('matchIndicator');
    const matchStatus = document.getElementById('matchStatus');
    const submitBtn = document.getElementById('submitBtn');
    const resetForm = document.getElementById('resetForm');

    // Password toggle functionality
    document.querySelectorAll('.password-toggle').forEach(toggle => {
        toggle.addEventListener('click', function() {
            const targetId = this.getAttribute('data-target');
            const targetInput = document.getElementById(targetId);
            const eyeIcon = this.querySelector('.fa-eye');
            const eyeSlashIcon = this.querySelector('.fa-eye-slash');
            
            const type = targetInput.getAttribute('type') === 'password' ? 'text' : 'password';
            targetInput.setAttribute('type', type);
            eyeIcon.classList.toggle('d-none');
            eyeSlashIcon.classList.toggle('d-none');
        });
    });

    // Password strength checker
    function checkPasswordStrength(password) {
        let strength = 0;
        const requirements = {
            length: password.length >= 8,
            uppercase: /[A-Z]/.test(password),
            lowercase: /[a-z]/.test(password),
            number: /[0-9]/.test(password),
            special: /[^A-Za-z0-9]/.test(password)
        };

        // Update requirement indicators
        Object.keys(requirements).forEach(req => {
            const element = passwordRequirements.querySelector(`[data-requirement="${req}"]`);
            const icon = element.querySelector('.requirement-icon');
            
            if (requirements[req]) {
                element.classList.add('met');
                icon.classList.remove('fa-times');
                icon.classList.add('fa-check');
                strength += 1;
            } else {
                element.classList.remove('met');
                icon.classList.remove('fa-check');
                icon.classList.add('fa-times');
            }
        });

        // Update strength meter
        const percentage = (strength / 5) * 100;
        strengthBar.style.width = `${percentage}%`;
        
        // Update strength text and color
        if (percentage < 30) {
            strengthText.textContent = 'Very Weak';
            strengthBar.style.background = '#dc3545';
            strengthText.style.color = '#dc3545';
        } else if (percentage < 50) {
            strengthText.textContent = 'Weak';
            strengthBar.style.background = '#fd7e14';
            strengthText.style.color = '#fd7e14';
        } else if (percentage < 75) {
            strengthText.textContent = 'Fair';
            strengthBar.style.background = '#ffc107';
            strengthText.style.color = '#ffc107';
        } else if (percentage < 100) {
            strengthText.textContent = 'Strong';
            strengthBar.style.background = '#28a745';
            strengthText.style.color = '#28a745';
        } else {
            strengthText.textContent = 'Very Strong';
            strengthBar.style.background = '#20c997';
            strengthText.style.color = '#20c997';
        }

        return requirements;
    }

    // Password match checker
    function checkPasswordMatch() {
        const password = passwordInput.value;
        const confirmPassword = confirmPasswordInput.value;
        
        if (!password || !confirmPassword) {
            matchIndicator.classList.remove('show');
            matchStatus.textContent = '';
            matchStatus.className = 'match-status';
            return false;
        }

        if (password === confirmPassword) {
            matchIndicator.classList.add('show');
            matchStatus.textContent = 'Passwords match ✓';
            matchStatus.className = 'match-status matching';
            return true;
        } else {
            matchIndicator.classList.remove('show');
            matchStatus.textContent = 'Passwords do not match ✗';
            matchStatus.className = 'match-status not-matching';
            return false;
        }
    }

    // Form validation
    function validateForm() {
        const password = passwordInput.value;
        const requirements = checkPasswordStrength(password);
        const passwordValid = Object.values(requirements).every(req => req);
        const passwordsMatch = checkPasswordMatch();
        
        const allValid = passwordValid && passwordsMatch;
        submitBtn.disabled = !allValid;
        
        return allValid;
    }

    // Event listeners
    passwordInput.addEventListener('input', function() {
        checkPasswordStrength(this.value);
        checkPasswordMatch();
        validateForm();
    });

    confirmPasswordInput.addEventListener('input', function() {
        checkPasswordMatch();
        validateForm();
    });

    // Initial validation
    validateForm();

    // Form submission
    resetForm.addEventListener('submit', function(e) {
        if (!validateForm()) {
            e.preventDefault();
            return;
        }
        
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Resetting Password...';
    });

    // Initialize
    checkPasswordStrength(passwordInput.value);
    checkPasswordMatch();
});
</script>
@endpush
@endsection