<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Verify OTP - Prolex Legal System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
    <style>
        :root {
            --primary-color: #87CEEB;
            --secondary-color: #87CEEB;
            --accent-color: #e74c3c;
            --success-color: #27ae60;
            --warning-color: #f39c12;
            --light-bg: #f8f9fa;
            --dark-text: #2c3e50;
            --light-text: #7f8c8d;
            --border-color: #e0e6ed;
            --shadow-light: 0 4px 6px rgba(0,0,0,0.1);
            --shadow-medium: 0 8px 25px rgba(0,0,0,0.15);
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 2rem;
        }

        .auth-container {
            display: flex;
            width: 100%;
            max-width: 1200px;
            min-height: 700px;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: var(--shadow-medium);
        }

        /* Left Panel - Branding */
        .auth-branding-panel {
            flex: 0 0 40%;
            background: linear-gradient(135deg, var(--primary-color) 0%, #1a252f 100%);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            position: relative;
            overflow: hidden;
            min-width: 500px;
        }

        .auth-branding-panel::before {
            content: '';
            position: absolute;
            top: -50%;
            right: -50%;
            width: 200%;
            height: 200%;
            background: radial-gradient(circle, rgba(255,255,255,0.1) 1px, transparent 1px);
            background-size: 50px 50px;
            animation: float 20s linear infinite;
        }

        @keyframes float {
            0% { transform: translate(0, 0) rotate(0deg); }
            100% { transform: translate(-50px, -50px) rotate(360deg); }
        }

        .branding-content {
            max-width: 480px;
            width: 90%;
            z-index: 1;
            text-align: center;
            padding: 2rem;
        }

        .logo-animation {
            position: relative;
            width: 120px;
            height: 120px;
            margin: 0 auto 2rem;
        }

        .logo-circle {
            position: absolute;
            width: 100%;
            height: 100%;
            border: 3px solid rgba(255,255,255,0.2);
            border-radius: 50%;
            animation: pulse 2s ease-in-out infinite;
        }

        @keyframes pulse {
            0%, 100% { transform: scale(1); opacity: 0.5; }
            50% { transform: scale(1.1); opacity: 0.8; }
        }

        .logo-image {
            width: 80px;
            height: 80px;
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            filter: brightness(0) invert(1);
            border-radius: 50%;
        }

        .logo-icon {
            font-size: 4rem;
            color: white;
        }

        .brand-title {
            font-size: 2.5rem;
            font-weight: 700;
            margin-bottom: 0.8rem;
            background: linear-gradient(45deg, #fff, #bdc3c7);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            line-height: 1.2;
        }

        .brand-tagline {
            font-size: 1.1rem;
            color: rgba(255,255,255,0.85);
            margin-bottom: 3rem;
            line-height: 1.5;
        }

        .brand-features {
            display: flex;
            flex-direction: column;
            gap: 1rem;
            margin-bottom: 3rem;
        }

        .feature-item {
            display: flex;
            align-items: center;
            gap: 1rem;
            padding: 1rem 1.2rem;
            background: rgba(255,255,255,0.1);
            border-radius: 10px;
            backdrop-filter: blur(10px);
            transition: transform 0.3s ease;
            font-size: 0.95rem;
        }

        .feature-item:hover {
            transform: translateX(8px);
            background: rgba(255,255,255,0.15);
        }

        .feature-item i {
            font-size: 1.1rem;
            color: var(--secondary-color);
            min-width: 24px;
        }

        .otp-info {
            background: rgba(255,255,255,0.08);
            border-radius: 12px;
            padding: 1.5rem;
            margin-top: 2rem;
            text-align: left;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255,255,255,0.1);
        }

        .otp-info h4 {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 1rem;
            margin-bottom: 1rem;
            color: var(--secondary-color);
        }

        .otp-info ul {
            list-style: none;
            padding: 0;
        }

        .otp-info li {
            display: flex;
            align-items: center;
            gap: 0.8rem;
            padding: 0.4rem 0;
            font-size: 0.9rem;
            color: rgba(255,255,255,0.9);
        }

        .otp-info li i {
            color: var(--secondary-color);
            font-size: 0.9rem;
        }

        /* Right Panel - OTP Form */
        .auth-form-panel {
            flex: 0 0 60%;
            background: white;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 3rem;
            min-width: 600px;
        }

        .auth-form-wrapper {
            width: 100%;
            max-width: 500px;
        }

        /* Form Header */
        .form-header {
            text-align: center;
            margin-bottom: 2.5rem;
        }

        .form-header-icon {
            font-size: 3.5rem;
            color: var(--secondary-color);
            margin-bottom: 1rem;
            animation: bounce 2s ease infinite;
        }

        @keyframes bounce {
            0%, 100% { transform: translateY(0); }
            50% { transform: translateY(-10px); }
        }

        .form-title {
            font-size: 2.2rem;
            color: var(--dark-text);
            margin-bottom: 0.5rem;
            font-weight: 700;
        }

        .form-subtitle {
            color: var(--light-text);
            font-size: 1.05rem;
            line-height: 1.5;
        }

        .email-display {
            background: linear-gradient(135deg, #f0f8ff, #e6f2ff);
            border-radius: 10px;
            padding: 1rem 1.5rem;
            margin: 1.5rem 0;
            display: flex;
            align-items: center;
            gap: 1rem;
            border-left: 4px solid var(--secondary-color);
        }

        .email-display i {
            color: var(--secondary-color);
            font-size: 1.2rem;
        }

        .email-display strong {
            color: var(--dark-text);
            font-weight: 600;
        }

        /* Alert Styles */
        .alert {
            padding: 1.2rem 1.5rem;
            border-radius: 12px;
            margin-bottom: 2rem;
            display: flex;
            align-items: flex-start;
            gap: 1.2rem;
            animation: slideIn 0.5s ease;
            border: none;
        }

        @keyframes slideIn {
            from { 
                opacity: 0; 
                transform: translateY(-20px); 
            }
            to { 
                opacity: 1; 
                transform: translateY(0); 
            }
        }

        .alert-info {
            background: linear-gradient(135deg, #e3f2fd, #bbdefb);
            border-left: 4px solid #2196f3;
        }

        .alert-danger {
            background: linear-gradient(135deg, #ffeaea, #f5c6cb);
            border-left: 4px solid var(--accent-color);
        }

        .alert-icon {
            font-size: 1.5rem;
            margin-top: 0.2rem;
        }

        .alert-info .alert-icon {
            color: #2196f3;
        }

        .alert-danger .alert-icon {
            color: var(--accent-color);
        }

        .alert-content {
            flex: 1;
        }

        .alert-content strong {
            display: block;
            margin-bottom: 0.3rem;
            font-size: 1rem;
            color: var(--dark-text);
        }

        /* OTP Input Styles */
        .otp-form {
            width: 100%;
        }

        .otp-container {
            margin: 2rem 0;
        }

        .otp-inputs {
            display: flex;
            justify-content: center;
            gap: 0.8rem;
            margin-bottom: 1.5rem;
        }

        .otp-input {
            width: 60px;
            height: 70px;
            text-align: center;
            font-size: 2rem;
            font-weight: 700;
            border: 2px solid var(--border-color);
            border-radius: 10px;
            background: var(--light-bg);
            transition: all 0.3s ease;
            caret-color: var(--secondary-color);
        }

        .otp-input:focus {
            outline: none;
            border-color: var(--secondary-color);
            background: white;
            box-shadow: 0 0 0 3px rgba(135, 206, 235, 0.1);
            transform: translateY(-2px);
        }

        .otp-input.filled {
            border-color: var(--secondary-color);
            background: rgba(135, 206, 235, 0.1);
        }

        .otp-hint {
            text-align: center;
            color: var(--light-text);
            font-size: 0.9rem;
            margin-bottom: 0.5rem;
        }

        .timer-container {
            text-align: center;
            margin: 1.5rem 0;
        }

        .timer {
            font-size: 1.1rem;
            font-weight: 600;
            color: var(--secondary-color);
            padding: 0.8rem 1.5rem;
            background: rgba(135, 206, 235, 0.1);
            border-radius: 10px;
            display: inline-block;
        }

        .timer.expiring {
            color: var(--accent-color);
            animation: pulseWarning 1s ease infinite;
        }

        @keyframes pulseWarning {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.7; }
        }

        .resend-link {
            text-align: center;
            margin: 1rem 0;
        }

        .resend-link a {
            color: var(--secondary-color);
            text-decoration: none;
            font-weight: 500;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s ease;
            padding: 0.5rem 1rem;
            border-radius: 6px;
        }

        .resend-link a:hover {
            background: rgba(135, 206, 235, 0.1);
        }

        .resend-link a.disabled {
            color: var(--light-text);
            cursor: not-allowed;
            opacity: 0.6;
        }

        .resend-link a.disabled:hover {
            background: transparent;
        }

        /* Submit Button */
        .btn-auth-primary {
            width: 100%;
            padding: 1rem 2rem;
            background: linear-gradient(135deg, var(--secondary-color), #2980b9);
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.8rem;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
            height: 56px;
            margin: 1.5rem 0;
        }

        .btn-auth-primary:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(52, 152, 219, 0.3);
        }

        .btn-auth-primary:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }

        .btn-auth-primary i {
            transition: transform 0.3s ease;
        }

        .btn-auth-primary:hover i {
            transform: translateX(5px);
        }

        /* Social Links */
        .social-links {
            display: flex;
            justify-content: center;
            gap: 1rem;
            margin-top: 2rem;
        }

        .social-link {
            width: 45px;
            height: 45px;
            border-radius: 50%;
            background: var(--light-bg);
            display: flex;
            align-items: center;
            justify-content: center;
            color: var(--light-text);
            text-decoration: none;
            transition: all 0.3s ease;
            font-size: 1.1rem;
        }

        .social-link:hover {
            background: var(--secondary-color);
            color: white;
            transform: translateY(-3px);
        }

        .social-link.facebook:hover {
            background: #1877f2;
        }

        .social-link.instagram:hover {
            background: linear-gradient(45deg, #405de6, #5851db, #833ab4, #c13584, #e1306c, #fd1d1d);
        }

        /* Action Links */
        .auth-actions {
            display: flex;
            justify-content: center;
            gap: 2rem;
            padding-top: 2rem;
            margin-top: 2rem;
            border-top: 1px solid var(--border-color);
        }

        .auth-action-link {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            color: var(--secondary-color);
            text-decoration: none;
            font-size: 0.9rem;
            font-weight: 500;
            transition: all 0.3s ease;
            padding: 0.5rem 1rem;
            border-radius: 6px;
        }

        .auth-action-link:hover {
            background: rgba(135, 206, 235, 0.1);
            transform: translateX(5px);
        }

        .auth-action-link i {
            font-size: 0.9rem;
        }

        /* Responsive Design */
        @media (max-width: 1200px) {
            .auth-container {
                max-width: 1000px;
            }
            
            .auth-branding-panel {
                min-width: 400px;
            }
            
            .auth-form-panel {
                min-width: 500px;
            }
        }

        @media (max-width: 1024px) {
            .auth-container {
                flex-direction: column;
                min-height: auto;
            }
            
            .auth-branding-panel {
                min-width: 100%;
                min-height: 350px;
            }
            
            .auth-form-panel {
                min-width: 100%;
                padding: 2rem;
            }
        }

        @media (max-width: 768px) {
            body {
                padding: 1rem;
            }

            .auth-container {
                border-radius: 15px;
            }

            .otp-input {
                width: 50px;
                height: 60px;
                font-size: 1.8rem;
            }

            .form-title {
                font-size: 1.8rem;
            }
            
            .auth-branding-panel {
                min-height: 300px;
                padding: 1.5rem;
            }
            
            .brand-title {
                font-size: 2rem;
            }
            
            .auth-actions {
                flex-direction: column;
                gap: 1rem;
            }
        }

        @media (max-width: 480px) {
            .auth-form-panel {
                padding: 1.5rem;
            }

            .otp-inputs {
                gap: 0.5rem;
            }

            .otp-input {
                width: 45px;
                height: 55px;
                font-size: 1.5rem;
            }

            .brand-title {
                font-size: 1.8rem;
            }
            
            .logo-animation {
                width: 100px;
                height: 100px;
            }
            
            .logo-image {
                width: 60px;
                height: 60px;
            }
        }
    </style>
</head>
<body>
<div class="auth-container">
    <!-- Left Panel - Branding -->
    <div class="auth-branding-panel">
        <div class="branding-content">
            <div class="logo-container">
                <i class="fas fa-balance-scale logo-icon"></i>
            </div>
            <h1 class="brand-title">Prolex Legal System</h1>
            <p class="brand-tagline">Secure Legal Management Platform</p>
            
            <div class="brand-features">
                <div class="feature-item">
                    <i class="fas fa-shield-alt"></i>
                    <span>Two-Factor Authentication</span>
                </div>
                <div class="feature-item">
                    <i class="fas fa-clock"></i>
                    <span>Time-sensitive Codes</span>
                </div>
                <div class="feature-item">
                    <i class="fas fa-lock"></i>
                    <span>Secure Verification</span>
                </div>
            </div>

            <!-- OTP Information -->
            <div class="otp-info">
                <h4><i class="fas fa-info-circle"></i> OTP Security Tips</h4>
                <ul>
                    <li><i class="fas fa-check"></i> OTP expires in 10 minutes</li>
                    <li><i class="fas fa-check"></i> Never share your OTP with anyone</li>
                    <li><i class="fas fa-check"></i> Enter the code immediately upon receipt</li>
                    <li><i class="fas fa-check"></i> If you don't see the email in your inbox, check your spam or junk folder</li>
                </ul>
            </div>
        </div>
    </div>

    <!-- Right Panel - OTP Form -->
    <div class="auth-form-panel">
        <div class="auth-form-wrapper">
            <div class="form-header">
                <div class="form-header-icon">
                    <i class="fas fa-mobile-alt"></i>
                </div>
                <h2 class="form-title">Verify OTP</h2>
                <p class="form-subtitle">Enter the 6-digit code sent to your email</p>
            </div>

            <!-- Email Display -->
            <div class="email-display">
                <i class="fas fa-envelope"></i>
                <div>
                    <strong>Verification Email:</strong>
                    <div>{{ $email }}</div>
                </div>
            </div>

            @if (session('message'))
                <div class="alert alert-info">
                    <div class="alert-icon">
                        <i class="fas fa-info-circle"></i>
                    </div>
                    <div class="alert-content">
                        <strong>Information</strong>
                        <p>{{ session('message') }}</p>
                    </div>
                </div>
            @endif

            @if ($errors->any())
                <div class="alert alert-danger">
                    <div class="alert-icon">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                    <div class="alert-content">
                        <strong>Verification Failed</strong>
                        @foreach ($errors->all() as $error)
                            <p>{{ $error }}</p>
                        @endforeach
                    </div>
                </div>
            @endif

            <form method="POST" action="{{ route('password.verify') }}" class="otp-form" id="otpForm">
                @csrf
                <div class="otp-container">
                    <p class="otp-hint">Enter the 6-digit code from your email</p>
                    
                    <div class="otp-inputs" id="otpInputs">
                        @for ($i = 0; $i < 6; $i++)
                            <input type="text"
                                   class="otp-input"
                                   maxlength="1"
                                   data-index="{{ $i }}"
                                   inputmode="numeric"
                                   pattern="[0-9]*"
                                   autocomplete="off">
                        @endfor
                        <input type="hidden" id="otp" name="otp" value="">
                    </div>

                    <!-- Timer -->
                    <div class="timer-container">
                        <div class="timer" id="timer">10:00</div>
                    </div>

                    <!-- Resend Link -->
                    <div class="resend-link">
                        <a href="#" id="resendLink" class="disabled">
                            <i class="fas fa-redo"></i>
                            <span>Resend OTP</span>
                            <span id="resendCountdown">(60s)</span>
                        </a>
                    </div>
                </div>

                <button type="submit" class="btn-auth-primary" id="submitBtn" disabled>
                    <i class="fas fa-check-circle"></i> Verify OTP
                    <i class="fas fa-arrow-right"></i>
                </button>
            </form>

            <!-- Social Links -->
            <div class="social-links">
                <a href="https://www.facebook.com/people/Melsoft-Zimbabwe/100072059177475/" target="_blank" class="social-link facebook">
                    <i class="fab fa-facebook-f"></i>
                </a>
                <a href="https://www.instagram.com/melsofttech/" target="_blank" class="social-link instagram">
                    <i class="fab fa-instagram"></i>
                </a>
            </div>

            <!-- Action Links -->
            <div class="auth-actions">
                <a href="{{ route('password.request') }}" class="auth-action-link">
                    <i class="fas fa-arrow-left"></i>
                    <span>Back to Forgot Password</span>
                </a>
                <a href="{{ route('login') }}" class="auth-action-link">
                    <i class="fas fa-sign-in-alt"></i>
                    <span>Back to Login</span>
                </a>
            </div>
        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
    document.addEventListener('DOMContentLoaded', function() {
        const otpInputs = document.querySelectorAll('.otp-input');
        const hiddenOtpInput = document.getElementById('otp');
        const submitBtn = document.getElementById('submitBtn');
        const otpForm = document.getElementById('otpForm');
        const timerElement = document.getElementById('timer');
        const resendLink = document.getElementById('resendLink');
        const resendCountdown = document.getElementById('resendCountdown');
        
        let otpCode = '';
        let otpTimer = 600; // 10 minutes in seconds
        let resendTimer = 60; // 60 seconds for resend
        let timerInterval;
        let resendInterval;

        // Initialize OTP input handling
        otpInputs.forEach((input, index) => {
            // Focus first input
            if (index === 0) {
                input.focus();
            }

            // Handle input
            input.addEventListener('input', function(e) {
                const value = e.target.value;
                
                // Allow only numbers
                if (!/^\d*$/.test(value)) {
                    e.target.value = '';
                    return;
                }

                // If a number is entered, move to next input
                if (value && index < otpInputs.length - 1) {
                    otpInputs[index + 1].focus();
                }

                // Update OTP code
                updateOTPCode();
            });

            // Handle backspace
            input.addEventListener('keydown', function(e) {
                if (e.key === 'Backspace' || e.key === 'Delete') {
                    e.preventDefault();
                    
                    // If input is empty, move to previous input
                    if (!this.value && index > 0) {
                        otpInputs[index - 1].focus();
                        otpInputs[index - 1].value = '';
                    } else {
                        this.value = '';
                    }
                    
                    updateOTPCode();
                }
            });

            // Handle paste
            input.addEventListener('paste', function(e) {
                e.preventDefault();
                const pasteData = e.clipboardData.getData('text');
                const numbers = pasteData.replace(/\D/g, '');
                
                // Fill inputs with pasted numbers
                numbers.split('').forEach((num, i) => {
                    if (otpInputs[i]) {
                        otpInputs[i].value = num;
                        otpInputs[i].classList.add('filled');
                    }
                });
                
                // Focus last filled input or next empty
                const lastFilledIndex = Math.min(numbers.length, otpInputs.length - 1);
                otpInputs[lastFilledIndex].focus();
                
                updateOTPCode();
            });

            // Handle focus
            input.addEventListener('focus', function() {
                this.select();
                this.classList.add('filled');
            });

            // Handle blur
            input.addEventListener('blur', function() {
                if (!this.value) {
                    this.classList.remove('filled');
                }
            });
        });

        function updateOTPCode() {
            otpCode = '';
            otpInputs.forEach(input => {
                otpCode += input.value;
                if (input.value) {
                    input.classList.add('filled');
                } else {
                    input.classList.remove('filled');
                }
            });
            
            hiddenOtpInput.value = otpCode;
            submitBtn.disabled = otpCode.length !== 6;
        }

        // Timer functionality
        function startOTPTimer() {
            clearInterval(timerInterval);
            otpTimer = 600; // Reset to 10 minutes
            
            timerInterval = setInterval(() => {
                otpTimer--;
                
                const minutes = Math.floor(otpTimer / 60);
                const seconds = otpTimer % 60;
                timerElement.textContent = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
                
                // Add warning style when less than 2 minutes remaining
                if (otpTimer < 120) {
                    timerElement.classList.add('expiring');
                }
                
                // Expire OTP
                if (otpTimer <= 0) {
                    clearInterval(timerInterval);
                    timerElement.textContent = 'Expired';
                    timerElement.classList.add('expiring');
                    submitBtn.disabled = true;
                    showNotification('OTP has expired. Please request a new one.', 'error');
                }
            }, 1000);
        }

        // Resend timer functionality
        function startResendTimer() {
            clearInterval(resendInterval);
            resendLink.classList.add('disabled');
            resendTimer = 60;
            
            resendInterval = setInterval(() => {
                resendTimer--;
                resendCountdown.textContent = `(${resendTimer}s)`;
                
                if (resendTimer <= 0) {
                    clearInterval(resendInterval);
                    resendLink.classList.remove('disabled');
                    resendCountdown.textContent = '';
                }
            }, 1000);
        }

        // Handle resend OTP
        resendLink.addEventListener('click', function(e) {
            e.preventDefault();
            
            if (this.classList.contains('disabled')) {
                return;
            }
            
            // Show loading state
            const originalContent = this.innerHTML;
            this.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Resending...';
            
            // Simulate API call
            setTimeout(() => {
                // Reset OTP inputs
                otpInputs.forEach(input => {
                    input.value = '';
                    input.classList.remove('filled');
                });
                otpInputs[0].focus();
                updateOTPCode();
                
                // Reset timers
                startOTPTimer();
                startResendTimer();
                
                // Show success message
                showNotification('New OTP has been sent to your email.', 'success');
                
                // Restore button content
                this.innerHTML = originalContent;
            }, 1500);
        });

        // Form submission
        otpForm.addEventListener('submit', function(e) {
            if (otpCode.length !== 6) {
                e.preventDefault();
                showNotification('Please enter the complete 6-digit OTP.', 'error');
                return;
            }
            
            if (otpTimer <= 0) {
                e.preventDefault();
                showNotification('OTP has expired. Please request a new one.', 'error');
                return;
            }
            
            // Show loading state
            submitBtn.disabled = true;
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Verifying...';
        });

        // Notification function
        function showNotification(message, type) {
            // Remove existing notifications
            const existingAlert = document.querySelector('.notification-alert');
            if (existingAlert) {
                existingAlert.remove();
            }
            
            // Create notification
            const alert = document.createElement('div');
            alert.className = `notification-alert alert alert-${type === 'error' ? 'danger' : 'success'}`;
            alert.innerHTML = `
                <div class="alert-icon">
                    <i class="fas fa-${type === 'error' ? 'exclamation-triangle' : 'check-circle'}"></i>
                </div>
                <div class="alert-content">
                    <strong>${type === 'error' ? 'Error' : 'Success'}</strong>
                    <p>${message}</p>
                </div>
            `;
            
            // Add styles for notification
            alert.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                z-index: 9999;
                animation: slideIn 0.3s ease;
                min-width: 300px;
                max-width: 400px;
            `;
            
            document.body.appendChild(alert);
            
            // Auto remove after 5 seconds
            setTimeout(() => {
                alert.style.animation = 'slideOut 0.3s ease forwards';
                setTimeout(() => alert.remove(), 300);
            }, 5000);
        }

        // Start timers on load
        startOTPTimer();
        startResendTimer();

        // Auto-submit when all inputs are filled
        otpInputs.forEach(input => {
            input.addEventListener('input', () => {
                if (otpCode.length === 6) {
                    setTimeout(() => {
                        submitBtn.focus();
                    }, 100);
                }
            });
        });
    });
</script>
</body>
</html>