@extends('layouts.master')

@section('title', 'Prolex Chat - Secure Legal Messaging')

@section('styles')
<style>
/* ========================================
   MODERN CHAT APPLICATION - IMPROVED VERSION
   ======================================== */

:root {
  /* Light theme colors */
  --bg-primary: #ffffff;
  --bg-secondary: #f8fafc;
  --bg-tertiary: #f1f5f9;
  --surface: #ffffff;
  --surface-hover: #f8fafc;
  
  /* Accent colors */
  --primary: #0369a1;
  --primary-light: #0ea5e9;
  --primary-dark: #075985;
  --primary-bg: #e0f2fe;
  --primary-bg-light: #f0f9ff;
  
  /* Status colors */
  --success: #10b981;
  --success-light: #d1fae5;
  --warning: #f59e0b;
  --warning-light: #fef3c7;
  --danger: #ef4444;
  --danger-light: #fee2e2;
  --info: #3b82f6;
  --info-light: #dbeafe;
  
  /* Text colors */
  --text-primary: #0f172a;
  --text-secondary: #475569;
  --text-tertiary: #94a3b8;
  --text-inverse: #ffffff;
  
  /* Border colors */
  --border-light: #e2e8f0;
  --border-medium: #cbd5e1;
  --border-dark: #94a3b8;
  
  /* Shadow */
  --shadow-sm: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
  --shadow-md: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
  --shadow-lg: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
  --shadow-xl: 0 20px 25px -5px rgba(0, 0, 0, 0.1);
  
  /* Message colors */
  --msg-sent-bg: #0ea5e9;
  --msg-sent-text: #ffffff;
  --msg-received-bg: #f1f5f9;
  --msg-received-text: #0f172a;
  
  /* Status indicators */
  --online: #10b981;
  --away: #f59e0b;
  --offline: #94a3b8;
  
  /* Transitions */
  --transition-fast: 150ms ease;
  --transition-base: 200ms ease;
  --transition-slow: 300ms ease;
  
  /* Spacing */
  --space-xs: 0.25rem;
  --space-sm: 0.5rem;
  --space-md: 1rem;
  --space-lg: 1.5rem;
  --space-xl: 2rem;
  
  /* Border radius */
  --radius-sm: 0.375rem;
  --radius-md: 0.5rem;
  --radius-lg: 0.75rem;
  --radius-xl: 1rem;
  --radius-full: 9999px;
}

/* Dark theme */
.chat-dark {
  --bg-primary: #0f172a;
  --bg-secondary: #1e293b;
  --bg-tertiary: #334155;
  --surface: #1e293b;
  --surface-hover: #334155;
  
  --primary: #38bdf8;
  --primary-light: #7dd3fc;
  --primary-dark: #0284c7;
  --primary-bg: #0c4a6e;
  --primary-bg-light: #075985;
  
  --text-primary: #f1f5f9;
  --text-secondary: #cbd5e1;
  --text-tertiary: #64748b;
  
  --border-light: #334155;
  --border-medium: #475569;
  --border-dark: #64748b;
  
  --msg-sent-bg: #0284c7;
  --msg-received-bg: #334155;
  --msg-received-text: #f1f5f9;
}

/* Reset and base styles */
* {
  margin: 0;
  padding: 0;
  box-sizing: border-box;
}

.chat-root {
  font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
  color: var(--text-primary);
  background: var(--bg-primary);
  transition: background var(--transition-base), color var(--transition-base);
}

/* Fullscreen mode */
.chat-fullscreen .sidebar,
.chat-fullscreen .topbar,
.chat-fullscreen .main-content {
  display: none !important;
}

.chat-fullscreen .chat-container {
  position: fixed;
  top: 0;
  left: 0;
  width: 100vw;
  height: 100vh;
  z-index: 9999;
  margin: 0 !important;
}

/* Main container */
.chat-container {
  display: flex;
  width: 100%;
  height: calc(100vh - 80px);
  margin-top: 80px;
  background: var(--bg-primary);
  overflow: hidden;
}

/* Resizer */
.chat-resizer {
  width: 4px;
  background: transparent;
  cursor: col-resize;
  position: relative;
  transition: background var(--transition-fast);
  flex-shrink: 0;
}

.chat-resizer::before {
  content: '';
  position: absolute;
  top: 0;
  left: -4px;
  right: -4px;
  bottom: 0;
}

.chat-resizer:hover {
  background: var(--primary-light);
}

.chat-resizer.resizing {
  background: var(--primary);
}

/* ========================================
   LEFT PANEL - CONTACTS & GROUPS
   ======================================== */

.chat-left {
  width: 320px;
  min-width: 280px;
  max-width: 500px;
  background: var(--surface);
  border-right: 1px solid var(--border-light);
  display: flex;
  flex-direction: column;
  overflow: hidden;
}

/* Header */
.chat-sidebar-header {
  padding: var(--space-lg);
  border-bottom: 1px solid var(--border-light);
  background: var(--surface);
}

.chat-brand {
  display: flex;
  align-items: center;
  gap: var(--space-md);
  margin-bottom: var(--space-lg);
}

.chat-logo {
  width: 48px;
  height: 48px;
  border-radius: var(--radius-lg);
  background: linear-gradient(135deg, var(--primary) 0%, var(--primary-light) 100%);
  display: flex;
  align-items: center;
  justify-content: center;
  color: var(--text-inverse);
  font-weight: 700;
  font-size: 20px;
  box-shadow: var(--shadow-lg);
}

.chat-brand-info h1 {
  font-size: 18px;
  font-weight: 700;
  color: var(--text-primary);
  margin-bottom: 2px;
}

.chat-brand-info p {
  font-size: 12px;
  color: var(--text-tertiary);
}

/* Tabs */
.chat-tabs {
  display: flex;
  gap: var(--space-xs);
  background: var(--bg-secondary);
  padding: 4px;
  border-radius: var(--radius-md);
}

.chat-tab {
  flex: 1;
  padding: var(--space-sm) var(--space-md);
  border-radius: var(--radius-sm);
  border: none;
  background: transparent;
  color: var(--text-secondary);
  cursor: pointer;
  font-size: 14px;
  font-weight: 600;
  transition: all var(--transition-fast);
  position: relative;
}

.chat-tab:hover {
  color: var(--text-primary);
  background: var(--surface-hover);
}

.chat-tab.active {
  background: var(--primary);
  color: var(--text-inverse);
  box-shadow: var(--shadow-sm);
}

/* Search */
.chat-search {
  padding: var(--space-md) var(--space-lg);
  border-bottom: 1px solid var(--border-light);
}

.chat-search-wrapper {
  position: relative;
  display: flex;
  align-items: center;
}

.chat-search-icon {
  position: absolute;
  left: var(--space-md);
  color: var(--text-tertiary);
  pointer-events: none;
}

.chat-search input {
  width: 100%;
  padding: 10px var(--space-md) 10px 40px;
  border-radius: var(--radius-lg);
  border: 1px solid var(--border-light);
  background: var(--bg-secondary);
  outline: none;
  font-size: 14px;
  color: var(--text-primary);
  transition: all var(--transition-fast);
}

.chat-search input:focus {
  border-color: var(--primary);
  background: var(--surface);
  box-shadow: 0 0 0 3px var(--primary-bg-light);
}

.chat-search input::placeholder {
  color: var(--text-tertiary);
}

/* Contacts list */
.chat-contacts {
  flex: 1;
  overflow-y: auto;
  padding: var(--space-sm);
}

.chat-contacts::-webkit-scrollbar {
  width: 6px;
}

.chat-contacts::-webkit-scrollbar-track {
  background: transparent;
}

.chat-contacts::-webkit-scrollbar-thumb {
  background: var(--border-medium);
  border-radius: var(--radius-full);
}

.chat-contacts::-webkit-scrollbar-thumb:hover {
  background: var(--border-dark);
}

/* Contact item */
.chat-contact {
  display: flex;
  align-items: center;
  gap: var(--space-md);
  padding: var(--space-md);
  border-radius: var(--radius-lg);
  cursor: pointer;
  transition: all var(--transition-fast);
  margin-bottom: var(--space-xs);
  position: relative;
}

.chat-contact:hover {
  background: var(--surface-hover);
}

.chat-contact.active {
  background: var(--primary-bg-light);
  border-left: 3px solid var(--primary);
}

.chat-contact.active .chat-cname {
  color: var(--primary);
  font-weight: 600;
}

.chat-avatar-wrapper {
  position: relative;
  flex-shrink: 0;
}

.chat-avatar {
  width: 48px;
  height: 48px;
  border-radius: var(--radius-lg);
  background: linear-gradient(135deg, var(--primary-bg) 0%, var(--primary-bg-light) 100%);
  display: flex;
  align-items: center;
  justify-content: center;
  font-weight: 700;
  color: var(--primary);
  font-size: 16px;
  border: 2px solid var(--border-light);
}

.chat-contact.active .chat-avatar {
  border-color: var(--primary);
  background: linear-gradient(135deg, var(--primary) 0%, var(--primary-light) 100%);
  color: var(--text-inverse);
}

.chat-status-indicator {
  position: absolute;
  bottom: 0;
  right: 0;
  width: 12px;
  height: 12px;
  border-radius: var(--radius-full);
  border: 2px solid var(--surface);
  background: var(--online);
}

.chat-status-indicator.away {
  background: var(--away);
}

.chat-status-indicator.offline {
  background: var(--offline);
}

.chat-cinfo {
  flex: 1;
  min-width: 0;
}

.chat-cinfo-header {
  display: flex;
  justify-content: space-between;
  align-items: baseline;
  margin-bottom: 4px;
}

.chat-cname {
  font-weight: 600;
  font-size: 15px;
  color: var(--text-primary);
}

.chat-time {
  font-size: 12px;
  color: var(--text-tertiary);
}

.chat-cmeta {
  font-size: 13px;
  color: var(--text-secondary);
  white-space: nowrap;
  overflow: hidden;
  text-overflow: ellipsis;
  display: flex;
  align-items: center;
  gap: var(--space-xs);
}

.chat-unread-badge {
  background: var(--primary);
  color: var(--text-inverse);
  font-size: 11px;
  font-weight: 700;
  padding: 2px 6px;
  border-radius: var(--radius-full);
  margin-left: auto;
}

/* Empty state */
.chat-empty-state {
  text-align: center;
  padding: var(--space-xl);
  color: var(--text-tertiary);
}

.chat-empty-state-icon {
  font-size: 48px;
  margin-bottom: var(--space-md);
  opacity: 0.5;
}

.chat-empty-state p {
  font-size: 14px;
}

/* ========================================
   MIDDLE PANEL - CHAT AREA
   ======================================== */

.chat-middle {
  flex: 1;
  display: flex;
  flex-direction: column;
  background: var(--bg-secondary);
  min-width: 0;
}

/* Chat header */
.chat-header {
  height: 70px;
  display: flex;
  align-items: center;
  justify-content: space-between;
  padding: 0 var(--space-lg);
  border-bottom: 1px solid var(--border-light);
  background: var(--surface);
  box-shadow: var(--shadow-sm);
}

.chat-header-left {
  display: flex;
  align-items: center;
  gap: var(--space-md);
  min-width: 0;
}

.chat-header .chat-avatar-sm {
  width: 44px;
  height: 44px;
  border-radius: var(--radius-lg);
  background: linear-gradient(135deg, var(--primary-bg) 0%, var(--primary-bg-light) 100%);
  display: flex;
  align-items: center;
  justify-content: center;
  font-weight: 700;
  color: var(--primary);
  border: 2px solid var(--border-light);
  flex-shrink: 0;
}

.chat-header-info {
  min-width: 0;
}

.chat-title {
  font-weight: 700;
  font-size: 16px;
  color: var(--text-primary);
  margin-bottom: 2px;
}

.chat-subtitle {
  font-size: 13px;
  color: var(--text-secondary);
  display: flex;
  align-items: center;
  gap: var(--space-xs);
}

.chat-typing-indicator {
  display: inline-flex;
  align-items: center;
  gap: 2px;
  color: var(--primary);
}

.chat-typing-dot {
  width: 4px;
  height: 4px;
  border-radius: var(--radius-full);
  background: currentColor;
  animation: typing 1.4s infinite;
}

.chat-typing-dot:nth-child(2) {
  animation-delay: 0.2s;
}

.chat-typing-dot:nth-child(3) {
  animation-delay: 0.4s;
}

@keyframes typing {
  0%, 60%, 100% {
    opacity: 0.3;
    transform: translateY(0);
  }
  30% {
    opacity: 1;
    transform: translateY(-4px);
  }
}

.chat-header-right {
  display: flex;
  align-items: center;
  gap: var(--space-sm);
}

.chat-icon-btn {
  width: 36px;
  height: 36px;
  display: flex;
  align-items: center;
  justify-content: center;
  background: transparent;
  border: 1px solid var(--border-light);
  border-radius: var(--radius-md);
  cursor: pointer;
  color: var(--text-secondary);
  transition: all var(--transition-fast);
  font-size: 16px;
}

.chat-icon-btn:hover {
  background: var(--surface-hover);
  color: var(--text-primary);
  transform: translateY(-1px);
}

.chat-icon-btn:active {
  transform: translateY(0);
}

.chat-icon-btn.primary {
  background: var(--primary);
  color: var(--text-inverse);
  border-color: var(--primary);
}

.chat-icon-btn.primary:hover {
  background: var(--primary-dark);
}

/* Messages area */
.chat-area {
  flex: 1;
  padding: var(--space-lg);
  overflow-y: auto;
  display: flex;
  flex-direction: column;
  gap: var(--space-md);
  background: var(--bg-secondary);
}

.chat-area::-webkit-scrollbar {
  width: 8px;
}

.chat-area::-webkit-scrollbar-track {
  background: transparent;
}

.chat-area::-webkit-scrollbar-thumb {
  background: var(--border-medium);
  border-radius: var(--radius-full);
}

.chat-area::-webkit-scrollbar-thumb:hover {
  background: var(--border-dark);
}

/* Date divider */
.chat-date-divider {
  display: flex;
  align-items: center;
  gap: var(--space-md);
  margin: var(--space-lg) 0;
}

.chat-date-divider::before,
.chat-date-divider::after {
  content: '';
  flex: 1;
  height: 1px;
  background: var(--border-light);
}

.chat-date-divider span {
  font-size: 12px;
  color: var(--text-tertiary);
  font-weight: 600;
  padding: 4px var(--space-md);
  background: var(--bg-tertiary);
  border-radius: var(--radius-full);
}

/* Message */
.chat-message {
  max-width: 70%;
  display: flex;
  flex-direction: column;
  gap: var(--space-xs);
  animation: messageSlideIn 0.2s ease;
}

@keyframes messageSlideIn {
  from {
    opacity: 0;
    transform: translateY(10px);
  }
  to {
    opacity: 1;
    transform: translateY(0);
  }
}

.chat-message.sent {
  align-self: flex-end;
  align-items: flex-end;
}

.chat-message.received {
  align-self: flex-start;
  align-items: flex-start;
}

.chat-message-sender {
  font-size: 12px;
  font-weight: 600;
  color: var(--text-secondary);
  padding: 0 var(--space-md);
}

.chat-message-bubble {
  padding: 10px 14px;
  border-radius: var(--radius-lg);
  font-size: 14px;
  line-height: 1.5;
  box-shadow: var(--shadow-sm);
  word-wrap: break-word;
  position: relative;
}

.chat-message.sent .chat-message-bubble {
  background: var(--msg-sent-bg);
  color: var(--msg-sent-text);
  border-bottom-right-radius: 4px;
}

.chat-message.received .chat-message-bubble {
  background: var(--msg-received-bg);
  color: var(--msg-received-text);
  border-bottom-left-radius: 4px;
}

.chat-message-meta {
  display: flex;
  align-items: center;
  gap: var(--space-xs);
  font-size: 11px;
  color: var(--text-tertiary);
  padding: 0 var(--space-sm);
}

.chat-message.sent .chat-message-meta {
  color: rgba(255, 255, 255, 0.7);
}

.chat-message-status {
  display: flex;
  align-items: center;
  gap: 2px;
}

.chat-tick {
  font-size: 14px;
}

.chat-tick.sent {
  color: rgba(255, 255, 255, 0.7);
}

.chat-tick.delivered {
  color: rgba(255, 255, 255, 0.9);
}

.chat-tick.read {
  color: var(--success);
}

/* System message */
.chat-system-message {
  align-self: center;
  text-align: center;
  font-size: 12px;
  color: var(--text-tertiary);
  background: var(--bg-tertiary);
  padding: 6px var(--space-md);
  border-radius: var(--radius-full);
  max-width: 80%;
}

/* Input area */
.chat-input-area {
  display: flex;
  align-items: flex-end;
  gap: var(--space-md);
  padding: var(--space-lg);
  border-top: 1px solid var(--border-light);
  background: var(--surface);
}

.chat-input-wrapper {
  flex: 1;
  display: flex;
  flex-direction: column;
  gap: var(--space-sm);
}

.chat-input-actions {
  display: flex;
  gap: var(--space-sm);
}

.chat-input-container {
  position: relative;
  display: flex;
  align-items: flex-end;
  background: var(--bg-secondary);
  border: 2px solid var(--border-light);
  border-radius: var(--radius-xl);
  padding: var(--space-sm);
  transition: all var(--transition-fast);
}

.chat-input-container:focus-within {
  border-color: var(--primary);
  box-shadow: 0 0 0 3px var(--primary-bg-light);
}

.chat-input-container textarea {
  flex: 1;
  border: none;
  outline: none;
  background: transparent;
  font-size: 14px;
  color: var(--text-primary);
  resize: none;
  max-height: 120px;
  padding: 8px var(--space-md);
  font-family: inherit;
}

.chat-input-container textarea::placeholder {
  color: var(--text-tertiary);
}

.chat-attachment-btn {
  width: 32px;
  height: 32px;
  display: flex;
  align-items: center;
  justify-content: center;
  background: transparent;
  border: none;
  border-radius: var(--radius-md);
  cursor: pointer;
  color: var(--text-secondary);
  transition: all var(--transition-fast);
}

.chat-attachment-btn:hover {
  background: var(--surface-hover);
  color: var(--primary);
}

.chat-send {
  width: 44px;
  height: 44px;
  display: flex;
  align-items: center;
  justify-content: center;
  background: var(--primary);
  color: var(--text-inverse);
  border: none;
  border-radius: var(--radius-lg);
  cursor: pointer;
  font-weight: 600;
  transition: all var(--transition-fast);
  box-shadow: var(--shadow-md);
}

.chat-send:hover {
  background: var(--primary-dark);
  transform: translateY(-2px);
  box-shadow: var(--shadow-lg);
}

.chat-send:active {
  transform: translateY(0);
}

.chat-send:disabled {
  opacity: 0.5;
  cursor: not-allowed;
  transform: none;
}

/* ========================================
   RIGHT PANEL - ACTIVITY & INFO
   ======================================== */

.chat-right {
  width: 280px;
  min-width: 240px;
  max-width: 400px;
  background: var(--surface);
  border-left: 1px solid var(--border-light);
  display: flex;
  flex-direction: column;
  overflow-y: auto;
}

.chat-right::-webkit-scrollbar {
  width: 6px;
}

.chat-right::-webkit-scrollbar-track {
  background: transparent;
}

.chat-right::-webkit-scrollbar-thumb {
  background: var(--border-medium);
  border-radius: var(--radius-full);
}

.chat-section {
  padding: var(--space-lg);
  border-bottom: 1px solid var(--border-light);
}

.chat-section:last-child {
  border-bottom: none;
}

.chat-section-header {
  display: flex;
  align-items: center;
  justify-content: space-between;
  margin-bottom: var(--space-md);
}

.chat-section h4 {
  font-size: 14px;
  font-weight: 700;
  color: var(--text-primary);
  display: flex;
  align-items: center;
  gap: var(--space-sm);
}

.chat-section-icon {
  font-size: 18px;
}

.chat-section-badge {
  background: var(--primary);
  color: var(--text-inverse);
  font-size: 11px;
  font-weight: 700;
  padding: 2px 8px;
  border-radius: var(--radius-full);
}

/* Notifications */
.chat-notification-item {
  padding: var(--space-md);
  border-radius: var(--radius-md);
  background: var(--bg-secondary);
  margin-bottom: var(--space-sm);
  cursor: pointer;
  transition: all var(--transition-fast);
  border-left: 3px solid transparent;
}

.chat-notification-item:hover {
  background: var(--surface-hover);
  border-left-color: var(--primary);
  transform: translateX(2px);
}

.chat-notification-item:last-child {
  margin-bottom: 0;
}

.chat-notification-title {
  font-size: 13px;
  font-weight: 600;
  color: var(--text-primary);
  margin-bottom: 4px;
}

.chat-notification-text {
  font-size: 12px;
  color: var(--text-secondary);
}

.chat-notification-time {
  font-size: 11px;
  color: var(--text-tertiary);
  margin-top: 4px;
}

/* Requests */
.chat-request-item {
  display: flex;
  align-items: center;
  gap: var(--space-md);
  padding: var(--space-md);
  border-radius: var(--radius-md);
  background: var(--bg-secondary);
  margin-bottom: var(--space-sm);
  transition: all var(--transition-fast);
}

.chat-request-item:hover {
  background: var(--surface-hover);
}

.chat-request-item:last-child {
  margin-bottom: 0;
}

.chat-request-avatar {
  width: 36px;
  height: 36px;
  border-radius: var(--radius-md);
  background: linear-gradient(135deg, var(--primary-bg) 0%, var(--primary-bg-light) 100%);
  display: flex;
  align-items: center;
  justify-content: center;
  font-weight: 700;
  color: var(--primary);
  font-size: 14px;
  flex-shrink: 0;
}

.chat-request-info {
  flex: 1;
  min-width: 0;
}

.chat-request-name {
  font-size: 13px;
  font-weight: 600;
  color: var(--text-primary);
  margin-bottom: 2px;
}

.chat-request-text {
  font-size: 12px;
  color: var(--text-secondary);
}

.chat-request-actions {
  display: flex;
  gap: var(--space-xs);
  flex-shrink: 0;
}

.chat-request-btn {
  width: 28px;
  height: 28px;
  display: flex;
  align-items: center;
  justify-content: center;
  border: none;
  border-radius: var(--radius-md);
  cursor: pointer;
  font-size: 14px;
  transition: all var(--transition-fast);
}

.chat-approve-btn {
  background: var(--success-light);
  color: var(--success);
}

.chat-approve-btn:hover {
  background: var(--success);
  color: var(--text-inverse);
}

.chat-decline-btn {
  background: var(--danger-light);
  color: var(--danger);
}

.chat-decline-btn:hover {
  background: var(--danger);
  color: var(--text-inverse);
}

/* ========================================
   RESPONSIVE DESIGN
   ======================================== */

@media (max-width: 1024px) {
  .chat-right {
    display: none;
  }
}

@media (max-width: 768px) {
  .chat-container {
    margin-top: 60px;
    height: calc(100vh - 60px);
  }
  
  .chat-left {
    position: absolute;
    left: 0;
    top: 0;
    height: 100%;
    z-index: 10;
    transform: translateX(-100%);
    transition: transform var(--transition-base);
    box-shadow: var(--shadow-xl);
  }
  
  .chat-left.mobile-open {
    transform: translateX(0);
  }
  
  .chat-middle {
    width: 100%;
  }
  
  .chat-resizer {
    display: none;
  }
}

/* ========================================
   UTILITIES
   ======================================== */

.hidden {
  display: none !important;
}

.sr-only {
  position: absolute;
  width: 1px;
  height: 1px;
  padding: 0;
  margin: -1px;
  overflow: hidden;
  clip: rect(0, 0, 0, 0);
  white-space: nowrap;
  border-width: 0;
}

/* Loading animation */
@keyframes spin {
  from {
    transform: rotate(0deg);
  }
  to {
    transform: rotate(360deg);
  }
}

.loading {
  animation: spin 1s linear infinite;
}

/* Pulse animation */
@keyframes pulse {
  0%, 100% {
    opacity: 1;
  }
  50% {
    opacity: 0.5;
  }
}

.pulse {
  animation: pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite;
}
</style>
@endsection

@section('content')
<div class="chat-root" id="chatRoot">
  <div class="chat-container" id="chatContainer">

    <!-- LEFT PANEL - Contacts & Groups -->
    <aside class="chat-left" id="chatLeft" aria-label="Contacts and groups">
      <div class="chat-sidebar-header">
        <div class="chat-brand">
          <div class="chat-logo">P</div>
          <div class="chat-brand-info">
            <h1>Prolex</h1>
            <p>Secure Legal Messaging</p>
          </div>
        </div>

        <div class="chat-tabs">
          <button class="chat-tab active" id="contactsTab" data-tab="contacts">
            <span>Contacts</span>
          </button>
          <button class="chat-tab" id="groupsTab" data-tab="groups">
            <span>Groups</span>
          </button>
          <button class="chat-tab" id="optionsTab" data-tab="options">
            <span>Settings</span>
          </button>
        </div>
      </div>

      <div class="chat-search">
        <div class="chat-search-wrapper">
          <span class="chat-search-icon">🔍</span>
          <input 
            id="searchInput" 
            type="search" 
            placeholder="Search..." 
            aria-label="Search contacts and groups"
            autocomplete="off"
          />
        </div>
      </div>

      <nav class="chat-contacts" id="contactsList" aria-label="Contacts list">
        <!-- Dynamically populated -->
      </nav>
    </aside>

    <!-- RESIZER 1 -->
    <div class="chat-resizer" id="resizer1" aria-hidden="true"></div>

    <!-- MIDDLE PANEL - Chat Area -->
    <main class="chat-middle" role="main">
      <header class="chat-header">
        <div class="chat-header-left">
          <button class="chat-icon-btn" id="toggleSidebarBtn" title="Toggle sidebar" style="display: none;">
            ☰
          </button>
          <div class="chat-avatar-sm" id="chatAvatar">—</div>
          <div class="chat-header-info">
            <div class="chat-title" id="chatTitle">Select a contact</div>
            <div class="chat-subtitle" id="chatSubtitle">
              <span id="chatStatus">No conversation selected</span>
              <span class="chat-typing-indicator hidden" id="typingIndicator">
                <span class="chat-typing-dot"></span>
                <span class="chat-typing-dot"></span>
                <span class="chat-typing-dot"></span>
              </span>
            </div>
          </div>
        </div>

        <div class="chat-header-right">
          <button class="chat-icon-btn" id="searchMsgBtn" title="Search messages" aria-label="Search messages">
            🔍
          </button>
          <button class="chat-icon-btn" id="callBtn" title="Audio call" aria-label="Audio call">
            📞
          </button>
          <button class="chat-icon-btn" id="videoBtn" title="Video call" aria-label="Video call">
            📹
          </button>
          <button class="chat-icon-btn" id="infoBtn" title="Conversation info" aria-label="Conversation info">
            ℹ️
          </button>
          <button class="chat-icon-btn primary" id="themeBtn" title="Toggle theme" aria-label="Toggle theme">
            🌙
          </button>
        </div>
      </header>

      <section class="chat-area" id="chatArea" aria-live="polite" aria-label="Messages">
        <div class="chat-system-message" id="welcomeMessage">
          Welcome to Prolex Secure Chat. Select a contact to start messaging.
        </div>
      </section>

      <div class="chat-input-area">
        <div class="chat-input-wrapper">
          <div class="chat-input-container">
            <button class="chat-attachment-btn" id="attachBtn" title="Attach file" aria-label="Attach file">
              📎
            </button>
            <textarea 
              id="messageInput" 
              placeholder="Type a message..." 
              aria-label="Message input"
              rows="1"
            ></textarea>
          </div>
        </div>
        <button class="chat-send" id="sendBtn" aria-label="Send message" disabled>
          <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M22 2L11 13M22 2l-7 20-4-9-9-4 20-7z"/>
          </svg>
        </button>
      </div>
    </main>

    <!-- RESIZER 2 -->
    <div class="chat-resizer" id="resizer2" aria-hidden="true"></div>

    <!-- RIGHT PANEL - Activity & Info -->
    <aside class="chat-right" id="chatRight" aria-label="Activity panel">
      <div class="chat-section">
        <div class="chat-section-header">
          <h4>
            <span class="chat-section-icon">🔔</span>
            Notifications
          </h4>
          <span class="chat-section-badge" id="notifBadge">0</span>
        </div>
        <div id="notificationsList">
          <div class="chat-empty-state">
            <div class="chat-empty-state-icon">🔕</div>
            <p>No notifications</p>
          </div>
        </div>
      </div>

      <div class="chat-section">
        <div class="chat-section-header">
          <h4>
            <span class="chat-section-icon">👥</span>
            Requests
          </h4>
          <span class="chat-section-badge" id="requestBadge">0</span>
        </div>
        <div id="requestsList">
          <div class="chat-empty-state">
            <div class="chat-empty-state-icon">✓</div>
            <p>No pending requests</p>
          </div>
        </div>
      </div>

      <div class="chat-section">
        <div class="chat-section-header">
          <h4>
            <span class="chat-section-icon">📁</span>
            Shared Files
          </h4>
        </div>
        <div class="chat-empty-state">
          <div class="chat-empty-state-icon">📄</div>
          <p>No shared files</p>
        </div>
      </div>
    </aside>

  </div>
</div>
@endsection

@push('scripts')
<script>
// ========================================
// PROLEX SECURE CHAT - IMPROVED VERSION
// ========================================

(function() {
  'use strict';

  // ========================================
  // CONFIGURATION & DATA
  // ========================================

  const CONFIG = {
    autoReplyDelay: 2000,
    typingIndicatorDelay: 1500,
    messageAnimation: true,
    soundEnabled: true,
    dateFormat: { hour: '2-digit', minute: '2-digit' },
  };

  // Mock data - replace with real API calls in production
  let contacts = [
    { 
      id: 1, 
      name: 'Jeff Reynolds', 
      initials: 'JR', 
      status: 'online',
      role: 'Senior Partner',
      lastSeen: new Date(),
      messages: [
        { id: 1, timestamp: new Date(Date.now() - 3600000), direction: 'received', text: 'Good morning, Ashley. I\'ve uploaded the contract documents to the secure folder.', status: 'read' },
        { id: 2, timestamp: new Date(Date.now() - 3000000), direction: 'sent', text: 'Thanks Jeff. I\'ll review them this afternoon.', status: 'read' },
        { id: 3, timestamp: new Date(Date.now() - 1800000), direction: 'received', text: 'Perfect. Let me know if you have any questions.', status: 'read' }
      ]
    },
    { 
      id: 2, 
      name: 'Henry Clarke', 
      initials: 'HC', 
      status: 'away',
      role: 'Legal Counsel',
      lastSeen: new Date(Date.now() - 900000),
      messages: [
        { id: 1, timestamp: new Date(Date.now() - 7200000), direction: 'received', text: 'Please review the case bundle when you get a chance.', status: 'read' },
        { id: 2, timestamp: new Date(Date.now() - 5400000), direction: 'sent', text: 'Will do. Should have feedback by EOD.', status: 'read' }
      ]
    },
    { 
      id: 3, 
      name: 'Anna Byrne', 
      initials: 'AB', 
      status: 'online',
      role: 'Paralegal',
      lastSeen: new Date(),
      messages: [
        { id: 1, timestamp: new Date(Date.now() - 10800000), direction: 'received', text: 'Available for a brief call at 2:00 PM to discuss the deposition.', status: 'read' },
        { id: 2, timestamp: new Date(Date.now() - 9000000), direction: 'sent', text: 'Perfect timing. I\'ll call you then.', status: 'read' },
        { id: 3, timestamp: new Date(Date.now() - 600000), direction: 'received', text: 'Great! Talk to you soon.', status: 'delivered' }
      ]
    },
    { 
      id: 4, 
      name: 'Michael Torres', 
      initials: 'MT', 
      status: 'offline',
      role: 'Associate',
      lastSeen: new Date(Date.now() - 86400000),
      messages: []
    },
    { 
      id: 5, 
      name: 'Sarah Mitchell', 
      initials: 'SM', 
      status: 'online',
      role: 'Litigation Specialist',
      lastSeen: new Date(),
      messages: [
        { id: 1, timestamp: new Date(Date.now() - 14400000), direction: 'received', text: 'The client meeting went well. I\'ll send over my notes.', status: 'read' }
      ]
    }
  ];

  let groups = [
    { 
      id: 1, 
      name: 'Legal Team', 
      initials: 'LT', 
      members: ['Jeff Reynolds', 'Henry Clarke', 'Anna Byrne', 'Michael Torres'],
      messages: [
        { id: 1, timestamp: new Date(Date.now() - 18000000), sender: 'Jeff Reynolds', text: 'Team meeting scheduled for 10 AM tomorrow.', status: 'read' },
        { id: 2, timestamp: new Date(Date.now() - 14400000), sender: 'Anna Byrne', text: 'I\'ll prepare the agenda.', status: 'read' }
      ]
    },
    { 
      id: 2, 
      name: 'Project Alpha', 
      initials: 'PA', 
      members: ['Jeff Reynolds', 'Anna Byrne', 'Sarah Mitchell'],
      messages: [
        { id: 1, timestamp: new Date(Date.now() - 21600000), sender: 'Anna Byrne', text: 'All documents are ready for final review.', status: 'read' },
        { id: 2, timestamp: new Date(Date.now() - 18000000), sender: 'Sarah Mitchell', text: 'Excellent work everyone!', status: 'read' }
      ]
    },
    { 
      id: 3, 
      name: 'Client Services', 
      initials: 'CS', 
      members: ['Henry Clarke', 'Sarah Mitchell', 'Michael Torres'],
      messages: []
    }
  ];

  let notifications = [
    { id: 1, title: 'Welcome!', text: 'Welcome to Prolex Secure Chat', time: new Date() },
    { id: 2, title: 'New Message', text: 'Jeff Reynolds sent you a message', time: new Date(Date.now() - 3600000) }
  ];

  let requests = [
    { id: 1, name: 'Rachel Green', initials: 'RG', text: 'wants to connect' },
    { id: 2, name: 'David Park', initials: 'DP', text: 'sent a contact request' }
  ];

  // ========================================
  // STATE MANAGEMENT
  // ========================================

  let state = {
    currentView: 'contacts', // 'contacts', 'groups', 'settings'
    currentChat: null,
    currentChatType: null, // 'contact' or 'group'
    isDarkMode: false,
    isSidebarOpen: true,
    isTyping: false,
    searchQuery: '',
    messageIdCounter: 1000
  };

  // ========================================
  // DOM ELEMENTS
  // ========================================

  const elements = {
    chatRoot: document.getElementById('chatRoot'),
    chatLeft: document.getElementById('chatLeft'),
    chatRight: document.getElementById('chatRight'),
    contactsList: document.getElementById('contactsList'),
    searchInput: document.getElementById('searchInput'),
    chatArea: document.getElementById('chatArea'),
    chatTitle: document.getElementById('chatTitle'),
    chatSubtitle: document.getElementById('chatSubtitle'),
    chatStatus: document.getElementById('chatStatus'),
    chatAvatar: document.getElementById('chatAvatar'),
    messageInput: document.getElementById('messageInput'),
    sendBtn: document.getElementById('sendBtn'),
    themeBtn: document.getElementById('themeBtn'),
    typingIndicator: document.getElementById('typingIndicator'),
    notificationsList: document.getElementById('notificationsList'),
    requestsList: document.getElementById('requestsList'),
    notifBadge: document.getElementById('notifBadge'),
    requestBadge: document.getElementById('requestBadge'),
    welcomeMessage: document.getElementById('welcomeMessage'),
    // Tabs
    contactsTab: document.getElementById('contactsTab'),
    groupsTab: document.getElementById('groupsTab'),
    optionsTab: document.getElementById('optionsTab'),
    // Buttons
    callBtn: document.getElementById('callBtn'),
    videoBtn: document.getElementById('videoBtn'),
    attachBtn: document.getElementById('attachBtn'),
    searchMsgBtn: document.getElementById('searchMsgBtn'),
    infoBtn: document.getElementById('infoBtn'),
    toggleSidebarBtn: document.getElementById('toggleSidebarBtn')
  };

  // ========================================
  // UTILITY FUNCTIONS
  // ========================================

  function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  function formatTime(date) {
    return date.toLocaleTimeString([], CONFIG.dateFormat);
  }

  function formatRelativeTime(date) {
    const now = new Date();
    const diff = now - date;
    const minutes = Math.floor(diff / 60000);
    const hours = Math.floor(diff / 3600000);
    const days = Math.floor(diff / 86400000);

    if (minutes < 1) return 'Just now';
    if (minutes < 60) return `${minutes}m ago`;
    if (hours < 24) return `${hours}h ago`;
    if (days < 7) return `${days}d ago`;
    return date.toLocaleDateString();
  }

  function generateId() {
    return ++state.messageIdCounter;
  }

  // ========================================
  // RENDERING FUNCTIONS
  // ========================================

  function renderContacts(filter = '') {
    const filtered = contacts.filter(c => 
      c.name.toLowerCase().includes(filter.toLowerCase()) ||
      c.role.toLowerCase().includes(filter.toLowerCase())
    );

    if (filtered.length === 0) {
      elements.contactsList.innerHTML = `
        <div class="chat-empty-state">
          <div class="chat-empty-state-icon">🔍</div>
          <p>No contacts found</p>
        </div>
      `;
      return;
    }

    elements.contactsList.innerHTML = filtered.map(contact => {
      const lastMessage = contact.messages[contact.messages.length - 1];
      const isActive = state.currentChat?.id === contact.id && state.currentChatType === 'contact';
      const unreadCount = contact.messages.filter(m => m.status !== 'read' && m.direction === 'received').length;

      return `
        <div class="chat-contact ${isActive ? 'active' : ''}" data-id="${contact.id}" data-type="contact" tabindex="0" role="button">
          <div class="chat-avatar-wrapper">
            <div class="chat-avatar">${contact.initials}</div>
            <div class="chat-status-indicator ${contact.status}"></div>
          </div>
          <div class="chat-cinfo">
            <div class="chat-cinfo-header">
              <div class="chat-cname">${escapeHtml(contact.name)}</div>
              ${lastMessage ? `<div class="chat-time">${formatRelativeTime(lastMessage.timestamp)}</div>` : ''}
            </div>
            <div class="chat-cmeta">
              ${lastMessage 
                ? `${lastMessage.direction === 'sent' ? '✓ ' : ''}${escapeHtml(lastMessage.text.substring(0, 35))}${lastMessage.text.length > 35 ? '...' : ''}`
                : `<span style="color: var(--text-tertiary);">${contact.role}</span>`
              }
              ${unreadCount > 0 ? `<span class="chat-unread-badge">${unreadCount}</span>` : ''}
            </div>
          </div>
        </div>
      `;
    }).join('');

    // Add event listeners
    document.querySelectorAll('.chat-contact').forEach(el => {
      el.addEventListener('click', () => selectChat(el));
      el.addEventListener('keydown', e => {
        if (e.key === 'Enter' || e.key === ' ') {
          e.preventDefault();
          selectChat(el);
        }
      });
    });
  }

  function renderGroups(filter = '') {
    const filtered = groups.filter(g => 
      g.name.toLowerCase().includes(filter.toLowerCase())
    );

    if (filtered.length === 0) {
      elements.contactsList.innerHTML = `
        <div class="chat-empty-state">
          <div class="chat-empty-state-icon">👥</div>
          <p>No groups found</p>
        </div>
      `;
      return;
    }

    elements.contactsList.innerHTML = filtered.map(group => {
      const lastMessage = group.messages[group.messages.length - 1];
      const isActive = state.currentChat?.id === group.id && state.currentChatType === 'group';

      return `
        <div class="chat-contact ${isActive ? 'active' : ''}" data-id="${group.id}" data-type="group" tabindex="0" role="button">
          <div class="chat-avatar-wrapper">
            <div class="chat-avatar">${group.initials}</div>
          </div>
          <div class="chat-cinfo">
            <div class="chat-cinfo-header">
              <div class="chat-cname">${escapeHtml(group.name)}</div>
              ${lastMessage ? `<div class="chat-time">${formatRelativeTime(lastMessage.timestamp)}</div>` : ''}
            </div>
            <div class="chat-cmeta">
              ${lastMessage 
                ? `${lastMessage.sender.split(' ')[0]}: ${escapeHtml(lastMessage.text.substring(0, 30))}${lastMessage.text.length > 30 ? '...' : ''}`
                : `${group.members.length} members`
              }
            </div>
          </div>
        </div>
      `;
    }).join('');

    // Add event listeners
    document.querySelectorAll('.chat-contact').forEach(el => {
      el.addEventListener('click', () => selectChat(el));
      el.addEventListener('keydown', e => {
        if (e.key === 'Enter' || e.key === ' ') {
          e.preventDefault();
          selectChat(el);
        }
      });
    });
  }

  function renderSettings() {
    elements.contactsList.innerHTML = `
      <div style="padding: var(--space-lg); overflow-y: auto; height: 100%;">
        <h3 style="margin-bottom: var(--space-lg); color: var(--text-primary); font-size: 16px; font-weight: 700;">Chat Settings</h3>

        <div style="margin-bottom: var(--space-xl);">
          <h4 style="margin-bottom: var(--space-md); color: var(--primary); font-size: 14px; font-weight: 600;">Preferences</h4>
          <div style="display: flex; flex-direction: column; gap: var(--space-md);">
            <label style="display: flex; align-items: center; gap: var(--space-sm); font-size: 14px; cursor: pointer;">
              <input type="checkbox" id="soundNotif" checked style="cursor: pointer;"> 
              <span>Sound notifications</span>
            </label>
            <label style="display: flex; align-items: center; gap: var(--space-sm); font-size: 14px; cursor: pointer;">
              <input type="checkbox" id="autoScroll" checked style="cursor: pointer;"> 
              <span>Auto-scroll messages</span>
            </label>
            <label style="display: flex; align-items: center; gap: var(--space-sm); font-size: 14px; cursor: pointer;">
              <input type="checkbox" id="showTyping" checked style="cursor: pointer;"> 
              <span>Typing indicators</span>
            </label>
            <label style="display: flex; align-items: center; gap: var(--space-sm); font-size: 14px; cursor: pointer;">
              <input type="checkbox" id="encryption" checked disabled style="cursor: not-allowed;"> 
              <span>End-to-end encryption (always on)</span>
            </label>
          </div>
        </div>

        <div style="margin-bottom: var(--space-xl);">
          <h4 style="margin-bottom: var(--space-md); color: var(--primary); font-size: 14px; font-weight: 600;">Send Contact Request</h4>
          <form id="contactRequestForm" style="display: flex; flex-direction: column; gap: var(--space-md);">
            <input 
              type="text" 
              id="requestName" 
              placeholder="Enter contact name or email..." 
              style="padding: 10px var(--space-md); border: 1px solid var(--border-light); border-radius: var(--radius-md); font-size: 14px; background: var(--bg-secondary);"
              required
            >
            <button 
              type="submit" 
              style="padding: 10px var(--space-md); background: var(--primary); color: var(--text-inverse); border: none; border-radius: var(--radius-md); cursor: pointer; font-weight: 600; transition: all var(--transition-fast);"
            >
              Send Request
            </button>
          </form>
        </div>

        <div style="margin-bottom: var(--space-xl);">
          <h4 style="margin-bottom: var(--space-md); color: var(--primary); font-size: 14px; font-weight: 600;">Create Group</h4>
          <form id="createGroupForm" style="display: flex; flex-direction: column; gap: var(--space-md);">
            <input 
              type="text" 
              id="groupNameInput" 
              placeholder="Group name..." 
              style="padding: 10px var(--space-md); border: 1px solid var(--border-light); border-radius: var(--radius-md); font-size: 14px; background: var(--bg-secondary);"
              required
            >
            <select 
              id="groupMembersSelect" 
              multiple 
              style="padding: 10px var(--space-md); border: 1px solid var(--border-light); border-radius: var(--radius-md); font-size: 14px; background: var(--bg-secondary); min-height: 120px;"
              required
            >
              ${contacts.map(c => `<option value="${c.id}">${c.name}</option>`).join('')}
            </select>
            <small style="color: var(--text-secondary); font-size: 12px;">Hold Ctrl/Cmd to select multiple members</small>
            <button 
              type="submit" 
              style="padding: 10px var(--space-md); background: var(--primary); color: var(--text-inverse); border: none; border-radius: var(--radius-md); cursor: pointer; font-weight: 600; transition: all var(--transition-fast);"
            >
              Create Group
            </button>
          </form>
        </div>

        <div>
          <h4 style="margin-bottom: var(--space-md); color: var(--primary); font-size: 14px; font-weight: 600;">About</h4>
          <p style="font-size: 13px; color: var(--text-secondary); line-height: 1.6;">
            Prolex Secure Chat v2.0<br>
            End-to-end encrypted legal messaging platform<br>
            © 2026 Prolex Legal Services
          </p>
        </div>
      </div>
    `;

    // Add form handlers
    document.getElementById('contactRequestForm')?.addEventListener('submit', handleContactRequest);
    document.getElementById('createGroupForm')?.addEventListener('submit', handleCreateGroup);
  }

  function renderMessages() {
    if (!state.currentChat) {
      elements.chatArea.innerHTML = `
        <div class="chat-system-message">
          Welcome to Prolex Secure Chat. Select a contact to start messaging.
        </div>
      `;
      return;
    }

    let messagesHTML = '';
    let lastDate = null;

    state.currentChat.messages.forEach((msg, index) => {
      const msgDate = msg.timestamp.toDateString();
      
      // Add date divider
      if (msgDate !== lastDate) {
        messagesHTML += `
          <div class="chat-date-divider">
            <span>${formatDateDivider(msg.timestamp)}</span>
          </div>
        `;
        lastDate = msgDate;
      }

      // Add message
      if (state.currentChatType === 'group' && msg.direction === 'received') {
        messagesHTML += `
          <div class="chat-message received" data-id="${msg.id}">
            <div class="chat-message-sender">${escapeHtml(msg.sender)}</div>
            <div class="chat-message-bubble">${escapeHtml(msg.text)}</div>
            <div class="chat-message-meta">
              <span>${formatTime(msg.timestamp)}</span>
            </div>
          </div>
        `;
      } else {
        messagesHTML += `
          <div class="chat-message ${msg.direction === 'sent' ? 'sent' : 'received'}" data-id="${msg.id}">
            <div class="chat-message-bubble">${escapeHtml(msg.text)}</div>
            <div class="chat-message-meta">
              <span>${formatTime(msg.timestamp)}</span>
              ${msg.direction === 'sent' ? `
                <span class="chat-message-status">
                  <span class="chat-tick ${msg.status || 'sent'}">${getStatusIcon(msg.status || 'sent')}</span>
                </span>
              ` : ''}
            </div>
          </div>
        `;
      }
    });

    elements.chatArea.innerHTML = messagesHTML;
    scrollToBottom();
  }

  function formatDateDivider(date) {
    const today = new Date();
    const yesterday = new Date(today);
    yesterday.setDate(yesterday.getDate() - 1);

    if (date.toDateString() === today.toDateString()) return 'Today';
    if (date.toDateString() === yesterday.toDateString()) return 'Yesterday';
    return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
  }

  function getStatusIcon(status) {
    switch(status) {
      case 'sent': return '✓';
      case 'delivered': return '✓✓';
      case 'read': return '✓✓';
      default: return '✓';
    }
  }

  function renderNotifications() {
    elements.notifBadge.textContent = notifications.length;

    if (notifications.length === 0) {
      elements.notificationsList.innerHTML = `
        <div class="chat-empty-state">
          <div class="chat-empty-state-icon">🔕</div>
          <p>No notifications</p>
        </div>
      `;
      return;
    }

    elements.notificationsList.innerHTML = notifications.map(notif => `
      <div class="chat-notification-item" data-id="${notif.id}">
        <div class="chat-notification-title">${escapeHtml(notif.title)}</div>
        <div class="chat-notification-text">${escapeHtml(notif.text)}</div>
        <div class="chat-notification-time">${formatRelativeTime(notif.time)}</div>
      </div>
    `).join('');

    // Add click handlers
    document.querySelectorAll('.chat-notification-item').forEach(el => {
      el.addEventListener('click', () => {
        const id = parseInt(el.dataset.id);
        notifications = notifications.filter(n => n.id !== id);
        renderNotifications();
      });
    });
  }

  function renderRequests() {
    elements.requestBadge.textContent = requests.length;

    if (requests.length === 0) {
      elements.requestsList.innerHTML = `
        <div class="chat-empty-state">
          <div class="chat-empty-state-icon">✓</div>
          <p>No pending requests</p>
        </div>
      `;
      return;
    }

    elements.requestsList.innerHTML = requests.map(req => `
      <div class="chat-request-item" data-id="${req.id}">
        <div class="chat-request-avatar">${req.initials}</div>
        <div class="chat-request-info">
          <div class="chat-request-name">${escapeHtml(req.name)}</div>
          <div class="chat-request-text">${escapeHtml(req.text)}</div>
        </div>
        <div class="chat-request-actions">
          <button class="chat-request-btn chat-approve-btn" data-action="approve">✓</button>
          <button class="chat-request-btn chat-decline-btn" data-action="decline">✗</button>
        </div>
      </div>
    `).join('');

    // Add click handlers
    document.querySelectorAll('.chat-request-btn').forEach(btn => {
      btn.addEventListener('click', (e) => {
        const item = btn.closest('.chat-request-item');
        const id = parseInt(item.dataset.id);
        const action = btn.dataset.action;
        handleRequest(id, action);
      });
    });
  }

  // ========================================
  // EVENT HANDLERS
  // ========================================

  function selectChat(element) {
    const id = parseInt(element.dataset.id);
    const type = element.dataset.type;

    if (type === 'contact') {
      state.currentChat = contacts.find(c => c.id === id);
      state.currentChatType = 'contact';
    } else {
      state.currentChat = groups.find(g => g.id === id);
      state.currentChatType = 'group';
    }

    // Update UI
    elements.chatTitle.textContent = state.currentChat.name;
    elements.chatAvatar.textContent = state.currentChat.initials;
    
    if (type === 'contact') {
      elements.chatStatus.textContent = getStatusText(state.currentChat.status);
      elements.chatSubtitle.innerHTML = `
        <span id="chatStatus">${getStatusText(state.currentChat.status)}</span>
        <span class="chat-typing-indicator hidden" id="typingIndicator">
          <span class="chat-typing-dot"></span>
          <span class="chat-typing-dot"></span>
          <span class="chat-typing-dot"></span>
        </span>
      `;
    } else {
      elements.chatSubtitle.textContent = `${state.currentChat.members.length} members`;
    }

    // Mark messages as read
    if (type === 'contact') {
      state.currentChat.messages.forEach(msg => {
        if (msg.direction === 'received') msg.status = 'read';
      });
    }

    renderMessages();
    renderContacts(state.searchQuery);
    elements.messageInput.focus();
  }

  function getStatusText(status) {
    switch(status) {
      case 'online': return '🟢 Online';
      case 'away': return '🟡 Away';
      case 'offline': return '⚫ Offline';
      default: return '';
    }
  }

  function sendMessage() {
    const text = elements.messageInput.value.trim();
    if (!text || !state.currentChat) return;

    const message = {
      id: generateId(),
      timestamp: new Date(),
      direction: 'sent',
      text: text,
      status: 'sent'
    };

    // Add to current chat
    state.currentChat.messages.push(message);

    // Clear input
    elements.messageInput.value = '';
    elements.messageInput.style.height = 'auto';
    elements.sendBtn.disabled = true;

    // Re-render
    renderMessages();
    renderContacts(state.searchQuery);

    // Simulate auto-reply for contacts
    if (state.currentChatType === 'contact') {
      simulateReply();
    }
  }

  function simulateReply() {
    // Show typing indicator
    const typingIndicator = document.getElementById('typingIndicator');
    if (typingIndicator) {
      typingIndicator.classList.remove('hidden');
    }

    setTimeout(() => {
      if (typingIndicator) {
        typingIndicator.classList.add('hidden');
      }

      const replies = [
        "Understood. I'll get back to you shortly.",
        "Thanks for the update. Will review and respond.",
        "Received. I'll take care of this right away.",
        "Got it. Let me know if you need anything else.",
        "Perfect. I'll follow up with the details."
      ];

      const reply = {
        id: generateId(),
        timestamp: new Date(),
        direction: 'received',
        text: replies[Math.floor(Math.random() * replies.length)],
        status: 'delivered'
      };

      state.currentChat.messages.push(reply);
      renderMessages();
      renderContacts(state.searchQuery);

      // Add notification
      notifications.unshift({
        id: generateId(),
        title: 'New Message',
        text: `${state.currentChat.name}: ${reply.text.substring(0, 40)}...`,
        time: new Date()
      });
      renderNotifications();

      // Simulate read status after 2 seconds
      setTimeout(() => {
        reply.status = 'read';
        renderMessages();
      }, 2000);
    }, CONFIG.typingIndicatorDelay);
  }

  function handleSearch(query) {
    state.searchQuery = query;
    if (state.currentView === 'contacts') {
      renderContacts(query);
    } else if (state.currentView === 'groups') {
      renderGroups(query);
    }
  }

  function switchTab(tab) {
    state.currentView = tab;

    // Update tab UI
    [elements.contactsTab, elements.groupsTab, elements.optionsTab].forEach(t => {
      t.classList.remove('active');
    });

    if (tab === 'contacts') {
      elements.contactsTab.classList.add('active');
      renderContacts(state.searchQuery);
    } else if (tab === 'groups') {
      elements.groupsTab.classList.add('active');
      renderGroups(state.searchQuery);
    } else if (tab === 'settings') {
      elements.optionsTab.classList.add('active');
      renderSettings();
    }
  }

  function toggleTheme() {
    state.isDarkMode = !state.isDarkMode;
    elements.chatRoot.classList.toggle('chat-dark');
    elements.themeBtn.textContent = state.isDarkMode ? '☀️' : '🌙';
    localStorage.setItem('prolexDarkMode', state.isDarkMode);
  }

  function handleRequest(id, action) {
    const request = requests.find(r => r.id === id);
    if (!request) return;

    if (action === 'approve') {
      // Add to contacts (in real app, this would be an API call)
      notifications.unshift({
        id: generateId(),
        title: 'Request Accepted',
        text: `You accepted ${request.name}'s contact request`,
        time: new Date()
      });
      renderNotifications();
    }

    // Remove from requests
    requests = requests.filter(r => r.id !== id);
    renderRequests();
  }

  function handleContactRequest(e) {
    e.preventDefault();
    const name = document.getElementById('requestName').value.trim();
    
    if (name) {
      notifications.unshift({
        id: generateId(),
        title: 'Request Sent',
        text: `Contact request sent to ${name}`,
        time: new Date()
      });
      renderNotifications();
      document.getElementById('requestName').value = '';
      alert('Contact request sent successfully!');
    }
  }

  function handleCreateGroup(e) {
    e.preventDefault();
    const name = document.getElementById('groupNameInput').value.trim();
    const selectedIds = Array.from(document.getElementById('groupMembersSelect').selectedOptions).map(o => parseInt(o.value));
    
    if (name && selectedIds.length > 0) {
      const members = contacts.filter(c => selectedIds.includes(c.id)).map(c => c.name);
      const initials = name.split(' ').map(w => w[0]).join('').toUpperCase().substring(0, 2);
      
      const newGroup = {
        id: generateId(),
        name: name,
        initials: initials,
        members: members,
        messages: []
      };
      
      groups.push(newGroup);
      
      notifications.unshift({
        id: generateId(),
        title: 'Group Created',
        text: `${name} group created with ${members.length} members`,
        time: new Date()
      });
      renderNotifications();
      
      document.getElementById('groupNameInput').value = '';
      document.getElementById('groupMembersSelect').selectedIndex = -1;
      
      alert('Group created successfully!');
      switchTab('groups');
    }
  }

  function scrollToBottom() {
    elements.chatArea.scrollTop = elements.chatArea.scrollHeight;
  }

  // ========================================
  // PANEL RESIZING
  // ========================================

  let isResizing = false;
  let currentResizer = null;

  function initResizers() {
    const resizer1 = document.getElementById('resizer1');
    const resizer2 = document.getElementById('resizer2');

    [resizer1, resizer2].forEach(resizer => {
      if (!resizer) return;

      resizer.addEventListener('mousedown', (e) => {
        isResizing = true;
        currentResizer = resizer.id;
        resizer.classList.add('resizing');
        document.body.style.cursor = 'col-resize';
        document.addEventListener('mousemove', handleResize);
        document.addEventListener('mouseup', stopResize);
      });
    });
  }

  function handleResize(e) {
    if (!isResizing) return;

    const container = document.getElementById('chatContainer');
    const rect = container.getBoundingClientRect();
    const x = e.clientX - rect.left;
    const totalWidth = rect.width;
    const percentage = (x / totalWidth) * 100;

    if (currentResizer === 'resizer1') {
      const newWidth = Math.max(20, Math.min(40, percentage));
      elements.chatLeft.style.width = `${newWidth}%`;
    } else if (currentResizer === 'resizer2') {
      const newWidth = Math.max(15, Math.min(30, 100 - percentage));
      elements.chatRight.style.width = `${newWidth}%`;
    }
  }

  function stopResize() {
    isResizing = false;
    currentResizer = null;
    document.body.style.cursor = '';
    document.querySelectorAll('.chat-resizer').forEach(r => r.classList.remove('resizing'));
    document.removeEventListener('mousemove', handleResize);
    document.removeEventListener('mouseup', stopResize);
  }

  // ========================================
  // INITIALIZATION
  // ========================================

  function init() {
    // Load saved theme
    const savedTheme = localStorage.getItem('prolexDarkMode');
    if (savedTheme === 'true') {
      state.isDarkMode = true;
      elements.chatRoot.classList.add('chat-dark');
      elements.themeBtn.textContent = '☀️';
    }

    // Render initial views
    renderContacts();
    renderNotifications();
    renderRequests();

    // Auto-select first contact
    if (contacts.length > 0) {
      setTimeout(() => {
        const firstContact = document.querySelector('.chat-contact');
        if (firstContact) firstContact.click();
      }, 100);
    }

    // Initialize resizers
    initResizers();

    // Event listeners - Tabs
    elements.contactsTab.addEventListener('click', () => switchTab('contacts'));
    elements.groupsTab.addEventListener('click', () => switchTab('groups'));
    elements.optionsTab.addEventListener('click', () => switchTab('settings'));

    // Event listeners - Search
    elements.searchInput.addEventListener('input', (e) => handleSearch(e.target.value));

    // Event listeners - Message input
    elements.messageInput.addEventListener('input', () => {
      const hasText = elements.messageInput.value.trim().length > 0;
      elements.sendBtn.disabled = !hasText;

      // Auto-resize textarea
      elements.messageInput.style.height = 'auto';
      elements.messageInput.style.height = elements.messageInput.scrollHeight + 'px';
    });

    elements.messageInput.addEventListener('keydown', (e) => {
      if (e.key === 'Enter' && !e.shiftKey) {
        e.preventDefault();
        sendMessage();
      }
    });

    elements.sendBtn.addEventListener('click', sendMessage);

    // Event listeners - Buttons
    elements.themeBtn.addEventListener('click', toggleTheme);
    elements.callBtn.addEventListener('click', () => alert('Audio call feature coming soon!'));
    elements.videoBtn.addEventListener('click', () => alert('Video call feature coming soon!'));
    elements.attachBtn.addEventListener('click', () => alert('File attachment feature coming soon!'));
    elements.searchMsgBtn.addEventListener('click', () => alert('Message search feature coming soon!'));
    elements.infoBtn.addEventListener('click', () => alert('Conversation info feature coming soon!'));

    // Mobile sidebar toggle (show button on mobile)
    if (window.innerWidth <= 768) {
      elements.toggleSidebarBtn.style.display = 'flex';
      elements.toggleSidebarBtn.addEventListener('click', () => {
        elements.chatLeft.classList.toggle('mobile-open');
      });
    }

    // Responsive handling
    window.addEventListener('resize', () => {
      if (window.innerWidth <= 768) {
        elements.toggleSidebarBtn.style.display = 'flex';
      } else {
        elements.toggleSidebarBtn.style.display = 'none';
        elements.chatLeft.classList.remove('mobile-open');
      }
    });

    // Simulate periodic notifications
    setInterval(() => {
      if (Math.random() < 0.2) {
        const randomContact = contacts[Math.floor(Math.random() * contacts.length)];
        notifications.unshift({
          id: generateId(),
          title: 'Activity',
          text: `${randomContact.name} is ${randomContact.status}`,
          time: new Date()
        });
        renderNotifications();
      }
    }, 30000);

    console.log('✅ Prolex Secure Chat initialized successfully');
  }

  // Start the application
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();
</script>
@endpush