@extends('layouts.master')

@section('title', 'Assignment Reports - Prolex Law Firm')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h4 class="card-title mb-0">Assignment Reports</h4>
                </div>
                <div class="card-body">

<style>
.reports-container {
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
}

.reports-container h1 {
    color: #2c3e50;
    margin-bottom: 30px;
    text-align: center;
    font-size: 2.5rem;
    font-weight: 300;
}

.reports-container h2 {
    color: #34495e;
    margin: 40px 0 20px 0;
    font-size: 1.8rem;
    font-weight: 400;
    border-bottom: 2px solid #ecf0f1;
    padding-bottom: 10px;
}

/* Summary Cards */
.summary-cards {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 20px;
    margin-bottom: 40px;
}

.summary-card {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 25px;
    border-radius: 15px;
    box-shadow: 0 8px 25px rgba(0,0,0,0.1);
    text-align: center;
    transition: transform 0.3s ease, box-shadow 0.3s ease;
}

.summary-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 12px 35px rgba(0,0,0,0.15);
}

.summary-number {
    font-size: 3rem;
    font-weight: bold;
    margin-bottom: 10px;
    display: block;
}

.summary-label {
    font-size: 1.1rem;
    opacity: 0.9;
    font-weight: 300;
}

/* Filters Section */
.filters-section {
    background: #f8f9fa;
    padding: 30px;
    border-radius: 15px;
    margin-bottom: 40px;
    box-shadow: 0 4px 15px rgba(0,0,0,0.05);
}

.filters-title {
    color: #2c3e50;
    margin-bottom: 20px;
    font-size: 1.5rem;
}

.filters-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    margin-bottom: 20px;
}

.filter-group {
    display: flex;
    flex-direction: column;
}

.filter-group label {
    font-weight: 600;
    color: #495057;
    margin-bottom: 8px;
    font-size: 0.9rem;
}

.filter-group select,
.filter-group input[type="date"] {
    padding: 12px 15px;
    border: 2px solid #e9ecef;
    border-radius: 8px;
    font-size: 1rem;
    transition: border-color 0.3s ease, box-shadow 0.3s ease;
}

.filter-group select:focus,
.filter-group input[type="date"]:focus {
    outline: none;
    border-color: #667eea;
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
}

/* Reports Grid */
.reports-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 25px;
    margin-bottom: 40px;
}

.report-card {
    background: white;
    border-radius: 15px;
    padding: 25px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    transition: transform 0.3s ease, box-shadow 0.3s ease;
    border: 1px solid #e9ecef;
}

.report-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 8px 30px rgba(0,0,0,0.12);
}

.report-icon {
    width: 60px;
    height: 60px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    margin-bottom: 15px;
}

.report-icon.productivity { background: linear-gradient(135deg, #667eea, #764ba2); color: white; }
.report-icon.status { background: linear-gradient(135deg, #f093fb, #f5576c); color: white; }
.report-icon.deadline { background: linear-gradient(135deg, #4facfe, #00f2fe); color: white; }
.report-icon.performance { background: linear-gradient(135deg, #43e97b, #38f9d7); color: white; }

.report-title {
    font-size: 1.3rem;
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 10px;
}

.report-description {
    color: #6c757d;
    font-size: 0.95rem;
    line-height: 1.5;
    margin-bottom: 20px;
}

.report-actions {
    text-align: center;
}

.report-actions .btn {
    padding: 10px 25px;
    border-radius: 25px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    transition: all 0.3s ease;
}

.report-actions .btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(0,0,0,0.2);
}

/* Chart Section */
.chart-section {
    margin-bottom: 40px;
}

.chart-title {
    color: #2c3e50;
    margin-bottom: 30px;
    text-align: center;
    font-size: 1.8rem;
}

.charts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
    gap: 30px;
}

.chart-container {
    background: white;
    padding: 25px;
    border-radius: 15px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    border: 1px solid #e9ecef;
}

.chart-container h4 {
    color: #2c3e50;
    margin-bottom: 20px;
    text-align: center;
    font-size: 1.2rem;
    font-weight: 600;
}

/* Detailed Section */
.detailed-section {
    background: white;
    padding: 30px;
    border-radius: 15px;
    box-shadow: 0 4px 20px rgba(0,0,0,0.08);
    border: 1px solid #e9ecef;
}

.section-title {
    color: #2c3e50;
    margin-bottom: 25px;
    font-size: 1.5rem;
}

.assignments-table-container {
    overflow-x: auto;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
}

.assignments-table {
    width: 100%;
    border-collapse: collapse;
    background: white;
}

.assignments-table th,
.assignments-table td {
    padding: 15px;
    text-align: left;
    border-bottom: 1px solid #e9ecef;
}

.assignments-table th {
    background: linear-gradient(135deg, #667eea, #764ba2);
    color: white;
    font-weight: 600;
    font-size: 0.9rem;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.assignments-table tbody tr:hover {
    background: #f8f9fa;
}

/* Priority and Status Badges */
.priority-badge,
.status-badge {
    padding: 6px 12px;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.priority-high { background: #fee2e2; color: #dc2626; }
.priority-medium { background: #fef3c7; color: #d97706; }
.priority-low { background: #dbeafe; color: #2563eb; }

.status-completed { background: #d1fae5; color: #059669; }
.status-in_progress { background: #dbeafe; color: #2563eb; }
.status-pending { background: #fef3c7; color: #d97706; }
.status-cancelled { background: #fee2e2; color: #dc2626; }

/* Progress Bar */
.progress-bar {
    width: 100%;
    height: 8px;
    background: #e5e7eb;
    border-radius: 4px;
    overflow: hidden;
    position: relative;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(90deg, #667eea, #764ba2);
    border-radius: 4px;
    transition: width 0.3s ease;
}

.progress-text {
    position: absolute;
    right: 0;
    top: -20px;
    font-size: 0.8rem;
    font-weight: 600;
    color: #374151;
}

/* Export Section */
.export-section {
    background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
    padding: 40px;
    border-radius: 15px;
    color: white;
}

.export-title {
    color: white;
    margin-bottom: 30px;
    text-align: center;
    font-size: 1.8rem;
}

.export-options {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 25px;
}

.export-option {
    background: rgba(255,255,255,0.1);
    padding: 25px;
    border-radius: 12px;
    text-align: center;
    cursor: pointer;
    transition: all 0.3s ease;
    backdrop-filter: blur(10px);
    border: 1px solid rgba(255,255,255,0.2);
}

.export-option:hover {
    background: rgba(255,255,255,0.2);
    transform: translateY(-3px);
    box-shadow: 0 8px 25px rgba(0,0,0,0.2);
}

.export-option i {
    font-size: 2.5rem;
    margin-bottom: 15px;
    display: block;
}

.export-option h4 {
    margin-bottom: 10px;
    font-size: 1.2rem;
    font-weight: 600;
}

.export-option p {
    font-size: 0.9rem;
    opacity: 0.9;
    margin: 0;
}

/* Button Styles */
.btn {
    display: inline-block;
    padding: 12px 24px;
    border: none;
    border-radius: 25px;
    text-decoration: none;
    font-weight: 600;
    text-align: center;
    cursor: pointer;
    transition: all 0.3s ease;
    font-size: 0.9rem;
}

.btn-primary {
    background: linear-gradient(135deg, #667eea, #764ba2);
    color: white;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #5a6fd8, #6a4190);
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
}

.btn-secondary {
    background: #6c757d;
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
    color: white;
}

/* Responsive Design */
@media (max-width: 768px) {
    .reports-container {
        padding: 15px;
    }

    .reports-container h1 {
        font-size: 2rem;
    }

    .summary-cards {
        grid-template-columns: 1fr;
    }

    .filters-grid {
        grid-template-columns: 1fr;
    }

    .reports-grid {
        grid-template-columns: 1fr;
    }

    .charts-grid {
        grid-template-columns: 1fr;
    }

    .export-options {
        grid-template-columns: 1fr;
    }

    .assignments-table {
        font-size: 0.8rem;
    }

    .assignments-table th,
    .assignments-table td {
        padding: 10px 8px;
    }
}

@media (max-width: 480px) {
    .summary-card {
        padding: 20px;
    }

    .summary-number {
        font-size: 2.5rem;
    }

    .report-card {
        padding: 20px;
    }

    .chart-container {
        padding: 20px;
    }
}

/* Loading States */
.loading {
    opacity: 0.6;
    pointer-events: none;
}

.spinner {
    display: inline-block;
    width: 20px;
    height: 20px;
    border: 3px solid rgba(255,255,255,0.3);
    border-radius: 50%;
    border-top-color: white;
    animation: spin 1s ease-in-out infinite;
}

@keyframes spin {
    to { transform: rotate(360deg); }
}

/* Print Styles */
@media print {
    .filters-section,
    .export-section,
    .report-actions {
        display: none !important;
    }

    .reports-container {
        box-shadow: none;
        padding: 0;
    }

    .summary-cards,
    .charts-grid {
        page-break-inside: avoid;
    }

    .assignments-table {
        page-break-inside: auto;
    }

    .assignments-table tr {
        page-break-inside: avoid;
        page-break-after: auto;
    }
}
</style>

<div class="reports-container">
    <h1><i class="fas fa-chart-bar"></i>Assignment Reports</h1>

    <!-- Summary Cards -->
    <div class="summary-cards">
        <div class="summary-card">
            <div class="summary-number" id="totalAssignments">{{ $stats['total_assignments'] }}</div>
            <div class="summary-label">Total Assignments</div>
        </div>
        <div class="summary-card">
            <div class="summary-number" id="completedAssignments">{{ $stats['completed_assignments'] }}</div>
            <div class="summary-label">Completed</div>
        </div>
        <div class="summary-card">
            <div class="summary-number" id="inProgressAssignments">{{ $stats['in_progress_assignments'] }}</div>
            <div class="summary-label">In Progress</div>
        </div>
        <div class="summary-card">
            <div class="summary-number" id="overdueAssignments">{{ $stats['overdue_assignments'] }}</div>
            <div class="summary-label">Overdue</div>
        </div>
    </div>

    <!-- Filters Section -->
    <div class="filters-section">
        <h2 class="filters-title">
            <i class="fas fa-filter"></i>
            Report Filters
        </h2>

        <form id="reportFilters" method="GET" action="{{ route('modules.processing.assignment.reports') }}">
            <div class="filters-grid">
                <div class="filter-group">
                    <label for="date_from">From Date</label>
                    <input type="date" id="date_from" name="date_from" value="{{ request('date_from') }}">
                </div>

                <div class="filter-group">
                    <label for="date_to">To Date</label>
                    <input type="date" id="date_to" name="date_to" value="{{ request('date_to') }}">
                </div>

                <div class="filter-group">
                    <label for="status">Status</label>
                    <select id="status" name="status">
                        <option value="">All Statuses</option>
                        <option value="pending" {{ request('status') == 'pending' ? 'selected' : '' }}>Pending</option>
                        <option value="in_progress" {{ request('status') == 'in_progress' ? 'selected' : '' }}>In Progress</option>
                        <option value="completed" {{ request('status') == 'completed' ? 'selected' : '' }}>Completed</option>
                        <option value="cancelled" {{ request('status') == 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                    </select>
                </div>

                <div class="filter-group">
                    <label for="priority">Priority</label>
                    <select id="priority" name="priority">
                        <option value="">All Priorities</option>
                        <option value="low" {{ request('priority') == 'low' ? 'selected' : '' }}>Low</option>
                        <option value="medium" {{ request('priority') == 'medium' ? 'selected' : '' }}>Medium</option>
                        <option value="high" {{ request('priority') == 'high' ? 'selected' : '' }}>High</option>
                    </select>
                </div>

                <div class="filter-group">
                    <label for="lawyer_id">Assigned Lawyer</label>
                    <select id="lawyer_id" name="lawyer_id">
                        <option value="">All Lawyers</option>
                        @foreach($lawyers as $lawyer)
                        <option value="{{ $lawyer->id }}" {{ request('lawyer_id') == $lawyer->id ? 'selected' : '' }}>
                            {{ $lawyer->first_name }} {{ $lawyer->last_name }}
                        </option>
                        @endforeach
                    </select>
                </div>

                <div class="filter-group">
                    <label for="report_type">Report Type</label>
                    <select id="report_type" name="report_type">
                        <option value="summary" {{ request('report_type') == 'summary' ? 'selected' : '' }}>Summary</option>
                        <option value="detailed" {{ request('report_type') == 'detailed' ? 'selected' : '' }}>Detailed</option>
                        <option value="productivity" {{ request('report_type') == 'productivity' ? 'selected' : '' }}>Productivity</option>
                    </select>
                </div>
            </div>

            <div style="margin-top: 20px; display: flex; gap: 10px; flex-wrap: wrap;">
                <button type="submit" class="btn btn-primary" id="generateReportBtn">
                    <i class="fas fa-search"></i>
                    Generate Report
                </button>
                <button type="button" class="btn btn-secondary" onclick="clearFilters()">
                    <i class="fas fa-times"></i>
                    Clear Filters
                </button>
            </div>
        </form>
    </div>

    <!-- Reports Grid -->
    <div class="reports-grid">
        <div class="report-card">
            <div class="report-icon productivity">
                <i class="fas fa-chart-line"></i>
            </div>
            <h3 class="report-title">Productivity Report</h3>
            <p class="report-description">Track lawyer productivity and assignment completion rates over time.</p>
            <div class="report-actions">
                <button class="btn btn-primary" onclick="generateProductivityReport()">
                    <i class="fas fa-chart-line"></i>
                    View Report
                </button>
            </div>
        </div>

        <div class="report-card">
            <div class="report-icon status">
                <i class="fas fa-tasks"></i>
            </div>
            <h3 class="report-title">Status Overview</h3>
            <p class="report-description">Current status distribution of all assignments and progress tracking.</p>
            <div class="report-actions">
                <button class="btn btn-primary" onclick="generateStatusReport()">
                    <i class="fas fa-tasks"></i>
                    View Report
                </button>
            </div>
        </div>

        <div class="report-card">
            <div class="report-icon deadline">
                <i class="fas fa-calendar-times"></i>
            </div>
            <h3 class="report-title">Deadline Analysis</h3>
            <p class="report-description">Assignments approaching deadlines and overdue items requiring attention.</p>
            <div class="report-actions">
                <button class="btn btn-primary" onclick="generateDeadlineReport()">
                    <i class="fas fa-calendar-times"></i>
                    View Report
                </button>
            </div>
        </div>

        <div class="report-card">
            <div class="report-icon performance">
                <i class="fas fa-trophy"></i>
            </div>
            <h3 class="report-title">Performance Metrics</h3>
            <p class="report-description">Detailed performance analysis including completion times and efficiency metrics.</p>
            <div class="report-actions">
                <button class="btn btn-primary" onclick="generatePerformanceReport()">
                    <i class="fas fa-trophy"></i>
                    View Report
                </button>
            </div>
        </div>
    </div>

    <!-- Chart Section -->
    <div class="chart-section">
        <h2 class="chart-title">
            <i class="fas fa-chart-pie"></i>
            Assignment Analytics
        </h2>
        <div class="charts-grid">
            <div class="chart-container">
                <h4>Status Distribution</h4>
                <canvas id="statusChart" width="300" height="200"></canvas>
            </div>
            <div class="chart-container">
                <h4>Priority Breakdown</h4>
                <canvas id="priorityChart" width="300" height="200"></canvas>
            </div>
            <div class="chart-container">
                <h4>Productivity by Lawyer</h4>
                <canvas id="productivityChart" width="300" height="200"></canvas>
            </div>
            <div class="chart-container">
                <h4>Deadline Analysis</h4>
                <canvas id="deadlineChart" width="300" height="200"></canvas>
            </div>
        </div>
    </div>

    <!-- Detailed Assignment List (shown when report_type is detailed) -->
    @if(request('report_type') == 'detailed')
    <div class="detailed-section">
        <h2 class="section-title">
            <i class="fas fa-list"></i>
            Detailed Assignment Information
        </h2>
        <div class="assignments-table-container">
            <table class="assignments-table">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Title</th>
                        <th>Lawyer</th>
                        <th>Client</th>
                        <th>Case</th>
                        <th>Priority</th>
                        <th>Status</th>
                        <th>Due Date</th>
                        <th>Progress</th>
                        <th>Creator</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    @foreach($assignments as $assignment)
                    <tr>
                        <td>{{ $assignment->id }}</td>
                        <td>{{ $assignment->assignment_title }}</td>
                        <td>{{ $assignment->lawyer ? $assignment->lawyer->first_name . ' ' . $assignment->lawyer->last_name : 'N/A' }}</td>
                        <td>{{ $assignment->client ? $assignment->client->client_name : 'N/A' }}</td>
                        <td>{{ $assignment->legalCase ? $assignment->legalCase->case_title : 'N/A' }}</td>
                        <td>
                            <span class="priority-badge priority-{{ $assignment->priority }}">
                                {{ ucfirst($assignment->priority) }}
                            </span>
                        </td>
                        <td>
                            <span class="status-badge status-{{ $assignment->status }}">
                                {{ ucfirst(str_replace('_', ' ', $assignment->status)) }}
                            </span>
                        </td>
                        <td>{{ $assignment->due_datetime ? \Carbon\Carbon::parse($assignment->due_datetime)->format('M d, Y H:i') : 'N/A' }}</td>
                        <td>
                            <div class="progress-bar">
                                <div class="progress-fill" style="width: {{ $assignment->progress_percentage }}%"></div>
                                <span class="progress-text">{{ $assignment->progress_percentage }}%</span>
                            </div>
                        </td>
                        <td>{{ $assignment->creator ? $assignment->creator->name : 'N/A' }}</td>
                        <td>
                            <button class="btn btn-sm btn-primary" onclick="viewAssignmentDetails({{ $assignment->id }})">
                                <i class="fas fa-eye"></i>
                            </button>
                        </td>
                    </tr>
                    @endforeach
                </tbody>
            </table>
        </div>
    </div>
    @endif

    <!-- Export Section -->
    <div class="export-section">
        <h2 class="export-title">
            <i class="fas fa-download"></i>
            Export Reports
        </h2>
        <div class="export-options">
            <div class="export-option" onclick="exportReport('pdf')">
                <i class="fas fa-file-pdf"></i>
                <h4>Export as PDF</h4>
                <p>Download report in PDF format</p>
            </div>
            <div class="export-option" onclick="exportReport('excel')">
                <i class="fas fa-file-excel"></i>
                <h4>Export as Excel</h4>
                <p>Download report in Excel format</p>
            </div>
            <div class="export-option" onclick="exportReport('csv')">
                <i class="fas fa-file-csv"></i>
                <h4>Export as CSV</h4>
                <p>Download report in CSV format</p>
            </div>
            <div class="export-option" onclick="exportReport('print')">
                <i class="fas fa-print"></i>
                <h4>Print Report</h4>
                <p>Print the current report view</p>
            </div>
        </div>
    </div>
</div>

<script>
    // Initialize page with sample data
    document.addEventListener('DOMContentLoaded', function() {
        loadSummaryData();
        loadLawyers();
        initializeCharts();
    });

    // Load summary statistics
    function loadSummaryData() {
        // Use data passed from controller
        document.getElementById('totalAssignments').textContent = '{{ $stats['total_assignments'] }}';
        document.getElementById('completedAssignments').textContent = '{{ $stats['completed_assignments'] }}';
        document.getElementById('inProgressAssignments').textContent = '{{ $stats['in_progress_assignments'] }}';
        document.getElementById('overdueAssignments').textContent = '{{ $stats['overdue_assignments'] }}';
    }

    // Load lawyers for filter dropdown
    function loadLawyers() {
        const lawyerSelect = document.getElementById('lawyer_id');
        const lawyers = @json($lawyers);

        lawyers.forEach(lawyer => {
            const option = document.createElement('option');
            option.value = lawyer.id;
            option.textContent = lawyer.first_name + ' ' + lawyer.last_name;
    if (lawyer.id == {{ request('lawyer_id') ?: 'null' }}) {
        option.selected = true;
    }
            lawyerSelect.appendChild(option);
        });
    }

    // Clear all filters
    function clearFilters() {
        document.getElementById('date_from').value = '';
        document.getElementById('date_to').value = '';
        document.getElementById('status').value = '';
        document.getElementById('priority').value = '';
        document.getElementById('lawyer_id').value = '';
        document.getElementById('report_type').value = 'summary';
        document.getElementById('reportFilters').submit();
    }

    // Generate productivity report
    function generateProductivityReport() {
        // Set report type to productivity and submit form
        document.getElementById('report_type').value = 'productivity';
        document.getElementById('reportFilters').submit();
    }

    // Generate status report
    function generateStatusReport() {
        // Set report type to summary and submit form
        document.getElementById('report_type').value = 'summary';
        document.getElementById('reportFilters').submit();
    }

    // Generate deadline report
    function generateDeadlineReport() {
        // Filter for overdue and approaching deadline assignments
        document.getElementById('status').value = 'pending,in_progress';
        document.getElementById('report_type').value = 'detailed';
        document.getElementById('reportFilters').submit();
    }

    // Generate performance report
    function generatePerformanceReport() {
        // Set report type to detailed for comprehensive performance data
        document.getElementById('report_type').value = 'detailed';
        document.getElementById('reportFilters').submit();
    }

    // Initialize charts
    function initializeCharts() {
        const statusData = @json($statusData);
        const priorityData = @json($priorityData);
        const productivityData = @json($productivityData);
        const deadlineData = @json($deadlineData);

        // Status Distribution Chart
        const statusCtx = document.getElementById('statusChart').getContext('2d');
        new Chart(statusCtx, {
            type: 'pie',
            data: {
                labels: statusData.labels,
                datasets: [{
                    data: statusData.data,
                    backgroundColor: [
                        '#fef3c7', // pending
                        '#dbeafe', // in_progress
                        '#d1fae5', // completed
                        '#fee2e2'  // cancelled
                    ],
                    borderWidth: 2,
                    borderColor: '#ffffff'
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'bottom',
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                const percentage = ((context.parsed / total) * 100).toFixed(1);
                                return context.label + ': ' + context.parsed + ' (' + percentage + '%)';
                            }
                        }
                    }
                }
            }
        });

        // Priority Breakdown Chart
        const priorityCtx = document.getElementById('priorityChart').getContext('2d');
        new Chart(priorityCtx, {
            type: 'bar',
            data: {
                labels: priorityData.labels,
                datasets: [{
                    label: 'Assignments',
                    data: priorityData.data,
                    backgroundColor: [
                        '#dbeafe', // low
                        '#fef3c7', // medium
                        '#fee2e2'  // high
                    ],
                    borderWidth: 1,
                    borderColor: '#ffffff'
                }]
            },
            options: {
                responsive: true,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            stepSize: 1
                        }
                    }
                },
                plugins: {
                    legend: {
                        display: false
                    }
                }
            }
        });

        // Productivity Chart
        const productivityCtx = document.getElementById('productivityChart').getContext('2d');
        new Chart(productivityCtx, {
            type: 'horizontalBar',
            data: {
                labels: productivityData.labels,
                datasets: [{
                    label: 'Assignments Completed',
                    data: productivityData.data,
                    backgroundColor: '#10b981',
                    borderWidth: 1,
                    borderColor: '#ffffff'
                }]
            },
            options: {
                indexAxis: 'y',
                responsive: true,
                scales: {
                    x: {
                        beginAtZero: true,
                        ticks: {
                            stepSize: 1
                        }
                    }
                },
                plugins: {
                    legend: {
                        display: false
                    }
                }
            }
        });

        // Deadline Analysis Chart
        const deadlineCtx = document.getElementById('deadlineChart').getContext('2d');
        new Chart(deadlineCtx, {
            type: 'doughnut',
            data: {
                labels: deadlineData.labels,
                datasets: [{
                    data: deadlineData.data,
                    backgroundColor: [
                        '#d1fae5', // on time
                        '#fef3c7', // approaching
                        '#fee2e2'  // overdue
                    ],
                    borderWidth: 2,
                    borderColor: '#ffffff'
                }]
            },
            options: {
                responsive: true,
                plugins: {
                    legend: {
                        position: 'bottom',
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                const percentage = ((context.parsed / total) * 100).toFixed(1);
                                return context.label + ': ' + context.parsed + ' (' + percentage + '%)';
                            }
                        }
                    }
                }
            }
        });
    }

    // View assignment details
    function viewAssignmentDetails(assignmentId) {
        // Redirect to assignment show page
        window.location.href = '{{ route("modules.processing.assignment.show", ":id") }}'.replace(':id', assignmentId);
    }

    // Export report functions
    function exportReport(format) {
        if (format === 'print') {
            // Print the current page
            window.print();
            return;
        }

        const currentUrl = window.location.href;
        const exportUrl = currentUrl.replace('/assignment-reports', '/assignment-reports/export/' + format);

        // Create a temporary link and click it to trigger download
        const link = document.createElement('a');
        link.href = exportUrl;
        link.style.display = 'none';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }

    // Auto-submit form when filters change
    document.querySelectorAll('#reportFilters select, #reportFilters input[type="date"]').forEach(element => {
        element.addEventListener('change', function() {
            // Debounce the submission to avoid too many requests
            clearTimeout(window.filterTimeout);
            window.filterTimeout = setTimeout(() => {
                document.getElementById('reportFilters').submit();
            }, 500);
        });
    });
</script>
@endsection

