@extends('layouts.master')

@section('title', 'Process Cashbook Entry - Prolex Law Firm')

@section('content')
<style>
    @import url('https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap');

    /* Reset and base */
    body, input, select, textarea, button {
        font-family: 'Inter', sans-serif;
    }

    .container {
        max-width: 1400px;
        margin: 3rem auto 4rem;
        background-color: #fff;
        padding: 2.5rem 3rem;
        border-radius: 1rem;
        box-shadow: 0 12px 24px rgba(0,0,0,0.08);
        color: #1f2937;
    }

    h1 {
        text-align: center;
        color: #2563eb;
        margin-bottom: 2rem;
        font-size: 2.75rem;
        font-weight: 700;
        letter-spacing: -0.02em;
    }

    /* Form Sections */
    .form-section {
        background: #f9fafb;
        border-radius: 0.75rem;
        padding: 2rem 2.5rem;
        margin-bottom: 2.5rem;
        box-shadow: 0 4px 12px rgba(37, 99, 235, 0.05);
        border: 1px solid #e0e7ff;
        transition: box-shadow 0.3s ease;
    }

    .form-section:hover {
        box-shadow: 0 6px 20px rgba(37, 99, 235, 0.15);
    }

    .section-title {
        font-size: 1.5rem;
        font-weight: 700;
        color: #1e293b;
        margin-bottom: 1.5rem;
        border-bottom: 2px solid #c7d2fe;
        padding-bottom: 0.5rem;
        display: flex;
        align-items: center;
        gap: 0.5rem;
    }

    /* Form Layout */
    .form-row {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
        gap: 1.5rem 2rem;
    }

    .form-group {
        display: flex;
        flex-direction: column;
    }

    label {
        font-weight: 600;
        margin-bottom: 0.5rem;
        color: #374151;
        font-size: 0.95rem;
        user-select: none;
    }

    .required::after {
        content: ' *';
        color: #ef4444;
    }

    input[type="text"],
    input[type="date"],
    select,
    textarea,
    input[type="number"] {
        background: #fff;
        border: 2px solid #d1d5db;
        border-radius: 0.5rem;
        padding: 0.75rem 1rem;
        font-size: 1rem;
        color: #111827;
        transition: border-color 0.3s ease, box-shadow 0.3s ease;
        font-weight: 500;
        outline-offset: 2px;
    }

    input[type="text"]::placeholder,
    textarea::placeholder {
        color: #9ca3af;
        font-weight: 400;
    }

    input[type="text"]:focus,
    input[type="date"]:focus,
    select:focus,
    textarea:focus,
    input[type="number"]:focus {
        border-color: #2563eb;
        box-shadow: 0 0 8px rgba(37, 99, 235, 0.3);
        outline: none;
    }

    /* Buttons */
    .flex {
        display: flex;
    }

    .justify-end {
        justify-content: flex-end;
    }

    .space-x-4 > * + * {
        margin-left: 1rem;
    }

    button.submit-btn,
    button.reset-btn {
        padding: 0.75rem 2.25rem;
        border-radius: 0.75rem;
        font-size: 1.125rem;
        font-weight: 700;
        cursor: pointer;
        border: none;
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        transition: background-color 0.3s ease, box-shadow 0.3s ease, transform 0.15s ease;
        user-select: none;
        text-decoration: none;
        box-shadow: 0 6px 12px rgba(37, 99, 235, 0.15);
    }

    button.submit-btn {
        background: linear-gradient(135deg, #22c55e, #16a34a);
        color: white;
        box-shadow: 0 8px 20px rgba(34, 197, 94, 0.4);
    }

    button.submit-btn:hover {
        background: linear-gradient(135deg, #16a34a, #15803d);
        box-shadow: 0 12px 28px rgba(22, 163, 74, 0.6);
        transform: translateY(-2px);
    }

    button.reset-btn {
        background-color: #6b7280;
        color: #f9fafb;
    }

    button.reset-btn:hover {
        background-color: #4b5563;
        box-shadow: 0 8px 16px rgba(75, 85, 99, 0.3);
        transform: translateY(-2px);
    }

    button.submit-btn:active,
    button.reset-btn:active {
        transform: translateY(0);
        box-shadow: 0 6px 12px rgba(37, 99, 235, 0.3);
    }

    /* Table Styles */
    .table-container {
        background: #f9fafb;
        border-radius: 0.75rem;
        padding: 2rem 2.5rem;
        box-shadow: 0 4px 12px rgba(37, 99, 235, 0.05);
        border: 1px solid #e0e7ff;
    }

    .table-title {
        font-size: 1.5rem;
        font-weight: 700;
        color: #1e293b;
        margin-bottom: 1.5rem;
        border-bottom: 2px solid #c7d2fe;
        padding-bottom: 0.5rem;
    }

    .table {
        width: 100%;
        border-collapse: collapse;
        background: white;
        border-radius: 0.5rem;
        overflow: hidden;
        box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    }

    .table th {
        background: linear-gradient(135deg, #2563eb, #1e40af);
        color: white;
        padding: 1rem;
        text-align: left;
        font-weight: 600;
        border: none;
    }

    .table td {
        padding: 1rem;
        border-bottom: 1px solid #e5e7eb;
        vertical-align: middle;
    }

    .table tbody tr:hover {
        background-color: #f8fafc;
    }

    .table tbody tr:last-child td {
        border-bottom: none;
    }

    .btn-action {
        padding: 0.5rem 1rem;
        border-radius: 0.5rem;
        font-size: 0.875rem;
        font-weight: 600;
        cursor: pointer;
        border: none;
        transition: all 0.3s ease;
        text-decoration: none;
        display: inline-flex;
        align-items: center;
        gap: 0.25rem;
    }

    .btn-edit {
        background-color: #3b82f6;
        color: white;
    }

    .btn-edit:hover {
        background-color: #2563eb;
        transform: translateY(-1px);
    }

    .btn-delete {
        background-color: #ef4444;
        color: white;
    }

    .btn-delete:hover {
        background-color: #dc2626;
        transform: translateY(-1px);
    }

    /* Responsive tweaks */
    @media (max-width: 1024px) {
        .form-row {
            grid-template-columns: 1fr;
        }
    }

    @media (max-width: 768px) {
        .container {
            padding: 1.5rem 2rem;
        }

        h1 {
            font-size: 2rem;
        }
    }

    /* Hidden class for dynamic elements */
    .hidden {
        display: none;
    }

    /* Success/Error Messages */
    .alert {
        padding: 1rem;
        border-radius: 0.5rem;
        margin-bottom: 1rem;
    }

    .alert-success {
        background-color: #d1fae5;
        color: #065f46;
        border: 1px solid #a7f3d0;
    }

    .alert-error {
        background-color: #fee2e2;
        color: #991b1b;
        border: 1px solid #fecaca;
    }
</style>

<div class="container" role="main" aria-labelledby="pageTitle">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 id="pageTitle">Process Cashbook Entry</h1>
        <a href="{{ route('modules.processing') }}" class="btn btn-outline-primary d-flex align-items-center gap-2" style="border-radius: 0.5rem; padding: 0.5rem 1rem; text-decoration: none;">
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-arrow-left" viewBox="0 0 16 16">
                <path fill-rule="evenodd" d="M15 8a.5.5 0 0 0-.5-.5H2.707l3.147-3.146a.5.5 0 1 0-.708-.708l-4 4a.5.5 0 0 0 0 .708l4 4a.5.5 0 0 0 .708-.708L2.707 8.5H14.5A.5.5 0 0 0 15 8z"/>
            </svg>
            Back to Processing
        </a>
    </div>

    <!-- Success/Error Messages -->
    <div id="messages">
        @if(session('success'))
            <div class="alert alert-success">
                {{ session('success') }}
            </div>
        @endif
        @if(session('error'))
            <div class="alert alert-error">
                {{ session('error') }}
            </div>
        @endif
    </div>

    <div class="row">
        <!-- Form Section -->
        <div class="col-12">
            <form id="cashbookForm" action="{{ route('cashbook.entry.store') }}" method="POST" novalidate>
                @csrf

                <div class="form-section" aria-labelledby="entryDetailsTitle">
                    <h2 id="entryDetailsTitle" class="section-title">📝 Entry Details</h2>

                    <div class="form-row">
                        <div class="form-group">
                            <label for="cashbook_id" class="required">Cashbook</label>
                            <select name="cashbook_id" id="cashbook_id" required aria-required="true">
                                <option value="">Select Cashbook</option>
                                @foreach($mainAccounts ?? [] as $account)
                                    <option value="{{ $account->id }}" {{ old('cashbook_id') == $account->id ? 'selected' : '' }}>
                                        {{ $account->code }} - {{ $account->account_name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>

                        <div class="form-group">
                            <label for="entry_date" class="required">Date</label>
                            <input type="date" id="entry_date" name="entry_date"
                                   value="{{ old('entry_date', '2025-09-20') }}" required aria-required="true">
                        </div>

                        <div class="form-group">
                            <label for="module" class="required">Module</label>
                            <select name="module" id="module" required aria-required="true">
                                <option value="">Select Module</option>
                                <option value="AR" {{ old('module') == 'AR' ? 'selected' : '' }}>AR (Accounts Receivable)</option>
                                <option value="AP" {{ old('module') == 'AP' ? 'selected' : '' }}>AP (Accounts Payable)</option>
                                <option value="GL" {{ old('module') == 'GL' ? 'selected' : '' }}>GL (General Ledger)</option>
                                <option value="CB" {{ old('module') == 'CB' ? 'selected' : '' }}>CB (Cash Book)</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label for="account_id" class="required">Account</label>
                            <select name="account_id" id="account_id" required aria-required="true">
                                <option value="">Select Account</option>
                                @foreach($mainAccounts ?? [] as $account)
                                    <option value="{{ $account->id }}" {{ old('account_id') == $account->id ? 'selected' : '' }}>
                                        {{ $account->code }} - {{ $account->account_name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>

                        <div class="form-group hidden" id="reference_group">
                            <label for="reference">Reference</label>
                            <input type="text" id="reference" name="reference"
                                   value="{{ old('reference') }}" placeholder="Enter reference">
                            <small class="text-gray-500">Reference number or code</small>
                        </div>

                        <div class="form-group">
                            <label for="description" class="required">Description</label>
                            <textarea id="description" name="description" rows="3"
                                      placeholder="Enter description" required aria-required="true">{{ old('description') }}</textarea>
                        </div>

                        <div class="form-group">
                            <label for="tr_code" class="required">TR Code</label>
                            <select name="tr_code" id="tr_code" required aria-required="true">
                                <option value="">Select TR Code</option>
                                <option value="DR" {{ old('tr_code') == 'DR' ? 'selected' : '' }}>DR (Debit)</option>
                                <option value="CR" {{ old('tr_code') == 'CR' ? 'selected' : '' }}>CR (Credit)</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label for="currency_id" class="required">Currency</label>
                            <select name="currency_id" id="currency_id" required aria-required="true">
                                <option value="">Select Currency</option>
                                @foreach($currencies ?? [] as $currency)
                                    <option value="{{ $currency->id }}"
                                            {{ (old('currency_id') == $currency->id || $currency->symbol == 'USD') ? 'selected' : '' }}>
                                        {{ $currency->symbol }} - {{ $currency->name }}
                                    </option>
                                @endforeach
                            </select>
                        </div>

                        <div class="form-group">
                            <label for="amount" class="required">Amount</label>
                            <input type="number" id="amount" name="amount" step="0.01" min="0"
                                   value="{{ old('amount', '00.00') }}" placeholder="00.00" required aria-required="true">
                        </div>
                    </div>
                </div>

                <div class="flex justify-end space-x-4">
                    <button type="button" class="reset-btn" onclick="resetForm()">
                        <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                        </svg>
                        Reset
                    </button>
                    <button type="submit" class="submit-btn">
                        <svg xmlns="http://www.w3.org/2000/svg" class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7" />
                        </svg>
                        Save Entry
                    </button>
                </div>
            </form>
        </div>

        <!-- Table Section -->
        <div class="col-12">
            <div class="table-container">
                <h2 class="table-title">📋 Cashbook Entries</h2>

                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Cashbook</th>
                                <th>Module</th>
                                <th>Account</th>
                                <th>Description</th>
                                <th>Reference</th>
                                <th>TR Code</th>
                                <th>Currency</th>
                                <th>Amount</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody id="cashbookEntries">
                            @forelse($cashbookEntries ?? [] as $entry)
                                <tr data-id="{{ $entry->id }}">
                                    <td>{{ $entry->entry_date ? $entry->entry_date->format('d/m/Y') : '' }}</td>
                                    <td>{{ optional($entry->cashbook)->account_name ?? 'N/A' }}</td>
                                    <td>{{ $entry->module ?? 'N/A' }}</td>
                                    <td>{{ optional($entry->account)->account_name ?? 'N/A' }}</td>
                                    <td>{{ Str::limit($entry->description, 30) }}</td>
                                    <td>{{ $entry->reference ?? '-' }}</td>
                                    <td>
                                        <span class="badge {{ $entry->tr_code == 'DR' ? 'bg-danger' : 'bg-success' }}">
                                            {{ $entry->tr_code }}
                                        </span>
                                    </td>
                                    <td>{{ optional($entry->currency)->symbol ?? 'N/A' }}</td>
                                    <td class="text-end">
                                        {{ number_format($entry->amount, 2) }}
                                    </td>
                                    <td>
                                        <button class="btn-action btn-edit" onclick="editEntry({{ $entry->id }})">
                                            <svg xmlns="http://www.w3.org/2000/svg" class="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z" />
                                            </svg>
                                            Edit
                                        </button>
                                        <button class="btn-action btn-delete" onclick="deleteEntry({{ $entry->id }})">
                                            <svg xmlns="http://www.w3.org/2000/svg" class="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                                            </svg>
                                            Delete
                                        </button>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="10" class="text-center text-muted py-4">
                                        No cashbook entries found. Create your first entry using the form.
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>

                <!-- Summary Section -->
                @if(isset($cashbookEntries) && $cashbookEntries->count() > 0)
                    <div class="mt-4 p-3 bg-white rounded">
                        <div class="row text-center">
                            <div class="col-md-4">
                                <strong>Total Entries:</strong> {{ $cashbookEntries->count() }}
                            </div>
                            <div class="col-md-4">
                                <strong>Total Debit:</strong>
                                <span class="text-danger">
                                    {{ number_format($cashbookEntries->where('tr_code', 'DR')->sum('amount'), 2) }}
                                </span>
                            </div>
                            <div class="col-md-4">
                                <strong>Total Credit:</strong>
                                <span class="text-success">
                                    {{ number_format($cashbookEntries->where('tr_code', 'CR')->sum('amount'), 2) }}
                                </span>
                            </div>
                        </div>
                    </div>
                @endif
            </div>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header bg-danger text-white">
                <h5 class="modal-title">Confirm Deletion</h5>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                Are you sure you want to delete this cashbook entry? This action cannot be undone.
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <form id="deleteForm" method="POST" style="display: inline;">
                    @csrf
                    @method('DELETE')
                    <button type="submit" class="btn btn-danger">Delete</button>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const moduleSelect = document.getElementById('module');
    const referenceGroup = document.getElementById('reference_group');
    const referenceInput = document.getElementById('reference');

    // Show/hide reference field based on module selection
    function toggleReferenceField() {
        const selectedModule = moduleSelect.value;
        if (selectedModule === 'AR' || selectedModule === 'AP') {
            referenceGroup.style.display = 'block';
            referenceInput.required = true;
        } else {
            referenceGroup.style.display = 'none';
            referenceInput.required = false;
            referenceInput.value = ''; // Clear value when hidden
        }
    }

    moduleSelect.addEventListener('change', toggleReferenceField);
    toggleReferenceField(); // Run on page load

    // Form validation
    const form = document.getElementById('cashbookForm');
    form.addEventListener('submit', function(e) {
        const requiredFields = form.querySelectorAll('[required]');
        let isValid = true;

        requiredFields.forEach(field => {
            if (!field.value.trim()) {
                isValid = false;
                field.style.borderColor = '#ef4444';
                field.addEventListener('input', function() {
                    if (this.value.trim()) {
                        this.style.borderColor = '#d1d5db';
                    }
                }, { once: true });
            }
        });

        if (!isValid) {
            e.preventDefault();
        }
    });

    // Auto-hide messages after 5 seconds
    const messages = document.getElementById('messages');
    if (messages.children.length > 0) {
        setTimeout(() => {
            messages.innerHTML = '';
        }, 5000);
    }
});

function resetForm() {
    document.getElementById('cashbookForm').reset();
    document.getElementById('entry_date').value = '2025-09-20';
    document.getElementById('amount').value = '00.00';

    // Reset border colors
    const fields = document.querySelectorAll('input, select, textarea');
    fields.forEach(field => {
        field.style.borderColor = '#d1d5db';
    });
}

function editEntry(id) {
    // This would typically make an AJAX call to get the entry data
    // and populate the form for editing
    alert('Edit functionality would be implemented here for entry ID: ' + id);
}

function deleteEntry(id) {
    const deleteForm = document.getElementById('deleteForm');
    deleteForm.action = `{{ route('cashbook.delete', ':id') }}`.replace(':id', id);
    const modal = new bootstrap.Modal(document.getElementById('deleteModal'));
    modal.show();
}

// Auto-format amount input
document.getElementById('amount').addEventListener('input', function(e) {
    let value = e.target.value.replace(/[^0-9.]/g, '');
    const parts = value.split('.');
    if (parts.length > 2) {
        value = parts[0] + '.' + parts.slice(1).join('');
    }
    if (parts[1] && parts[1].length > 2) {
        value = parts[0] + '.' + parts[1].substring(0, 2);
    }
    e.target.value = value;
});
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
@endsection

