@extends('layouts.master')

@section('styles')
<style>
/* Root color palette for a sleek, modern theme */
:root {
  --primary: #3b82f6; /* Blue-500 */
  --primary-glow: rgba(59, 130, 246, 0.2);
  --success: #22c55e; /* Green-500 */
  --success-glow: rgba(34, 197, 94, 0.2);
  --warning: #facc15; /* Yellow-400 */
  --warning-glow: rgba(250, 202, 21, 0.2);
  --accent: #ef4444; /* Red-500 */
  --accent-glow: rgba(239, 68, 68, 0.2);
  --background: #f3f4f6; /* Gray-100 background */
  --card-bg: #fff; /* Card background */
  --border: #d1d5db; /* Gray-300 */
  --muted: #6b7280; /* Gray-500 muted text */
  --foreground: #111827; /* Gray-900 text */
}

/* Reset some default styles for consistency */
body {
  font-family: 'Inter', 'Helvetica Neue', Helvetica, Arial, sans-serif;
  background: linear-gradient(135deg, #f9fafb, #e0e7ff);
  color: var(--foreground);
  min-height: 100vh;
}

/* Container setup */
.container {
  max-width: 1280px;
  margin: 0 auto;
  padding: 2rem;
}

/* Back button style */
.btn-back {
  display: inline-flex;
  align-items: center;
  padding: 0.5rem 1rem;
  color: var(--primary);
  font-weight: 500;
  border-radius: 6px;
  transition: background 0.3s, transform 0.2s;
}
.btn-back:hover {
  background-color: rgba(59, 130, 246, 0.1);
  transform: translateY(-1px);
}

/* Header styles */
.header {
  background: var(--card-bg);
  border-radius: 1rem;
  padding: 1.5rem;
  box-shadow: 0 4px 12px rgba(0,0,0,0.05);
  animation: fadeIn 0.5s ease-in-out;
}
.header h1 {
  font-size: 2.25rem;
  font-weight: 700;
  background: linear-gradient(to right, #3b82f6, #6366f1);
  -webkit-background-clip: text;
  -webkit-text-fill-color: transparent;
  margin-bottom: 0.5rem;
}
.header p {
  color: var(--muted);
  font-size: 1rem;
}
.header .buttons {
  display: flex;
  gap: 0.75rem;
}
.btn {
  padding: 0.75rem 1.25rem;
  border-radius: 0.75rem;
  font-weight: 600;
  cursor: pointer;
  transition: all 0.3s ease;
  font-size: 0.95rem;
}
.btn-primary {
  background-color: var(--primary);
  color: #fff;
}
.btn-primary:hover {
  background-color: #2563eb;
  box-shadow: 0 4px 10px rgba(59, 130, 246, 0.2);
}
.btn-secondary {
  background-color: #e5e7eb;
  color: var(--foreground);
}
.btn-secondary:hover {
  background-color: #d1d5db;
}
.btn-gradient {
  background: linear-gradient(to right, var(--primary), #6366f1);
  color: #fff;
}
.btn-gradient:hover {
  opacity: 0.9;
}

/* Filters section */
.filters {
  background: var(--card-bg);
  border-radius: 1rem;
  padding: 1.5rem;
  box-shadow: 0 4px 12px rgba(0,0,0,0.05);
  display: grid;
  grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
  gap: 1rem;
  margin-bottom: 2rem;
  animation: fadeIn 0.5s ease-in-out;
}

/* Metrics cards styles */
.metrics {
  display: grid;
  grid-template-columns: repeat(auto-fit, minmax(180px, 240px));
  gap: 1rem;
  margin-bottom: 2rem;
}
.metric-card {
  background: var(--card-bg);
  border-radius: 1rem;
  padding: 0.75rem;
  box-shadow: 0 4px 12px rgba(0,0,0,0.05);
  transition: transform 0.3s, box-shadow 0.3s;
  max-width: 240px;
}
.metric-card:hover {
  transform: translateY(-4px);
  box-shadow: 0 8px 20px rgba(0,0,0,0.1);
}
.metric-header {
  display: flex;
  justify-content: space-between;
  align-items: center;
  margin-bottom: 0.5rem;
}
.metric-title {
  font-size: 0.8rem;
  font-weight: 600;
  color: var(--muted);
  letter-spacing: 0.05em;
}
.metric-value {
  font-size: 1.5rem;
  font-weight: 700;
  color: var(--foreground);
}

/* Table styles */
.table-container {
  overflow-x: auto;
  border-radius: 1rem;
  box-shadow: 0 4px 12px rgba(0,0,0,0.05);
  background: var(--card-bg);
  animation: fadeIn 0.5s ease-in-out;
}
table {
  width: 100%;
  border-collapse: collapse;
}
thead {
  background: #f9fafb;
}
th {
  padding: 1rem;
  font-weight: 600;
  font-size: 0.95rem;
  cursor: pointer;
  position: relative;
  transition: background 0.3s;
}
th:hover {
  background-color: #f3f4f6;
}
th .sort-icon {
  margin-left: 0.5rem;
  font-size: 0.75rem;
}
tbody tr {
  transition: background 0.2s;
}
tbody tr:hover {
  background-color: #f3f4f6;
}
td {
  padding: 1rem;
  font-size: 0.95rem;
}
.status-badge {
  padding: 0.25rem 0.75rem;
  border-radius: 9999px;
  font-size: 0.75rem;
  font-weight: 600;
  display: inline-block;
}
.status-active {
  background-color: rgba(34, 197, 94, 0.2);
  color: #22c55e;
}
.status-completed {
  background-color: rgba(37, 99, 235, 0.2);
  color: #3b82f6;
}
.status-pending {
  background-color: rgba(250, 202, 21, 0.2);
  color: #facc15;
}

/* Pagination styles */
.pagination {
  display: flex;
  gap: 0.5rem;
  justify-content: flex-end;
  margin-top: 1rem;
}
.pagination button {
  padding: 0.5rem 0.75rem;
  border-radius: 0.75rem;
  border: none;
  background-color: #e5e7eb;
  cursor: pointer;
  font-weight: 600;
  transition: all 0.2s;
}
.pagination button:hover {
  background-color: #d1d5db;
}
.pagination .active {
  background-color: var(--primary);
  color: #fff;
}

/* Animations */
@keyframes fadeIn {
  from { opacity: 0; transform: translateY(10px); }
  to { opacity: 1; transform: translateY(0); }
}
</style>
@endsection

@section('content')
<div class="min-h-screen">
  <div class="container mx-auto px-4 py-8">
    <!-- Back Button -->
    <a href="{{ route('modules.reports') }}" class="btn-back mb-4">
      <i class="fas fa-arrow-left mr-2"></i> Back to Reports
    </a>

    <!-- Header -->
    <div class="header mb-6">
      <div class="flex justify-between items-center">
        <div>
          <h1 class="text-4xl font-bold mb-2 bg-clip-text text-transparent bg-gradient-to-r from-primary to-indigo-600">
            Lawyer Performance Statement
          </h1>
          <p class="text-muted">Comprehensive analytics and performance metrics</p>
        </div>
        <div class="flex space-x-3">
          <button class="btn btn-secondary shadow-sm hover:shadow-md">Export Excel</button>
          <button class="btn btn-primary shadow-sm hover:shadow-md">Download PDF</button>
        </div>
      </div>
    </div>

    <!-- Filters -->
    <div class="filters mb-6">
      <form action="{{ route('modules.reports.lawyer-statement.generate') }}" method="POST" id="statementForm">
        @csrf
        <div class="grid gap-4 md:grid-cols-4">
          <div>
            <label class="block mb-2 text-sm font-semibold text-muted">Lawyer Account</label>
            <select class="w-full border border-gray-300 rounded-lg px-4 py-2 focus:outline-none focus:ring-2 focus:ring-primary" name="customer_account" id="customer_account" required>
              <option value="" disabled selected>Select Lawyer</option>
              <option value="all">All Lawyers</option>
              @if($lawyers->isEmpty())
                <option disabled>No lawyers available</option>
              @else
                @foreach($lawyers as $lawyer)
                  <option value="{{ $lawyer->id }}" {{ $selectedLawyer && $lawyer->id == $selectedLawyer->id ? 'selected' : '' }}>
                    {{ $lawyer->full_name }} - {{ $lawyer->specialization }}
                  </option>
                @endforeach
              @endif
            </select>
          </div>
          <div>
            <label class="block mb-2 text-sm font-semibold text-muted">Date From</label>
            <input type="date" id="date_from" name="date_from" class="w-full border border-gray-300 rounded-lg px-4 py-2 focus:outline-none focus:ring-2 focus:ring-primary" value="{{ date('Y-m-d', strtotime('first day of this month')) }}" required>
          </div>
          <div>
            <label class="block mb-2 text-sm font-semibold text-muted">Date To</label>
            <input type="date" id="date_to" name="date_to" class="w-full border border-gray-300 rounded-lg px-4 py-2 focus:outline-none focus:ring-2 focus:ring-primary" value="{{ date('Y-m-d') }}" required>
          </div>
          <div>
            <label class="block mb-2 text-sm font-semibold text-muted">Currency</label>
            <select class="w-full border border-gray-300 rounded-lg px-4 py-2 focus:outline-none focus:ring-2 focus:ring-primary" name="currency" id="currency" required>
              <option value="" disabled selected>Select Currency</option>
              <option value="USD">USD - United States Dollar</option>
              <option value="ZAR">ZAR - RSA Rand</option>
              <option value="EUR">EUR - Euro</option>
              <option value="GBP">GBP - British Pound</option>
              <option value="ZMW">ZMW - Zambian Kwacha</option>
              <option value="BWP">BWP - Botswana Pula</option>
              <option value="ZWL">ZWL - Zimbabwe Dollar</option>
            </select>
          </div>
        </div>
        <div class="flex justify-end mt-4">
          <button type="submit" class="btn btn-primary me-2">Generate Report</button>
          <button type="reset" class="btn btn-secondary" onclick="resetForm()">Refresh</button>
        </div>
      </form>
    </div>

    <!-- Lawyer Profile Card -->
    @if($selectedLawyer)
    <div class="bg-card rounded-xl shadow-lg p-6 mb-6 animate-in fade-in duration-500 delay-200">
      <div class="flex items-center gap-6">
        <div class="w-24 h-24 rounded-full bg-gradient-to-br from-primary to-indigo-600 flex items-center justify-center text-white shadow-lg">
          <svg class="w-12 h-12" fill="currentColor" viewBox="0 0 20 20">
            <path fill-rule="evenodd" d="M10 9a3 3 0 100-6 3 3 0 000 6zm-7 9a7 7 0 1114 0H3z" clip-rule="evenodd"></path>
          </svg>
        </div>
        <div class="flex-1">
          <h2 class="text-3xl font-bold mb-1">{{ $selectedLawyer->full_name }}</h2>
          <p class="text-lg text-muted mb-2">{{ $selectedLawyer->specialization }}</p>
          <div class="flex space-x-4 text-sm">
            <div>
              <span class="font-semibold">Hourly Rate:</span> ${{ $metrics['hourlyRate'] }}/hr
            </div>
            <div>
              <span class="font-semibold">Success Rate:</span> {{ $metrics['successRate'] }}%
            </div>
          </div>
        </div>
      </div>
    </div>
    @else
    <div class="bg-card rounded-xl shadow-lg p-6 mb-6 animate-in fade-in duration-500 delay-200">
      <p class="text-muted">No lawyer selected or no lawyers available.</p>
    </div>
    @endif

    <!-- Summary Cards -->
    <div class="grid md:grid-cols-4 gap-6 mb-8">
        <div class="card p-3">
            <h3 class="text-sm font-semibold text-gray-600 mb-1">Total Fees Generated</h3>
            <p class="text-xl font-bold text-blue-600">${{ number_format($metrics['totalFeesGenerated']) }}</p>
        </div>
        <div class="card p-3">
            <h3 class="text-sm font-semibold text-gray-600 mb-1">Active Cases</h3>
            <p class="text-xl font-bold text-green-600">{{ $metrics['activeCases'] }}</p>
        </div>
        <div class="card p-3">
            <h3 class="text-sm font-semibold text-gray-600 mb-1">Billable Hours</h3>
            <p class="text-xl font-bold text-red-600">{{ number_format($metrics['billableHours']) }}</p>
        </div>
        <div class="card p-3">
            <h3 class="text-sm font-semibold text-gray-600 mb-1">Total Clients</h3>
            <p class="text-xl font-bold text-purple-600">{{ $metrics['clientCount'] }}</p>
        </div>
    </div>

    {{-- Recent Cases Table --}}
    <div class="overflow-hidden rounded-xl shadow-lg bg-white mb-8">
      <div class="px-6 py-4 border-b border-gray-200 flex justify-between items-center">
        <h3 class="text-xl font-bold text-foreground">Recent Cases</h3>
        <span class="text-sm text-muted">{{ $cases->count() }} cases</span>
      </div>
      <div class="overflow-x-auto max-h-96">
        <table class="w-full text-sm divide-y divide-gray-200">
          <thead class="bg-gray-50 sticky top-0 z-10">
            <tr>
              <th class="px-4 py-3 text-left font-semibold cursor-pointer hover:bg-gray-100" data-sort="case_number">Case Number <span class="sort-icon">↕</span></th>
              <th class="px-4 py-3 text-left font-semibold cursor-pointer hover:bg-gray-100" data-sort="client_name">Client Name <span class="sort-icon">↕</span></th>
              <th class="px-4 py-3 text-left font-semibold cursor-pointer hover:bg-gray-100" data-sort="case_type">Case Type <span class="sort-icon">↕</span></th>
              <th class="px-4 py-3 text-left font-semibold cursor-pointer hover:bg-gray-100" data-sort="status">Status <span class="sort-icon">↕</span></th>
              <th class="px-4 py-3 text-left font-semibold cursor-pointer hover:bg-gray-100" data-sort="filed_date">Filed Date <span class="sort-icon">↕</span></th>
              <th class="px-4 py-3 text-right font-semibold cursor-pointer hover:bg-gray-100" data-sort="fees">Fees Generated <span class="sort-icon">↕</span></th>
              <th class="px-4 py-3 text-right font-semibold cursor-pointer hover:bg-gray-100" data-sort="hours">Hours Spent <span class="sort-icon">↕</span></th>
            </tr>
          </thead>
          <tbody class="divide-y divide-gray-200">
            @foreach($cases as $case)
            <tr class="hover:bg-gray-50 transition-colors duration-200">
              <td class="px-4 py-3 font-medium text-foreground">CASE-{{ $case->id }}</td>
              <td class="px-4 py-3 text-muted-foreground">{{ $case->client_name }}</td>
              <td class="px-4 py-3 text-muted-foreground">{{ $case->type }}</td>
              <td class="px-4 py-3">
                <span class="px-3 py-1 rounded-full text-xs font-semibold
                  @if($case->status == 'active') bg-primary/20 text-primary
                  @elseif($case->status == 'completed') bg-success/20 text-success
                  @else bg-warning/20 text-warning
                  @endif">
                  {{ ucfirst($case->status) }}
                </span>
              </td>
              <td class="px-4 py-3">{{ $case->created_at->format('M j, Y') }}</td>
              <td class="px-4 py-3 text-right font-semibold text-success">${{ number_format($case->fees_generated, 2) }}</td>
              <td class="px-4 py-3 text-right text-muted-foreground">{{ $case->hours_spent }} hrs</td>
            </tr>
            @endforeach
          </tbody>
        </table>
      </div>
    </div>
  </div>
</div>

<script>
function resetForm() {
    document.getElementById('customer_account').selectedIndex = 0;
    document.getElementById('date_from').value = '{{ date('Y-m-d', strtotime('first day of this month')) }}';
    document.getElementById('date_to').value = '{{ date('Y-m-d') }}';
    document.getElementById('currency').selectedIndex = 0;
}

// Make lawyer select searchable
document.addEventListener('DOMContentLoaded', function() {
    const lawyerSelect = document.getElementById('customer_account');
    const searchInput = document.createElement('input');
    const dropdownContainer = document.createElement('div');
    const optionsList = document.createElement('div');

    // Style the search input
    searchInput.type = 'text';
    searchInput.placeholder = 'Search lawyers...';
    searchInput.className = 'w-full border border-gray-300 rounded-lg px-4 py-2 focus:outline-none focus:ring-2 focus:ring-primary mb-2';

    // Style the dropdown container
    dropdownContainer.className = 'relative';
    dropdownContainer.style.display = 'none';
    dropdownContainer.style.position = 'absolute';
    dropdownContainer.style.top = '100%';
    dropdownContainer.style.left = '0';
    dropdownContainer.style.right = '0';
    dropdownContainer.style.background = 'white';
    dropdownContainer.style.border = '1px solid #d1d5db';
    dropdownContainer.style.borderRadius = '0.5rem';
    dropdownContainer.style.boxShadow = '0 4px 12px rgba(0,0,0,0.1)';
    dropdownContainer.style.zIndex = '1000';
    dropdownContainer.style.maxHeight = '200px';
    dropdownContainer.style.overflowY = 'auto';

    // Style the options list
    optionsList.className = 'py-1';

    // Get all options
    const options = Array.from(lawyerSelect.options);

    // Function to render options
    function renderOptions(filter = '') {
        optionsList.innerHTML = '';
        const filteredOptions = options.filter(option =>
            option.text.toLowerCase().includes(filter.toLowerCase())
        );

        filteredOptions.forEach(option => {
            const optionDiv = document.createElement('div');
            optionDiv.className = 'px-4 py-2 hover:bg-gray-100 cursor-pointer';
            optionDiv.textContent = option.text;
            optionDiv.addEventListener('click', () => {
                lawyerSelect.value = option.value;
                searchInput.value = option.text;
                dropdownContainer.style.display = 'none';
            });
            optionsList.appendChild(optionDiv);
        });
    }

    // Toggle dropdown on input focus/click
    searchInput.addEventListener('focus', () => {
        renderOptions();
        dropdownContainer.style.display = 'block';
    });

    searchInput.addEventListener('input', (e) => {
        renderOptions(e.target.value);
        dropdownContainer.style.display = 'block';
    });

    // Hide dropdown when clicking outside
    document.addEventListener('click', (e) => {
        if (!dropdownContainer.contains(e.target) && e.target !== searchInput) {
            dropdownContainer.style.display = 'none';
        }
    });

    // Replace the select with the search input and dropdown
    lawyerSelect.parentNode.insertBefore(searchInput, lawyerSelect);
    lawyerSelect.parentNode.insertBefore(dropdownContainer, lawyerSelect);
    dropdownContainer.appendChild(optionsList);
    lawyerSelect.style.display = 'none';

    // Set initial value if selected
    const selectedOption = options.find(option => option.selected);
    if (selectedOption) {
        searchInput.value = selectedOption.text;
    }
});


</script>
@endsection

