@extends('layouts.master')

@section('styles')
<style>
/* Core variables */
:root {
  --primary: #3b82f6;
  --primary-hover: #2563eb;
  --accent: #ef4444;
  --success: #22c55e;
  --warning: #facc15;
  --background: #f3f4f6;
  --card-bg: #fff;
  --border: #d1d5db;
  --muted: #6b7280;
  --foreground: #111827;
}

/* Page reset */
body {
  font-family: 'Inter', sans-serif;
  background: linear-gradient(135deg,#f9fafb,#e0e7ff);
  color: var(--foreground);
  min-height: 100vh;
}

/* Buttons */
.btn {
  padding: 0.5rem 1rem;
  border-radius: 0.75rem;
  font-weight: 600;
  cursor: pointer;
  transition: all 0.3s ease;
  display: inline-flex;
  align-items: center;
  gap: 0.25rem;
}
.btn-primary {
  background: linear-gradient(to right, var(--primary), var(--primary-hover));
  color: #fff;
}
.btn-primary:hover {
  opacity: 0.9;
}
.btn-secondary {
  background-color: #e5e7eb;
  color: var(--foreground);
}
.btn-secondary:hover {
  background-color: #d1d5db;
}

/* Cards */
.card {
  background: var(--card-bg);
  border-radius: 1rem;
  padding: 1.5rem;
  box-shadow: 0 4px 12px rgba(0,0,0,0.05);
  transition: transform 0.3s, box-shadow 0.3s;
}
.card:hover {
  transform: translateY(-3px);
  box-shadow: 0 8px 20px rgba(0,0,0,0.1);
}

/* Table */
.table-container {
  overflow-x: auto;
  border-radius: 1rem;
  box-shadow: 0 4px 12px rgba(0,0,0,0.05);
}
table {
  width: 100%;
  border-collapse: collapse;
}
thead {
  background: #f9fafb;
}
th, td {
  padding: 0.75rem 1rem;
  text-align: left;
}
tbody tr:hover {
  background-color: #f3f4f6;
}

/* Status badges */
.status-badge {
  padding: 0.25rem 0.75rem;
  border-radius: 9999px;
  font-size: 0.75rem;
  font-weight: 600;
  display: inline-block;
}
.status-paid {
  background-color: rgba(34,197,94,0.2);
  color: #22c55e;
}
.status-partially_paid {
  background-color: rgba(250,202,21,0.2);
  color: #facc15;
}
.status-unpaid {
  background-color: rgba(239,68,68,0.2);
  color: #ef4444;
}

/* Animations */
@keyframes fadeIn {
  from { opacity: 0; transform: translateY(10px); }
  to { opacity: 1; transform: translateY(0); }
}
.animate-fade-in {
  animation: fadeIn 0.5s ease-in-out;
}
</style>
@endsection

@section('content')
<div class="min-h-screen p-4 md:p-8" x-data="totalFeesDashboard()" x-init="fetchData()">
  <div class="max-w-7xl mx-auto space-y-8 animate-fade-in">

    <!-- Back Navigation -->
    <div class="flex justify-end mb-3">
      <a href="{{ route('modules.reports') }}" class="text-primary hover:underline flex items-center">
        <i class="fas fa-arrow-left mr-2"></i> Back to Reports
      </a>
    </div>

    <!-- Header -->
    <div class="flex flex-col md:flex-row md:justify-between md:items-center gap-4">
      <div>
        <h1 class="text-4xl font-bold bg-gradient-to-r from-primary to-accent bg-clip-text text-transparent">
          Total Fees Report
        </h1>
        <p class="text-muted mt-1">Comprehensive overview of all fee collections</p>
      </div>
      <div class="flex flex-wrap gap-3">
        <button class="btn btn-secondary" @click="exportExcel">
          <i class="fas fa-file-excel"></i> Export Excel
        </button>
        <button class="btn btn-primary" @click="downloadPDF">
          <i class="fas fa-file-pdf"></i> Download PDF
        </button>
      </div>
    </div>

    <!-- Filters -->
    <div class="card">
      <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
        <div>
          <label class="text-sm font-semibold text-foreground">Client</label>
          <select x-model="filters.client" @change="fetchData()" class="w-full border rounded-lg px-4 py-2">
            <option value="all">All Clients</option>
            <template x-for="client in clients" :key="client.id">
              <option :value="client.id" x-text="client.full_name"></option>
            </template>
          </select>
        </div>
        <div>
          <label class="text-sm font-semibold text-foreground">Date Range</label>
          <input type="date" x-model="filters.startDate" @change="fetchData()" class="w-full border rounded-lg px-4 py-2" />
          <input type="date" x-model="filters.endDate" @change="fetchData()" class="w-full border rounded-lg px-4 py-2 mt-1" />
        </div>
        <div>
          <label class="text-sm font-semibold text-foreground">Status</label>
          <select x-model="filters.status" @change="fetchData()" class="w-full border rounded-lg px-4 py-2">
            <option value="all">All Status</option>
            <option value="paid">Paid</option>
            <option value="partially_paid">Partially Paid</option>
            <option value="unpaid">Unpaid</option>
          </select>
        </div>
        <div class="flex items-end">
          <button class="btn btn-primary w-full" @click="fetchData()">Apply Filters</button>
        </div>
      </div>
    </div>

    <!-- Metrics -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
      <template x-for="[label, value, color] in Object.entries(metrics)" :key="label">
        <div class="card">
          <h3 class="text-sm font-semibold text-muted uppercase tracking-wide mb-2" x-text="label.replace(/([A-Z])/g,' $1')"></h3>
          <p class="text-3xl font-bold" :class="color" x-text="value"></p>
        </div>
      </template>
    </div>

    <!-- Invoice Table -->
    <div class="table-container mt-6">
      <table class="w-full">
        <thead>
          <tr>
            <th>Client Name</th>
            <th>Invoice #</th>
            <th>Invoice Date</th>
            <th>Due Date</th>
            <th class="text-right">Total Amount</th>
            <th class="text-right">Paid Amount</th>
            <th class="text-right">Balance Due</th>
            <th>Status</th>
            <th>Currency</th>
          </tr>
        </thead>
        <tbody>
          <template x-for="invoice in invoices" :key="invoice.id">
            <tr class="hover:bg-gray-50">
              <td x-text="invoice.client_name"></td>
              <td x-text="invoice.invoice_number"></td>
              <td x-text="invoice.invoice_date"></td>
              <td x-text="invoice.due_date"></td>
              <td class="text-right" x-text="'$'+invoice.total.toLocaleString()"></td>
              <td class="text-right" x-text="'$'+invoice.paid.toLocaleString()"></td>
              <td class="text-right font-bold" :class="invoice.total - invoice.paid > 0 ? 'text-danger' : 'text-success'" x-text="'$'+(invoice.total - invoice.paid).toLocaleString()"></td>
              <td>
                <span class="status-badge" :class="'status-'+invoice.status" x-text="invoice.status.replace('_',' ')"></span>
              </td>
              <td x-text="invoice.currency"></td>
            </tr>
          </template>
          <tr x-show="invoices.length===0">
            <td colspan="9" class="text-center py-6 text-muted">No invoices found.</td>
          </tr>
        </tbody>
      </table>
    </div>

  </div>
</div>
@endsection

@section('scripts')
<script>
function totalFeesDashboard() {
  return {
    clients: @json($clients ?? []),
    filters: {
      client: 'all',
      status: 'all',
      startDate: '{{ date('Y-m-d', strtotime('first day of this month')) }}',
      endDate: '{{ date('Y-m-d') }}'
    },
    metrics: {},
    invoices: [],

    fetchData() {
      // Fetch metrics and invoices dynamically from your backend API
      fetch(`/api/total-fees?client=${this.filters.client}&status=${this.filters.status}&start=${this.filters.startDate}&end=${this.filters.endDate}`)
        .then(res => res.json())
        .then(data => {
          this.metrics = {
            'Total Fees': `$${data.metrics.totalFees.toLocaleString()}`,
            'Paid Fees': `$${data.metrics.paidFees.toLocaleString()}`,
            'Outstanding Fees': `$${data.metrics.outstandingFees.toLocaleString()}`,
            'Total Invoices': data.metrics.totalInvoices
          };
          this.invoices = data.invoices.map(inv => ({
            id: inv.id,
            client_name: inv.client_name,
            invoice_number: inv.invoice_number,
            invoice_date: inv.invoice_date,
            due_date: inv.due_date,
            total: inv.total,
            paid: inv.paid,
            status: inv.status,
            currency: inv.currency
          }));
        });
    },

    exportExcel() {
      window.location.href = `/export/total-fees/excel?client=${this.filters.client}&status=${this.filters.status}&start=${this.filters.startDate}&end=${this.filters.endDate}`;
    },

    downloadPDF() {
      window.location.href = `/export/total-fees/pdf?client=${this.filters.client}&status=${this.filters.status}&start=${this.filters.startDate}&end=${this.filters.endDate}`;
    }
  }
}
</script>
@endsection

